# File: main_dblp.py
# Description: Main script for evaluating our proposed algorithms using binary classifiers as predictors.


import os
import numpy as np
from insertion_pivot_alg import *
from our_insertion_alg import *
from our_dynamic_alg import *
import time as tm
import psutil


def run_dynamic_alg(graph_file, streaming_file, prediction_file, output_dir, repeated_times):
    with open(streaming_file, "r") as f:
        lines = f.readline()
        num = int(lines.strip())
    degrees = [1 for i in range(num)]

    original_graph = nx.read_edgelist(graph_file, create_using=nx.Graph, nodetype=int)
    degrees += [original_graph.degree[i] for i in range(original_graph.number_of_nodes())]
        
    print("CKLPU24 Algorithm is running...")
    approx_ratio = 3
    predictor = None

    cost_3 = []
    for time in range(repeated_times):
        output_file = output_dir + f"/dynamic_approx_{approx_ratio}.txt"

        start_time = tm.time()
        random.seed(time)
        np.random.seed(time)

        start_memory = psutil.Process().memory_info().rss
        alg = OurDynamicAlgorithm(num, degrees, predictor)
        alg.run_approx_algorithm(original_graph, streaming_file, approx_ratio, output_file)
        end_memory = psutil.Process().memory_info().rss
        print(f"Total memory used: {(end_memory - start_memory) / (1024 * 1024):.2f} MB")  if time == 0 else None

        cost_3.append(alg_pay_cost(num, streaming_file, output_file))
        end_time = tm.time()
        print(f"Time taken for {time+1} run: {end_time - start_time} seconds")
        print(f"Cost: {cost_3}")

    print(f"cost each time: {cost_3}")
    print(f"cost: ({np.mean(cost_3)}, {np.std(cost_3)})")


    print("Algorithm 1 is running...")
    approx_ratio = 2.06
    predictor = load_prediction(prediction_file)

    cost_2 = []
    for time in range(repeated_times):
        output_file = output_dir + f"/approx_{approx_ratio}.txt"

        start_time = tm.time()
        random.seed(time)
        np.random.seed(time)

        start_memory = psutil.Process().memory_info().rss
        alg = OurDynamicAlgorithm(num, degrees, predictor)
        alg.run_approx_algorithm(original_graph, streaming_file, approx_ratio, output_file)
        end_memory = psutil.Process().memory_info().rss
        print(f"Total memory used: {(end_memory - start_memory) / (1024 * 1024):.2f} MB")  if time == 0 else None

        cost_2.append(alg_pay_cost(num, streaming_file, output_file))
        end_time = tm.time()
        print(f"Time taken for {time+1} run: {end_time - start_time} seconds")
        
    cost_min = [min(a, b) for a, b in zip(cost_3, cost_2)]
    
    print(f"Cost each time: {cost_min}")
    print(f"Cost: ({np.mean(cost_min)}, {np.std(cost_min)})")

def run_insertion_pivot_alg(num_of_nodes, streaming_file, k_list, repeated_times):
    cost_dict = {}
    for k in k_list:
        print(f"k: {k}")
        cost = []
        for time in range(repeated_times):
            output_file = os.path.join("./results", f"pivotAlg_output_k_{k}.txt")

            random.seed(time)
            np.random.seed(time)
            
            start_memory = psutil.Process().memory_info().rss
            alg = PivotAlgorithm(num_of_nodes, k)
            alg.run(streaming_file, output_file)
            end_memory = psutil.Process().memory_info().rss
            print(f"Total memory used: {(end_memory - start_memory) / (1024 * 1024):.2f} MB")  if time == 0 else None
            
            cost.append(alg_pay_cost(num_of_nodes, streaming_file, output_file))

        print(f"Cost each time: {cost}")
        cost_dict[k] = (np.mean(cost), np.std(cost))

        print(f"Cost: ({cost_dict[k][0]}, {cost_dict[k][1]})")

    return cost_dict

def run_our_insertion_alg(num_of_nodes, streaming_file, prediction_file, k_list, output_dir, repeated_times):
    beta_list = {}
    cost_dict = {}
    for k in k_list:
        print(f"k: {k}")
        beta_list[k] = {}
        cost_dict[k] = {}

        predictor = load_prediction(prediction_file)

        cost = []
        for time in range(repeated_times):
            output_file1 = os.path.join(output_dir, f"ourAlg_output1_k_{k}.txt")
            output_file2 = os.path.join(output_dir, f"ourAlg_output2_k_{k}.txt")

            random.seed(time)
            np.random.seed(time)
            
            start_memory = psutil.Process().memory_info().rss
            alg = OurInsertionAlgorithm(num_of_nodes, k, predictor)
            alg.run(streaming_file, output_file1, output_file2)
            end_memory = psutil.Process().memory_info().rss
            print(f"Total memory used: {(end_memory - start_memory) / (1024 * 1024):.2f} MB")  if time == 0 else None

            cost1 = alg_pay_cost(num_of_nodes, streaming_file, output_file1)
            cost2 = alg_pay_cost(num_of_nodes, streaming_file, output_file2)

            cost.append(min(cost1, cost2))
        print(f"Cost each time: {cost}")
        cost_dict[k] = (np.mean(cost), np.std(cost))

        print(f"Cost: {cost_dict[k]}")

    return beta_list, cost_dict

if __name__ == "__main__":
    print("CM23 Algorithm is running...")
    start_time = tm.time()
    run_insertion_pivot_alg(1200, "../data/sbm/nodes_1200/streaming-SBM-1200.txt", [20, 50], 5)
    end_time = tm.time()
    print(f"Time taken: {(end_time - start_time)} seconds")

    print("Algorithm 8 is running...")
    start_time = tm.time()
    run_our_insertion_alg(20000, "../data/dblp/streaming-dblp10000.txt", "../data/dblp/dblp10000_prediction_binary.txt", [50, 100, 150, 200], "./results", 5)
    end_time = tm.time()
    print(f"Time taken: {(end_time - start_time)} seconds")
    run_our_insertion_alg(1200, "../data/sbm/nodes_1200/streaming-SBM-1200.txt", "../data/sbm/nodes_1200/SBM_1200_prediction_binary.txt", [20, 50], "./results", 5)

    run_dynamic_alg("../data/dblp/dblp10000-edges.csv", "../data/dblp/streaming-dblp10000.txt", "../data/dblp/dblp10000_prediction_binary.txt", "./results", 5)
    run_dynamic_alg("../data/sbm/nodes_1200/SBM-1200-edges.csv", "../data/sbm/nodes_1200/streaming-SBM-1200.txt", "../data/sbm/nodes_1200/SBM_1200_prediction_binary.txt", "./results", 5)
