# Copyright (c) Facebook, Inc. and its affiliates.
# 
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

''' Provides Python helper function to read My scannet dataset.

Author: Charles R. Qi
Date: October, 2017

Updated by Charles R. Qi
Date: December, 2018
Note: removed basis loading.
'''
import numpy as np
import cv2
import os
import scipy.io as sio # to load .mat files for depth points

type2class={'human': 0, 'sneakers': 1, 'chair': 2, 'hat': 3, 'lamp': 4, 'bottle': 5, 'cabinet/shelf': 6, 'cup': 7, 'car': 8, 'glasses': 9, 'picture/frame': 10, 'desk': 11, 'handbag': 12, 'street_lights': 13, 'book': 14, 'plate': 15, 'helmet': 16, 'leather_shoes': 17, 'pillow': 18, 'glove': 19, 'potted_plant': 20, 'bracelet': 21, 'flower': 22, 'monitor': 23, 'storage_box': 24, 'plants_pot/vase': 25, 'bench': 26, 'wine_glass': 27, 'boots': 28, 'dining_table': 29, 'umbrella': 30, 'boat': 31, 'flag': 32, 'speaker': 33, 'trash_bin/can': 34, 'stool': 35, 'backpack': 36, 'sofa': 37, 'belt': 38, 'carpet': 39, 'basket': 40, 'towel/napkin': 41, 'slippers': 42, 'bowl': 43, 'barrel/bucket': 44, 'coffee_table': 45, 'suv': 46, 'toy': 47, 'tie': 48, 'bed': 49, 'traffic_light': 50, 'pen/pencil': 51, 'microphone': 52, 'sandals': 53, 'canned': 54, 'necklace': 55, 'mirror': 56, 'faucet': 57, 'bicycle': 58, 'bread': 59, 'high_heels': 60, 'ring': 61, 'van': 62, 'watch': 63, 'combine_with_bowl': 64, 'sink': 65, 'horse': 66, 'fish': 67, 'apple': 68, 'traffic_sign': 69, 'camera': 70, 'candle': 71, 'stuffed_animal': 72, 'cake': 73, 'motorbike/motorcycle': 74, 'wild_bird': 75, 'laptop': 76, 'knife': 77, 'cellphone': 78, 'paddle': 79, 'truck': 80, 'cow': 81, 'power_outlet': 82, 'clock': 83, 'drum': 84, 'fork': 85, 'bus': 86, 'hanger': 87, 'nightstand': 88, 'pot/pan': 89, 'sheep': 90, 'guitar': 91, 'traffic_cone': 92, 'tea_pot': 93, 'keyboard': 94, 'tripod': 95, 'hockey_stick': 96, 'fan': 97, 'dog': 98, 'spoon': 99, 'blackboard/whiteboard': 100, 'balloon': 101, 'air_conditioner': 102, 'cymbal': 103, 'mouse': 104, 'telephone': 105, 'pickup_truck': 106, 'orange': 107, 'banana': 108, 'airplane': 109, 'luggage': 110, 'skis': 111, 'soccer': 112, 'trolley': 113, 'oven': 114, 'remote': 115, 'combine_with_glove': 116, 'paper_towel': 117, 'refrigerator': 118, 'train': 119, 'tomato': 120, 'machinery_vehicle': 121, 'tent': 122, 'shampoo/shower_gel': 123, 'head_phone': 124, 'lantern': 125, 'donut': 126, 'cleaning_products': 127, 'sailboat': 128, 'tangerine': 129, 'pizza': 130, 'kite': 131, 'computer_box': 132, 'elephant': 133, 'toiletries': 134, 'gas_stove': 135, 'broccoli': 136, 'toilet': 137, 'stroller': 138, 'shovel': 139, 'baseball_bat': 140, 'microwave': 141, 'skateboard': 142, 'surfboard': 143, 'surveillance_camera': 144, 'gun': 145, 'Life_saver': 146, 'cat': 147, 'lemon': 148, 'liquid_soap': 149, 'zebra': 150, 'duck': 151, 'sports_car': 152, 'giraffe': 153, 'pumpkin': 154, 'Accordion/keyboard/piano': 155, 'radiator': 156, 'converter': 157, 'tissue': 158, 'carrot': 159, 'washing_machine': 160, 'vent': 161, 'cookies': 162, 'cutting/chopping_board': 163, 'tennis_racket': 164, 'candy': 165, 'skating_and_skiing_shoes': 166, 'scissors': 167, 'folder': 168, 'baseball': 169, 'strawberry': 170, 'bow_tie': 171, 'pigeon': 172, 'pepper': 173, 'coffee_machine': 174, 'bathtub': 175, 'snowboard': 176, 'suitcase': 177, 'grapes': 178, 'ladder': 179, 'pear': 180, 'american_football': 181, 'basketball': 182, 'potato': 183, 'paint_brush': 184, 'printer': 185, 'billiards': 186, 'fire_hydrant': 187, 'goose': 188, 'projector': 189, 'sausage': 190, 'fire_extinguisher': 191, 'extension_cord': 192, 'facial_mask': 193, 'tennis_ball': 194, 'chopsticks': 195, 'Electronic_stove_and_gas_stove': 196, 'pie': 197, 'frisbee': 198, 'kettle': 199, 'hamburger': 200, 'golf_club': 201, 'cucumber': 202, 'clutch': 203, 'blender': 204, 'tong': 205, 'slide': 206, 'hot_dog': 207, 'toothbrush': 208, 'facial_cleanser': 209, 'mango': 210, 'deer': 211, 'egg': 212, 'violin': 213, 'marker': 214, 'ship': 215, 'chicken': 216, 'onion': 217, 'ice_cream': 218, 'tape': 219, 'wheelchair': 220, 'plum': 221, 'bar_soap': 222, 'scale': 223, 'watermelon': 224, 'cabbage': 225, 'router/modem': 226, 'golf_ball': 227, 'pine_apple': 228, 'crane': 229, 'fire_truck': 230, 'peach': 231, 'cello': 232, 'notepaper': 233, 'tricycle': 234, 'toaster': 235, 'helicopter': 236, 'green_beans': 237, 'brush': 238, 'carriage': 239, 'cigar': 240, 'earphone': 241, 'penguin': 242, 'hurdle': 243, 'swing': 244, 'radio': 245, 'CD': 246, 'parking_meter': 247, 'swan': 248, 'garlic': 249, 'french_fries': 250, 'horn': 251, 'avocado': 252, 'saxophone': 253, 'trumpet': 254, 'sandwich': 255, 'cue': 256, 'kiwi_fruit': 257, 'bear': 258, 'fishing_rod': 259, 'cherry': 260, 'tablet': 261, 'green_vegetables': 262, 'nuts': 263, 'corn': 264, 'key': 265, 'screwdriver': 266, 'globe': 267, 'broom': 268, 'pliers': 269, 'hammer': 270, 'volleyball': 271, 'eggplant': 272, 'trophy': 273, 'board_eraser': 274, 'dates': 275, 'rice': 276, 'tape_measure/ruler': 277, 'dumbbell': 278, 'hamimelon': 279, 'stapler': 280, 'camel': 281, 'lettuce': 282, 'goldfish': 283, 'meat_balls': 284, 'medal': 285, 'toothpaste': 286, 'antelope': 287, 'shrimp': 288, 'rickshaw': 289, 'trombone': 290, 'pomegranate': 291, 'coconut': 292, 'jellyfish': 293, 'mushroom': 294, 'calculator': 295, 'treadmill': 296, 'butterfly': 297, 'egg_tart': 298, 'cheese': 299, 'pomelo': 300, 'pig': 301, 'race_car': 302, 'rice_cooker': 303, 'tuba': 304, 'crosswalk_sign': 305, 'papaya': 306, 'hair_dryer': 307, 'green_onion': 308, 'chips': 309, 'dolphin': 310, 'sushi': 311, 'urinal': 312, 'donkey': 313, 'electric_drill': 314, 'spring_rolls': 315, 'tortoise/turtle': 316, 'parrot': 317, 'flute': 318, 'measuring_cup': 319, 'shark': 320, 'steak': 321, 'poker_card': 322, 'binoculars': 323, 'llama': 324, 'radish': 325, 'noodles': 326, 'mop': 327, 'yak': 328, 'crab': 329, 'microscope': 330, 'barbell': 331, 'Bread/bun': 332, 'baozi': 333, 'lion': 334, 'red_cabbage': 335, 'polar_bear': 336, 'lighter': 337, 'mangosteen': 338, 'seal': 339, 'comb': 340, 'eraser': 341, 'pitaya': 342, 'scallop': 343, 'pencil_case': 344, 'saw': 345, 'table_tennis_paddle': 346, 'okra': 347, 'starfish': 348, 'monkey': 349, 'eagle': 350, 'durian': 351, 'rabbit': 352, 'game_board': 353, 'french_horn': 354, 'ambulance': 355, 'asparagus': 356, 'hoverboard': 357, 'pasta': 358, 'target': 359, 'hotair_balloon': 360, 'chainsaw': 361, 'lobster': 362, 'iron': 363, 'flashlight': 364}

class2type = {type2class[t]:t for t in type2class}


def flip_axis_to_camera(pc):
    ''' Flip X-right,Y-forward,Z-up to X-right,Y-down,Z-forward
        Input and output are both (N,3) array
    '''
    pc2 = np.copy(pc)
    pc2[:,[0,1,2]] = pc2[:,[0,2,1]] # cam X,Y,Z = depth X,-Z,Y
    pc2[:,1] *= -1
    return pc2

def flip_axis_to_depth(pc):
    pc2 = np.copy(pc)
    pc2[:,[0,1,2]] = pc2[:,[0,2,1]] # depth X,Y,Z = cam X,Z,-Y
    pc2[:,2] *= -1
    return pc2


class SUNObject3d(object):
    def __init__(self, line):
        data = line.split(' ')
        data[1:] = [float(x) for x in data[1:]]
        self.classname = data[0]
        self.xmin = data[1] 
        self.ymin = data[2]
        self.xmax = data[1]+data[3]
        self.ymax = data[2]+data[4]
        self.box2d = np.array([self.xmin,self.ymin,self.xmax,self.ymax])
        self.centroid = np.array([data[5],data[6],data[7]])
        self.unused_dimension = np.array([data[8],data[9],data[10]])
        self.w = data[8]
        self.l = data[9]
        self.h = data[10]
        self.orientation = np.zeros((3,))
        self.orientation[0] = data[11]
        self.orientation[1] = data[12]
        self.heading_angle = -1 * np.arctan2(self.orientation[1], self.orientation[0])

class scannet_Calibration(object):
    ''' Calibration matrices and utils
        We define five coordinate system in SUN RGBD dataset

        camera coodinate:
            Z is forward, Y is downward, X is rightward

        depth coordinate:
            Just change axis order and flip up-down axis from camera coord

        upright depth coordinate: tilted depth coordinate by Rtilt such that Z is gravity direction,
            Z is up-axis, Y is forward, X is right-ward

        upright camera coordinate:
            Just change axis order and flip up-down axis from upright depth coordinate

        image coordinate:
            ----> x-axis (u)
           |
           v
            y-axis (v) 

        depth points are stored in upright depth coordinate.
        labels for 3d box (basis, centroid, size) are in upright depth coordinate.
        2d boxes are in image coordinate

        We generate frustum point cloud and 3d box in upright camera coordinate
    '''

    def __init__(self, calib_filepath):
        lines = [line.rstrip() for line in open(calib_filepath)]
        Rtilt = np.array([float(x) for x in lines[0].split(' ')])
        self.Rtilt = np.reshape(Rtilt, (3,3), order='F')
        K = np.array([float(x) for x in lines[1].split(' ')])
        self.K = np.reshape(K, (3,3), order='F')
        self.f_u = self.K[0,0]
        self.f_v = self.K[1,1]
        self.c_u = self.K[0,2]
        self.c_v = self.K[1,2]
   
    def project_upright_depth_to_camera(self, pc):
        ''' project point cloud from depth coord to camera coordinate
            Input: (N,3) Output: (N,3)
        '''
        # Project upright depth to depth coordinate
        pc2 = np.dot(np.transpose(self.Rtilt), np.transpose(pc[:,0:3])) # (3,n)
        return flip_axis_to_camera(np.transpose(pc2))

    def project_upright_depth_to_image(self, pc):
        ''' Input: (N,3) Output: (N,2) UV and (N,) depth '''
        pc2 = self.project_upright_depth_to_camera(pc)
        uv = np.dot(pc2, np.transpose(self.K)) # (n,3)
        uv[:,0] /= uv[:,2]
        uv[:,1] /= uv[:,2]
        return uv[:,0:2], pc2[:,2]

    def project_upright_depth_to_upright_camera(self, pc):
        return flip_axis_to_camera(pc)

    def project_upright_camera_to_upright_depth(self, pc):
        return flip_axis_to_depth(pc)

    def project_image_to_camera(self, uv_depth):
        n = uv_depth.shape[0]
        x = ((uv_depth[:,0]-self.c_u)*uv_depth[:,2])/self.f_u
        y = ((uv_depth[:,1]-self.c_v)*uv_depth[:,2])/self.f_v
        pts_3d_camera = np.zeros((n,3))
        pts_3d_camera[:,0] = x
        pts_3d_camera[:,1] = y
        pts_3d_camera[:,2] = uv_depth[:,2]
        return pts_3d_camera

    def project_image_to_upright_camerea(self, uv_depth):
        pts_3d_camera = self.project_image_to_camera(uv_depth)
        pts_3d_depth = flip_axis_to_depth(pts_3d_camera)
        pts_3d_upright_depth = np.transpose(np.dot(self.Rtilt, np.transpose(pts_3d_depth)))
        return self.project_upright_depth_to_upright_camera(pts_3d_upright_depth)

 
 
def rotx(t):
    """Rotation about the x-axis."""
    c = np.cos(t)
    s = np.sin(t)
    return np.array([[1,  0,  0],
                     [0,  c, -s],
                     [0,  s,  c]])


def roty(t):
    """Rotation about the y-axis."""
    c = np.cos(t)
    s = np.sin(t)
    return np.array([[c,  0,  s],
                     [0,  1,  0],
                     [-s, 0,  c]])


def rotz(t):
    """Rotation about the z-axis."""
    c = np.cos(t)
    s = np.sin(t)
    return np.array([[c, -s,  0],
                     [s,  c,  0],
                     [0,  0,  1]])


def transform_from_rot_trans(R, t):
    """Transforation matrix from rotation matrix and translation vector."""
    R = R.reshape(3, 3)
    t = t.reshape(3, 1)
    return np.vstack((np.hstack([R, t]), [0, 0, 0, 1]))


def inverse_rigid_trans(Tr):
    """Inverse a rigid body transform matrix (3x4 as [R|t])
        [R'|-R't; 0|1]
    """ 
    inv_Tr = np.zeros_like(Tr) # 3x4
    inv_Tr[0:3,0:3] = np.transpose(Tr[0:3,0:3])
    inv_Tr[0:3,3] = np.dot(-np.transpose(Tr[0:3,0:3]), Tr[0:3,3])
    return inv_Tr

def read_scannet_label(label_filename):
    lines = [line.rstrip() for line in open(label_filename)]
    objects = [SUNObject3d(line) for line in lines]
    return objects

def load_image(img_filename):
    return cv2.imread(img_filename)

def load_depth_points(depth_filename):
    depth = np.loadtxt(depth_filename)
    return depth

def load_depth_points_mat(depth_filename):
    depth = sio.loadmat(depth_filename)['instance']
    return depth

def random_shift_box2d(box2d, shift_ratio=0.1):
    ''' Randomly shift box center, randomly scale width and height 
    '''
    r = shift_ratio
    xmin,ymin,xmax,ymax = box2d
    h = ymax-ymin
    w = xmax-xmin
    cx = (xmin+xmax)/2.0
    cy = (ymin+ymax)/2.0
    cx2 = cx + w*r*(np.random.random()*2-1)
    cy2 = cy + h*r*(np.random.random()*2-1)
    h2 = h*(1+np.random.random()*2*r-r) # 0.9 to 1.1
    w2 = w*(1+np.random.random()*2*r-r) # 0.9 to 1.1
    return np.array([cx2-w2/2.0, cy2-h2/2.0, cx2+w2/2.0, cy2+h2/2.0])
 
def in_hull(p, hull):
    from scipy.spatial import Delaunay
    if not isinstance(hull,Delaunay):
        hull = Delaunay(hull)
    return hull.find_simplex(p)>=0

def extract_pc_in_box3d(pc, box3d):
    ''' pc: (N,3), box3d: (8,3) '''
    box3d_roi_inds = in_hull(pc[:,0:3], box3d)
    return pc[box3d_roi_inds,:], box3d_roi_inds


def my_compute_box_3d(center, size, heading_angle):
    R = rotz(-1*heading_angle)
    l,w,h = size
    x_corners = [-l,l,l,-l,-l,l,l,-l]
    y_corners = [w,w,-w,-w,w,w,-w,-w]
    z_corners = [h,h,h,h,-h,-h,-h,-h]
    corners_3d = np.dot(R, np.vstack([x_corners, y_corners, z_corners]))
    corners_3d[0,:] += center[0]
    corners_3d[1,:] += center[1]
    corners_3d[2,:] += center[2]
    return np.transpose(corners_3d)


def compute_box_3d(obj, calib):
    ''' Takes an object and a projection matrix (P) and projects the 3d
        bounding box into the image plane.
        Returns:
            corners_2d: (8,2) array in image coord.
            corners_3d: (8,3) array in in upright depth coord.
    '''
    center = obj.centroid

    # compute rotational matrix around yaw axis
    R = rotz(-1*obj.heading_angle)
    #b,a,c = dimension
    #print R, a,b,c
    
    # 3d bounding box dimensions
    l = obj.l # along heading arrow
    w = obj.w # perpendicular to heading arrow
    h = obj.h

    # rotate and translate 3d bounding box
    x_corners = [-l,l,l,-l,-l,l,l,-l]
    y_corners = [w,w,-w,-w,w,w,-w,-w]
    z_corners = [h,h,h,h,-h,-h,-h,-h]
    corners_3d = np.dot(R, np.vstack([x_corners, y_corners, z_corners]))
    corners_3d[0,:] += center[0]
    corners_3d[1,:] += center[1]
    corners_3d[2,:] += center[2]

    # project the 3d bounding box into the image plane
    corners_2d,_ = calib.project_upright_depth_to_image(np.transpose(corners_3d))
    #print 'corners_2d: ', corners_2d
    return corners_2d, np.transpose(corners_3d)

def compute_orientation_3d(obj, calib):
    ''' Takes an object and a projection matrix (P) and projects the 3d
        object orientation vector into the image plane.
        Returns:
            orientation_2d: (2,2) array in image coord.
            orientation_3d: (2,3) array in depth coord.
    '''
    
    # orientation in object coordinate system
    ori = obj.orientation
    orientation_3d = np.array([[0, ori[0]],[0, ori[1]],[0,0]])
    center = obj.centroid
    orientation_3d[0,:] = orientation_3d[0,:] + center[0]
    orientation_3d[1,:] = orientation_3d[1,:] + center[1]
    orientation_3d[2,:] = orientation_3d[2,:] + center[2]
    
    # project orientation into the image plane
    orientation_2d,_ = calib.project_upright_depth_to_image(np.transpose(orientation_3d))
    return orientation_2d, np.transpose(orientation_3d)

def draw_projected_box3d(image, qs, color=(255,255,255), thickness=2):
    ''' Draw 3d bounding box in image
        qs: (8,2) array of vertices for the 3d box in following order:
            1 -------- 0
           /|         /|
          2 -------- 3 .
          | |        | |
          . 5 -------- 4
          |/         |/
          6 -------- 7
    '''
    qs = qs.astype(np.int32)
    for k in range(0,4):
       #http://docs.enthought.com/mayavi/mayavi/auto/mlab_helper_functions.html
       i,j=k,(k+1)%4
       cv2.line(image, (qs[i,0],qs[i,1]), (qs[j,0],qs[j,1]), color, thickness, cv2.CV_AA) # use LINE_AA for opencv3

       i,j=k+4,(k+1)%4 + 4
       cv2.line(image, (qs[i,0],qs[i,1]), (qs[j,0],qs[j,1]), color, thickness, cv2.CV_AA)

       i,j=k,k+4
       cv2.line(image, (qs[i,0],qs[i,1]), (qs[j,0],qs[j,1]), color, thickness, cv2.CV_AA)
    return image


import pickle
import gzip

def save_zipped_pickle(obj, filename, protocol=-1):
    with gzip.open(filename, 'wb') as f:
        pickle.dump(obj, f, protocol)

def load_zipped_pickle(filename):
    with gzip.open(filename, 'rb') as f:
        loaded_object = pickle.load(f)
        return loaded_object
