from typing import Dict

import torch

from openfold.np.residue_constants import RESTYPE_ATOM37_MASK
from proteinfoundation.nn.feature_factory import FeatureFactory, FeatureFactoryUidxMotif
from proteinfoundation.nn.modules.attn_n_transition import MultiheadAttnAndTransition
from proteinfoundation.nn.modules.pair_update import PairReprUpdate
from proteinfoundation.nn.modules.seq_transition_af3 import Transition
from proteinfoundation.nn.modules.pair_rep_initial import PairReprBuilder


def get_atom_mask(device: torch.device = None):
    return torch.from_numpy(RESTYPE_ATOM37_MASK).to(dtype=torch.bool, device=device)


class LocalLatentsTransformerMotifUidx(torch.nn.Module):

    def __init__(self, **kwargs):

        super(LocalLatentsTransformerMotifUidx, self).__init__()
        self.nlayers = kwargs["nlayers"]
        self.token_dim = kwargs["token_dim"]
        self.pair_repr_dim = kwargs["pair_repr_dim"]
        self.update_pair_repr = kwargs["update_pair_repr"]
        self.update_pair_repr_every_n = kwargs["update_pair_repr_every_n"]
        self.use_tri_mult = kwargs["use_tri_mult"]
        self.use_qkln = kwargs["use_qkln"]
        self.output_param = kwargs["output_parameterization"]

        self.init_repr_factory = FeatureFactory(
            feats=kwargs["feats_seq"],
            dim_feats_out=kwargs["token_dim"],
            use_ln_out=False,
            mode="seq",
            **kwargs,
        )

        self.cond_factory = FeatureFactory(
            feats=kwargs["feats_cond_seq"],
            dim_feats_out=kwargs["dim_cond"],
            use_ln_out=False,
            mode="seq",
            **kwargs,
        )

        self.motif_uidx_factory = FeatureFactoryUidxMotif(
            dim_feats_out=kwargs["token_dim"],
            use_ln_out=False,
            **kwargs,
        )

        self.transition_c_1 = Transition(kwargs["dim_cond"], expansion_factor=2)
        self.transition_c_2 = Transition(kwargs["dim_cond"], expansion_factor=2)

        self.pair_repr_builder = PairReprBuilder(
            feats_repr=kwargs["feats_pair_repr"],
            feats_cond=kwargs["feats_pair_cond"],
            dim_feats_out=kwargs["pair_repr_dim"],
            dim_cond_pair=kwargs["dim_cond"],
            **kwargs,
        )

        self.transformer_layers = torch.nn.ModuleList(
            [
                MultiheadAttnAndTransition(
                    dim_token=self.token_dim,
                    dim_pair=self.pair_repr_dim,
                    nheads=kwargs["nheads"],
                    dim_cond=kwargs["dim_cond"],
                    residual_mha=True,
                    residual_transition=True,
                    parallel_mha_transition=False,
                    use_attn_pair_bias=True,
                    use_qkln=self.use_qkln,
                )
                for _ in range(self.nlayers)
            ]
        )

        if self.update_pair_repr:
            self.pair_update_layers = torch.nn.ModuleList(
                [
                    (
                        PairReprUpdate(
                            token_dim=kwargs["token_dim"],
                            pair_dim=kwargs["pair_repr_dim"],
                            use_tri_mult=self.use_tri_mult,
                        )
                        if i % self.update_pair_repr_every_n == 0
                        else None
                    )
                    for i in range(self.nlayers - 1)
                ]
            )

        self.local_latents_linear = torch.nn.Sequential(
            torch.nn.LayerNorm(self.token_dim),
            torch.nn.Linear(self.token_dim, kwargs["latent_dim"], bias=False),
        )
        self.ca_linear = torch.nn.Sequential(
            torch.nn.LayerNorm(self.token_dim),
            torch.nn.Linear(self.token_dim, 3, bias=False),
        )

    def forward(self, input: Dict) -> Dict[str, Dict[str, torch.Tensor]]:

        mask = input["mask"]

        c = self.cond_factory(input)
        c = self.transition_c_2(self.transition_c_1(c, mask), mask)

        motif_uidx, motif_mask = self.motif_uidx_factory(input)

        seq_f_repr = self.init_repr_factory(input)
        seqs = seq_f_repr * mask[..., None]

        pair_rep = self.pair_repr_builder(input)

        b, n_orig, _ = seqs.shape
        _, n_motif, _ = motif_uidx.shape
        dim_pair = pair_rep.shape[-1]
        dim_cond = c.shape[-1]

        seqs = torch.cat([seqs, motif_uidx], dim=1)
        zero_tensor = torch.zeros(b, n_motif, dim_cond, device=seqs.device)
        c = torch.cat([c, zero_tensor], dim=1)

        mask = torch.cat([mask, motif_mask], dim=1)

        zero_pad_d1 = torch.zeros(b, n_motif, n_orig, dim_pair, device=seqs.device)
        pair_rep = torch.cat([pair_rep, zero_pad_d1], dim=1)

        zero_pad_d2 = torch.zeros(
            b, n_orig + n_motif, n_motif, dim_pair, device=seqs.device
        )
        pair_rep = torch.cat([pair_rep, zero_pad_d2], dim=2)

        for i in range(self.nlayers):
            seqs = self.transformer_layers[i](seqs, pair_rep, c, mask)

            if self.update_pair_repr:
                if i < self.nlayers - 1:
                    if self.pair_update_layers[i] is not None:
                        pair_rep = self.pair_update_layers[i](seqs, pair_rep, mask)

        local_latents_out = self.local_latents_linear(seqs) * mask[..., None]
        ca_nm_out = self.ca_linear(seqs) * mask[..., None]

        local_latents_out = local_latents_out[:, :n_orig, :]
        ca_nm_out = ca_nm_out[:, :n_orig, :]
        nn_out = {}
        nn_out["bb_ca"] = {self.output_param["bb_ca"]: ca_nm_out}
        nn_out["local_latents"] = {
            self.output_param["local_latents"]: local_latents_out
        }
        return nn_out
