import os
import pathlib
import tarfile


def flatten_directory(
    directory: pathlib.Path,
    replace_ending: bool = False,
    old_ending: str = ".ent",
    new_ending: str = ".pdb",
) -> None:

    for file_path in directory.rglob("*"):
        if file_path.is_file():
            new_file_path = directory / (
                file_path.stem
                + (
                    file_path.suffix.replace(old_ending, new_ending)
                    if replace_ending
                    else file_path.suffix
                )
            )
            file_path.rename(new_file_path)

    for dir_path in directory.rglob("*"):
        if dir_path.is_dir() and not any(dir_path.iterdir()):
            dir_path.rmdir()


def extract_archive(
    tar_file: pathlib.Path, destination_dir: pathlib.Path, extracted_dir_name: str
) -> pathlib.Path:

    destination_dir.mkdir(parents=True, exist_ok=True)

    renamed_extracted_dir = destination_dir / extracted_dir_name

    with tarfile.open(tar_file, "r:gz") as tar:
        top_dir = os.path.commonpath(tar.getnames())
        tar.extractall(path=destination_dir, filter="data")

    tar_file.unlink()

    extracted_dir = pathlib.Path(destination_dir) / top_dir
    extracted_dir.rename(renamed_extracted_dir)

    return renamed_extracted_dir
