# Copyright (c) 2021, NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#
# SPDX-FileCopyrightText: Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES
# SPDX-License-Identifier: MIT

import argparse
import pathlib

from se3_transformer.data_loading import QM9DataModule
from se3_transformer.data_loading import ShapeNetModule
from se3_transformer.model import IOSE3Transformer, SE3TransformerPooled
from se3_transformer.runtime.utils import str2bool
import os

PARSER = argparse.ArgumentParser(description='SE(3)-Transformer')

paths = PARSER.add_argument_group('Paths')
paths.add_argument('--data_path', type=pathlib.Path, default=pathlib.Path('/Datasets/ShapeNet'),
        help='Directory where the data is located or should be downloaded')
paths.add_argument('--config', type=pathlib.Path,  default=pathlib.Path('configs/eqat_onet.yaml'), 
        help='Path to configuration file.')
paths.add_argument('--log_dir', type=pathlib.Path, default=pathlib.Path('tensorboard'), 
        help='Directory where the results logs should be saved (relative to base dir)')
paths.add_argument('--base_dir', type=pathlib.Path, default=pathlib.Path('./results'),
        help='Base directory for the experiment.')            
paths.add_argument('--dllogger_name', type=str, default='dllogger_results.json',
        help='Name for the resulting DLLogger JSON file')
paths.add_argument('--save_ckpt_path', type=pathlib.Path, default='trained_model.pth',
        help='File where the checkpoint should be saved')
paths.add_argument('--load_ckpt_path', type=pathlib.Path, default=None,
        help='File of the checkpoint to be loaded')
paths.add_argument('--vis_dir', type=pathlib.Path, default=pathlib.Path('visualization'),
        help='Directory where the visualization should be saved')
paths.add_argument('--envir', type=str, default='cluster', choices=['cluster','aws'],
        help='Enviroment where the script will run')


optimizer = PARSER.add_argument_group('Optimizer')
optimizer.add_argument('--optimizer', choices=['adam', 'sgd', 'lamb'], default='adam')
optimizer.add_argument('--learning_rate', '--lr', dest='learning_rate', type=float, default=1e-4) #1e-4? 0.002
optimizer.add_argument('--min_learning_rate', '--min_lr', dest='min_learning_rate', type=float, default=None)
optimizer.add_argument('--momentum', type=float, default=0.9)
optimizer.add_argument('--weight_decay', type=float, default=0.1)

PARSER.add_argument('--epochs', type=int, default=100, help='Number of training epochs')
PARSER.add_argument('--batch_size', type=int, default=100, help='Batch size')
PARSER.add_argument('--mini_batch', type=int, default=1, help='Number of mini batches')
PARSER.add_argument('--kmode', type=str, default='full', help='Attention Neighbourhood for the pointcloud')
PARSER.add_argument('--knnq', type=int, default=50, help='Attention Neighbourhood for the query points')
PARSER.add_argument('--seed', type=int, default=None, help='Set a seed globally')
PARSER.add_argument('--num_workers', type=int, default=4, help='Number of dataloading workers')
PARSER.add_argument('--fixed_percentage', type=str2bool, nargs='?', const=True, default=False, help='Setting fixed percentage true')
PARSER.add_argument('--iou_number_points', help='Number of query points for evaluation', type=int, default=1000)
PARSER.add_argument('--val_mini_batch', help='Number of minibatches for evaluation', type=int, default=4)
PARSER.add_argument('--threshold_occ', help='Threshold level for occupied voxels', type=float, default=0.2)

PARSER.add_argument('--amp', type=str2bool, nargs='?', const=True, default=False, help='Use Automatic Mixed Precision')
PARSER.add_argument('--gradient_clip', type=float, default=None, help='Clipping of the gradient norms')
PARSER.add_argument('--accumulate_grad_batches', type=int, default=1, help='Gradient accumulation')
PARSER.add_argument('--ckpt_interval', type=int, default=1, help='Save a checkpoint every N epochs')
PARSER.add_argument('--eval_interval', type=int, default=2000,
                    help='Do an evaluation round every N batches')
PARSER.add_argument('--visualize_interval', dest='visualize_interval', type=int, default=200,
                    help='Do an evaluation round every N batches')

PARSER.add_argument('--silent', type=str2bool, nargs='?', const=True, default=False,
                    help='Minimize stdout output')
PARSER.add_argument('--wandb', type=str2bool, nargs='?', const=True, default=False,
                    help='Enable W&B logging')
PARSER.add_argument('--tensorboard', type=str2bool, nargs='?', const=True, default=True,
                    help='Enable TensorBoard logging')

PARSER.add_argument('--benchmark', type=str2bool, nargs='?', const=True, default=False,
                    help='Benchmark mode')
PARSER.add_argument('--data_rotation', type=str, default='aligned', choices=['aligned','so3','z'],
                    help='Rotation augmentation')
PARSER.add_argument('--data_rotation_folder', type=str, default='/rotations', help='Precomputed Rotation folder')


#QM9DataModule.add_argparse_args(PARSER)
#SE3TransformerPooled.add_argparse_args(PARSER)
ShapeNetModule.add_argparse_args(PARSER)
IOSE3Transformer.add_argparse_args(PARSER)
