import math
from functools import partial
from collections import OrderedDict

import numpy as np
import scipy.io as sio
import torch
import torch.nn as nn
from torch import _assert
from timm.models.vision_transformer import Block
from timm.models.layers import to_2tuple, DropPath

import clip
from pos_embed import get_2d_sincos_pos_embed


class PatchEmbed(nn.Module):
    """ 2D Image to Patch Embedding
    """

    def __init__(self, img_size=(224, 224), patch_size=16, in_chans=3, embed_dim=768, norm_layer=None, flatten=True):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)

        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]
        self.flatten = flatten

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        _assert(H == self.img_size[0], f"Input image height ({H}) doesn't match model ({self.img_size[0]}).")
        _assert(W == self.img_size[1], f"Input image width ({W}) doesn't match model ({self.img_size[1]}).")
        x = self.proj(x)
        if self.flatten:
            x = x.flatten(2).transpose(1, 2)  # BCHW -> BNC
        x = self.norm(x)
        return x

class QuickGELU(nn.Module):
    def forward(self, x: torch.Tensor):
        return x * torch.sigmoid(1.702 * x)

class ResidualAttentionBlock(nn.Module):
    def __init__(self, d_model: int, n_head: int, attn_mask: torch.Tensor = None):
        super().__init__()

        self.attn = nn.MultiheadAttention(d_model, n_head)
        self.ln_1 = LayerNorm(d_model)
        self.mlp = nn.Sequential(OrderedDict([
            ("c_fc", nn.Linear(d_model, d_model * 4)),
            ("gelu", QuickGELU()),
            ("c_proj", nn.Linear(d_model * 4, d_model))
        ]))
        self.ln_2 = LayerNorm(d_model)
        self.attn_mask = attn_mask

    def attention(self, x: torch.Tensor):
        self.attn_mask = self.attn_mask.to(dtype=x.dtype, device=x.device) if self.attn_mask is not None else None
        return self.attn(x, x, x, need_weights=False, attn_mask=self.attn_mask)[0]

    def forward(self, x: torch.Tensor):
        x = x + self.attention(self.ln_1(x))
        x = x + self.mlp(self.ln_2(x))
        return x

class Transformer(nn.Module):
    def __init__(self, width: int, layers: int, heads: int, attn_mask: torch.Tensor = None):
        super().__init__()
        self.width = width
        self.layers = layers
        self.resblocks = nn.Sequential(*[ResidualAttentionBlock(width, heads, attn_mask) for _ in range(layers)])

    def forward(self, x: torch.Tensor):
        return self.resblocks(x)

class LayerNorm(nn.LayerNorm):
    """Subclass torch's LayerNorm to handle fp16."""

    def forward(self, x: torch.Tensor):
        orig_type = x.dtype
        ret = super().forward(x.type(torch.float32))
        return ret.type(orig_type)

class Text_encoder(nn.Module):
    def __init__(self,
                 embed_dim: int,
                 context_length: int,
                 vocab_size: int,
                 transformer_width: int,
                 transformer_heads: int,
                 transformer_layers: int
                 ):
        super().__init__()
        self.context_length = context_length


        self.transformer = Transformer(
            width=transformer_width,
            layers=transformer_layers,
            heads=transformer_heads,
            attn_mask=self.build_attention_mask()
        )

        self.vocab_size = vocab_size
        self.token_embedding = nn.Embedding(vocab_size, transformer_width)
        self.positional_embedding = nn.Parameter(torch.empty(self.context_length, transformer_width))
        self.ln_final = LayerNorm(transformer_width) #在模型中归一化最终的输出
        self.text_projection = nn.Parameter(torch.empty(transformer_width, embed_dim)) #用于将Transformer模型的输出投影到一个更低维度的表示,transformer_width是输入维度，而embed_dim是输出维度
        self.logit_scale = nn.Parameter(torch.ones([]) * np.log(1 / 0.07)) #调整模型输出的尺度
        self.initialize_parameters()

    def initialize_parameters(self):
        nn.init.normal_(self.token_embedding.weight, std=0.02)
        nn.init.normal_(self.positional_embedding, std=0.01)

        proj_std = (self.transformer.width ** -0.5) * ((2 * self.transformer.layers) ** -0.5)
        attn_std = self.transformer.width ** -0.5
        fc_std = (2 * self.transformer.width) ** -0.5
        for block in self.transformer.resblocks:
            nn.init.normal_(block.attn.in_proj_weight, std=attn_std)
            nn.init.normal_(block.attn.out_proj.weight, std=proj_std)
            nn.init.normal_(block.mlp.c_fc.weight, std=fc_std)
            nn.init.normal_(block.mlp.c_proj.weight, std=proj_std)

        if self.text_projection is not None:
            nn.init.normal_(self.text_projection, std=self.transformer.width ** -0.5)

    def build_attention_mask(self):
        # lazily create causal attention mask, with full attention between the vision tokens
        # pytorch uses additive attention mask; fill with -inf
        mask = torch.empty(self.context_length, self.context_length)
        mask.fill_(float("-inf"))
        mask.triu_(1)  # zero out the lower diagonal
        return mask

    def forward(self, text):
        x = self.token_embedding(text).type(torch.float32)  # [batch_size, n_ctx, d_model]

        x = x + self.positional_embedding.type(torch.float32)
        x = x.permute(1, 0, 2)  # NLD -> LND
        x = self.transformer(x)
        x = x.permute(1, 0, 2)  # LND -> NLD
        x = self.ln_final(x).type(torch.float32)

        # x.shape = [batch_size, n_ctx, transformer.width]
        # take features from the eot embedding (eot_token is the highest number in each sequence)
        x = x[torch.arange(x.shape[0]), text.argmax(dim=-1)] @ self.text_projection

        return x

class vit_HSI_LIDAR(nn.Module):


    def __init__(self, img_size=(224, 224), patch_size=16, num_classes=1000, in_chans=3, in_chans_LIDAR = 1, hid_chans=32,
                 hid_chans_LIDAR=128,embed_dim=1024, depth=24, num_heads=16, drop_rate=0., attn_drop_rate=0., drop_path_rate=0.,
                 mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False, global_pool=False):
        super().__init__()
        self.patch_size = patch_size

        # --------------------------------------------------------------------------
        # HSI
        # MAE encoder specifics
        self.dimen_redu = nn.Sequential(
            nn.Conv2d(in_chans, hid_chans, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(hid_chans),
            nn.ReLU(),

            nn.Conv2d(hid_chans, hid_chans, 1, 1, 0, bias=True),
            nn.BatchNorm2d(hid_chans),
            nn.ReLU(),
        )


        self.patch_embed = PatchEmbed(img_size, patch_size, hid_chans, embed_dim)
        num_patches = self.patch_embed.num_patches

        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim),
                                      requires_grad=True)  # fixed sin-cos embedding

        self.blocks = nn.ModuleList([
            Block(embed_dim, num_heads, mlp_ratio, drop=drop_rate, attn_drop=attn_drop_rate, drop_path=drop_path_rate,
                  qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)
        self.head = nn.Linear(embed_dim * 2, num_classes, bias=True)

        self.global_pool = global_pool
        if self.global_pool:
            self.fc_norm = norm_layer(embed_dim)
            del self.norm

        # LIDAR
        self.dimen_redu_LIDAR = nn.Sequential(
            nn.Conv2d(in_chans, hid_chans_LIDAR, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(hid_chans_LIDAR),
            nn.ReLU(),

            nn.Conv2d(hid_chans_LIDAR, hid_chans_LIDAR, 1, 1, 0, bias=True),
            nn.BatchNorm2d(hid_chans_LIDAR),
            nn.ReLU(),
        )

        # --------------------------------------------------------------------------
        self.patch_embed_LIDAR = PatchEmbed(img_size, patch_size, hid_chans_LIDAR, embed_dim)
        num_patches = self.patch_embed_LIDAR.num_patches

        self.cls_token_LIDAR = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed_LIDAR = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim),
                                      requires_grad=True)  # fixed sin-cos embedding

        self.blocks_LIDAR = nn.ModuleList([
            Block(embed_dim, num_heads, mlp_ratio, drop=drop_rate, attn_drop=attn_drop_rate, drop_path=drop_path_rate,
                  qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm_LIDAR = norm_layer(embed_dim)
        self.global_pool_LIDAR = global_pool
        if self.global_pool_LIDAR:
            self.fc_norm_LIDAR = norm_layer(embed_dim)
            del self.norm_LIDAR

        self.mlp = nn.Sequential(
                nn.Linear(embed_dim * 2, embed_dim, bias=True),
                nn.SiLU(),
                nn.Linear(embed_dim, embed_dim, bias=True),
            )
        self.mlp_lidar = nn.Sequential(
                nn.Linear(embed_dim, embed_dim, bias=True),
                nn.SiLU(),
                nn.Linear(embed_dim, embed_dim, bias=True),
            )
    def initialize_weights(self):
        # initialization
        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed = get_2d_sincos_pos_embed(self.pos_embed.shape[-1], int(self.patch_embed.num_patches ** .5),
                                            cls_token=True)
        self.pos_embed.data.copy_(torch.from_numpy(pos_embed).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w = self.patch_embed.proj.weight.data
        torch.nn.init.xavier_uniform_(w.view([w.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token, std=.02)

        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed_LIDAR = get_2d_sincos_pos_embed(self.pos_embed_LIDAR.shape[-1], int(self.patch_embed_LIDAR.num_patches ** .5),
                                            cls_token=True)
        self.pos_embed_LIDAR.data.copy_(torch.from_numpy(pos_embed_LIDAR).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w_LIDAR = self.patch_embed_LIDAR.proj.weight.data
        torch.nn.init.xavier_uniform_(w_LIDAR.view([w_LIDAR.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token_LIDAR, std=.02)

        # initialize nn.Linear and nn.LayerNorm
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # we use xavier_uniform following official JAX ViT:
            torch.nn.init.xavier_uniform_(m.weight)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def forward_features(self, x, x_LIDAR):
        x = self.dimen_redu(x)
        x_LIDAR = self.dimen_redu_LIDAR(x_LIDAR)

        # embed patches
        x = self.patch_embed(x)
        x_LIDAR = self.patch_embed_LIDAR(x_LIDAR)

        # add pos embed w/o cls token
        x = x + self.pos_embed[:, 1:, :]
        x_LIDAR = x_LIDAR + self.pos_embed_LIDAR[:, 1:, :]

        # append cls token
        cls_token = self.cls_token + self.pos_embed[:, :1, :]
        cls_tokens = cls_token.expand(x.shape[0], -1, -1)


        x = torch.cat((cls_tokens, x), dim=1)


        cls_token_LIDAR = self.cls_token_LIDAR + self.pos_embed_LIDAR[:, :1, :]
        cls_tokens_LIDAR = cls_token_LIDAR.expand(x_LIDAR.shape[0], -1, -1)

        x_LIDAR = torch.cat((cls_tokens_LIDAR, x_LIDAR), dim=1)

        # apply Transformer blocks
        for blk in self.blocks:
            x = blk(x)
        if self.global_pool:
            x = x[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome = self.fc_norm(x)
        else:
            x = self.norm(x)
            outcome = x[:, 0]

        for blk_LIDAR in self.blocks:
            x_LIDAR = blk_LIDAR(x_LIDAR)
        if self.global_pool_LIDAR:
            x_LIDAR = x_LIDAR[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome_LIDAR = self.fc_norm(x_LIDAR)
        else:
            x_LIDAR = self.norm(x_LIDAR)
            outcome_LIDAR = x_LIDAR[:, 0]

        outcome_all = torch.cat((outcome, outcome_LIDAR),dim = 1)
        return outcome_all

    def forward_features_lidar(self, x_LIDAR):
        x_LIDAR = self.dimen_redu_LIDAR(x_LIDAR)

        # embed patches
        x_LIDAR = self.patch_embed_LIDAR(x_LIDAR)

        # add pos embed w/o cls token
        x_LIDAR = x_LIDAR + self.pos_embed_LIDAR[:, 1:, :]

        # append cls token

        cls_token_LIDAR = self.cls_token_LIDAR + self.pos_embed_LIDAR[:, :1, :]
        cls_tokens_LIDAR = cls_token_LIDAR.expand(x_LIDAR.shape[0], -1, -1)

        x_LIDAR = torch.cat((cls_tokens_LIDAR, x_LIDAR), dim=1)

        # apply Transformer blocks
        for blk_LIDAR in self.blocks:
            x_LIDAR = blk_LIDAR(x_LIDAR)
        if self.global_pool_LIDAR:
            x_LIDAR = x_LIDAR[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome_LIDAR = self.fc_norm(x_LIDAR)
        else:
            x_LIDAR = self.norm(x_LIDAR)
            outcome_LIDAR = x_LIDAR[:, 0]

        return outcome_LIDAR

    def forward(self, x, x_LIDAR):
        x = self.forward_features(x, x_LIDAR)
        x = self.mlp(x)
        # logits = self.li(x)
        # x = F.normalize(x, p=2, dim=1)
        return x
    def forward_lidar(self, x_LIDAR):
        x_LIDAR = self.forward_features_lidar(x_LIDAR)
        x_LIDAR = self.mlp_lidar(x_LIDAR)
        # logits = self.li(x)
        # x = F.normalize(x, p=2, dim=1)
        return x_LIDAR

class vit_HSI(nn.Module):


    def __init__(self, img_size=(224, 224), patch_size=16, num_classes=1000, in_chans=3, in_chans_LIDAR=1, hid_chans=32,
                 hid_chans_LIDAR=128, embed_dim=1024, depth=24, num_heads=16, drop_rate=0., attn_drop_rate=0.,
                 drop_path_rate=0.,
                 mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False, global_pool=False):
        super().__init__()
        self.patch_size = patch_size

        # --------------------------------------------------------------------------
        # HSI

        self.dimen_redu = nn.Sequential(
            nn.Conv2d(in_chans, hid_chans, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(hid_chans),
            nn.ReLU(),

            nn.Conv2d(hid_chans, hid_chans, 1, 1, 0, bias=True),
            nn.BatchNorm2d(hid_chans),
            nn.ReLU(),
        )

        # --------------------------------------------------------------------------

        self.patch_embed = PatchEmbed(img_size, patch_size, hid_chans, embed_dim)
        num_patches = self.patch_embed.num_patches

        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim),
                                      requires_grad=True)  # fixed sin-cos embedding
        self.prompt = nn.Parameter(torch.zeros(1, 16, embed_dim))

        self.blocks = nn.ModuleList([
            Block(embed_dim, num_heads, mlp_ratio, drop=drop_rate, attn_drop=attn_drop_rate, drop_path=drop_path_rate,
                  qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)
        self.head = nn.Linear(embed_dim * 2, num_classes, bias=True)

        self.global_pool = global_pool
        if self.global_pool:
            self.fc_norm = norm_layer(embed_dim)
            del self.norm

        self.mlp = nn.Sequential(
            nn.Linear(embed_dim, embed_dim, bias=True),
            nn.SiLU(),
            nn.Linear(embed_dim, embed_dim, bias=True),
        )

    def initialize_weights(self):
        # initialization
        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed = get_2d_sincos_pos_embed(self.pos_embed.shape[-1], int(self.patch_embed.num_patches ** .5),
                                            cls_token=True)
        self.pos_embed.data.copy_(torch.from_numpy(pos_embed).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w = self.patch_embed.proj.weight.data
        torch.nn.init.xavier_uniform_(w.view([w.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token, std=.02)
        torch.nn.init.normal_(self.prompt, std=.02)

        # initialize nn.Linear and nn.LayerNorm
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # we use xavier_uniform following official JAX ViT:
            torch.nn.init.xavier_uniform_(m.weight)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def forward_features(self, x):
        x = self.dimen_redu(x)

        # embed patches
        x = self.patch_embed(x)

        # add pos embed w/o cls token
        x = x + self.pos_embed[:, 1:, :]

        # append cls token
        cls_token = self.cls_token + self.pos_embed[:, :1, :]
        cls_tokens = cls_token.expand(x.shape[0], -1, -1)

        x = torch.cat((cls_tokens, x), dim=1)

        #############  attend prompt
        prompt = self.prompt
        prompt = prompt.expand(x.shape[0], -1, -1)
        x = torch.cat((x, prompt), dim=1)

        # apply Transformer blocks
        for blk in self.blocks:
            x = blk(x)
            promptnew = self.prompt
            promptnew = promptnew.expand(x.shape[0], -1, -1)
            x = torch.cat((x, promptnew), dim=1)
        if self.global_pool:
            x = x[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome = self.fc_norm(x)
        else:
            x = self.norm(x)
            outcome = x[:, 0]
        return outcome

    def forward(self, x):
        x = self.forward_features(x)
        x = self.mlp(x)
        # x = F.normalize(x, p=2, dim=1)
        return x

class vit_LIDAR(nn.Module):


    def __init__(self, img_size=(224, 224), patch_size=16, num_classes=1000, in_chans=3, in_chans_LIDAR = 1, hid_chans=32,
                 hid_chans_LIDAR=128,embed_dim=1024, depth=24, num_heads=16, drop_rate=0., attn_drop_rate=0., drop_path_rate=0.,
                 mlp_ratio=4., norm_layer=nn.LayerNorm, norm_pix_loss=False, global_pool=False):
        super().__init__()
        self.patch_size = patch_size

        # --------------------------------------------------------------------------
        # HSI

        self.blocks = nn.ModuleList([
            Block(embed_dim, num_heads, mlp_ratio, drop=drop_rate, attn_drop=attn_drop_rate, drop_path=drop_path_rate,
                  qkv_bias=True, norm_layer=norm_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)
        self.head = nn.Linear(embed_dim * 2, num_classes, bias=True)

        self.global_pool = global_pool
        if self.global_pool:
            self.fc_norm = norm_layer(embed_dim)
            del self.norm

        # LIDAR

        self.dimen_redu_LIDAR = nn.Sequential(
            nn.Conv2d(in_chans, hid_chans_LIDAR, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(hid_chans_LIDAR),
            nn.ReLU(),

            nn.Conv2d(hid_chans_LIDAR, hid_chans_LIDAR, 1, 1, 0, bias=True),
            nn.BatchNorm2d(hid_chans_LIDAR),
            nn.ReLU(),
        )

        # --------------------------------------------------------------------------

        self.patch_embed_LIDAR = PatchEmbed(img_size, patch_size, hid_chans_LIDAR, embed_dim)
        num_patches = self.patch_embed_LIDAR.num_patches

        self.cls_token_LIDAR = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.pos_embed_LIDAR = nn.Parameter(torch.zeros(1, num_patches + 1, embed_dim),
                                      requires_grad=True)  # fixed sin-cos embedding
        self.norm_LIDAR = norm_layer(embed_dim)
        self.global_pool_LIDAR = global_pool
        if self.global_pool_LIDAR:
            self.fc_norm_LIDAR = norm_layer(embed_dim)
            del self.norm_LIDAR
        self.mlp = nn.Sequential(
                nn.Linear(embed_dim, embed_dim, bias=True),
                nn.SiLU(),
                nn.Linear(embed_dim, embed_dim, bias=True),
            )
    def initialize_weights(self):
        # initialization

        # initialize (and freeze) pos_embed by sin-cos embedding
        pos_embed_LIDAR = get_2d_sincos_pos_embed(self.pos_embed_LIDAR.shape[-1], int(self.patch_embed_LIDAR.num_patches ** .5),
                                            cls_token=True)
        self.pos_embed_LIDAR.data.copy_(torch.from_numpy(pos_embed_LIDAR).float().unsqueeze(0))

        # initialize patch_embed like nn.Linear (instead of nn.Conv2d)
        w_LIDAR = self.patch_embed_LIDAR.proj.weight.data
        torch.nn.init.xavier_uniform_(w_LIDAR.view([w_LIDAR.shape[0], -1]))

        # timm's trunc_normal_(std=.02) is effectively normal_(std=0.02) as cutoff is too big (2.)
        torch.nn.init.normal_(self.cls_token_LIDAR, std=.02)

        # initialize nn.Linear and nn.LayerNorm
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            # we use xavier_uniform following official JAX ViT:
            torch.nn.init.xavier_uniform_(m.weight)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def forward_features(self, x_LIDAR):
        x_LIDAR = self.dimen_redu_LIDAR(x_LIDAR)

        # embed patches
        x_LIDAR = self.patch_embed_LIDAR(x_LIDAR)

        # add pos embed w/o cls token
        x_LIDAR = x_LIDAR + self.pos_embed_LIDAR[:, 1:, :]

        # append cls token
        cls_token_LIDAR = self.cls_token_LIDAR + self.pos_embed_LIDAR[:, :1, :]
        cls_tokens_LIDAR = cls_token_LIDAR.expand(x_LIDAR.shape[0], -1, -1)

        x_LIDAR = torch.cat((cls_tokens_LIDAR, x_LIDAR), dim=1)

        # apply Transformer blocks
        for blk_LIDAR in self.blocks:
            x_LIDAR = blk_LIDAR(x_LIDAR)
        if self.global_pool_LIDAR:
            x_LIDAR = x_LIDAR[:, 1:, :].mean(dim=1)  # global pool without cls token
            outcome_LIDAR = self.fc_norm(x_LIDAR)
        else:
            x_LIDAR = self.norm(x_LIDAR)
            outcome_LIDAR = x_LIDAR[:, 0]

        return outcome_LIDAR

    def forward(self, x_LIDAR):
        x = self.forward_features(x_LIDAR)
        x = self.mlp(x)
        # logits = self.li(x)
        # x = F.normalize(x, p=2, dim=1)
        return x

class LPCnet_LIDAR(nn.Module):
    def __init__(self ,in_chans_LIDAR = 1, num_classes = 15, **kwargs):
        super(LPCnet_LIDAR, self).__init__()

        self.Shared_vision_net = vit_HSI_LIDAR(**kwargs)
        self.HSI_vision_net = vit_HSI(**kwargs)
        self.Lidar_vision_net = vit_LIDAR(**kwargs)
        self.HSI_text_net = Text_encoder(embed_dim=512,  ## 输出维度，可更改,模型最后一层就得改
                                         context_length=77,
                                         vocab_size=49408,
                                         transformer_width=512,
                                         transformer_heads=8,
                                         transformer_layers=12)

        self.lidar_dim = nn.Sequential(
            nn.Conv2d(in_chans_LIDAR, 30, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(30),
            nn.ReLU(),
            nn.Conv2d(30, 30, 1, 1, 0, bias=True),
            nn.BatchNorm2d(30),
            nn.ReLU(),
        )
        self.hsi_dim = nn.Sequential(
            nn.Conv2d(in_chans_LIDAR, 30, kernel_size=1, stride=1, padding=0, bias=True),
            nn.BatchNorm2d(30),
            nn.ReLU(),
            nn.Conv2d(30, 30, 1, 1, 0, bias=True),
            nn.BatchNorm2d(30),
            nn.ReLU(),
        )
        self.hsi_projector_vision = nn.Sequential(nn.Linear(64, 512))
        self.mlp_stage3_lidar = nn.Linear(64 *3, num_classes)

        self.softmax = nn.Softmax(dim=1)

    def forward(self, patch_lidar, HSI_text):
        with torch.no_grad():
            HSI_text = clip.tokenize(HSI_text).cuda(0)  #
            text_features = self.HSI_text_net(HSI_text)
        patch_hsi = self.hsi_dim(patch_lidar)
        patch_lidar = self.lidar_dim(patch_lidar)
        Shared_vision_features = self.Shared_vision_net.forward_lidar(patch_lidar)
        HSI_vision_features = self.HSI_vision_net(patch_hsi)
        Lidar_vision_features = self.Lidar_vision_net(patch_lidar)

        proj_HSI_vision_features = self.hsi_projector_vision(HSI_vision_features)

        vision_features = torch.cat((Shared_vision_features, HSI_vision_features, Lidar_vision_features), dim=1)

        # vision_features = Shared_vision_features
        logits = self.mlp_stage3_lidar(vision_features)
        return logits, proj_HSI_vision_features, text_features