from copy import deepcopy
import math
from functools import partialmethod

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.jit import Final
import torch.autograd as autograd

from timm.layers import Mlp, DropPath
from timm.models._registry import get_arch_name
from timm.models.vision_transformer import LayerScale, _create_vision_transformer
from timm.models.vision_transformer import default_cfgs as timm_vit_default_cfgs
from timm.models._registry import generate_default_cfgs, register_model
from timm.data import IMAGENET_INCEPTION_MEAN, IMAGENET_INCEPTION_STD, OPENAI_CLIP_MEAN, OPENAI_CLIP_STD


def mask_init(module):
    scores = torch.Tensor(module.weight.size())
    nn.init.kaiming_uniform_(scores, a=math.sqrt(5))
    return scores


class GetSubnet(autograd.Function):
    @staticmethod
    def forward(ctx, scores, k):
        # Get the supermask by sorting the scores and using the top k%
        out = scores.clone()
        _, idx = scores.flatten().sort()
        j = int((1 - k) * scores.numel())

        # flat_out and out access the same memory.
        flat_out = out.flatten()
        flat_out[idx[:j]] = 0
        flat_out[idx[j:]] = 1

        return out

    @staticmethod
    def backward(ctx, g):
        # send the gradient g straight-through on the backward pass.
        return g, None


class MultitaskMaskLinear(nn.Linear):

    def __init__(self, in_features: int, out_features: int, bias: bool = True, device=None, dtype=None, sparsity=0.5, num_tasks=9) -> None:
        super().__init__(in_features, out_features, bias, device, dtype)

        self.sparsity = sparsity
        self.num_tasks = num_tasks

        self.scores = nn.ParameterList([
            nn.Parameter(mask_init(self)) for _ in range(num_tasks)
        ])

    def forward(self, x):
        
        if self.task_idx > 0:
            subnet = GetSubnet.apply(self.scores[self.task_idx].abs(), self.sparsity)
            w = self.weight * subnet
        else:
            w = self.weight
        return F.linear(x, w, self.bias)

    def set_task_idx(self, task_idx):
        self.task_idx = task_idx


def _MultitaskMaskLinear(cls, sparsity, num_tasks):
    class _ParamMultitaskMaskLinear(cls):
        __init__ = partialmethod(cls.__init__, sparsity=sparsity, num_tasks=num_tasks)
    return _ParamMultitaskMaskLinear


class SupSupAttention(nn.Module):
    fast_attn: Final[bool]

    def __init__(
            self,
            dim,
            num_heads=8,
            qkv_bias=False,
            qk_norm=False,
            attn_drop=0.,
            proj_drop=0.,
            norm_layer=nn.LayerNorm,
            masked_linear_layer=MultitaskMaskLinear,
    ):
        super().__init__()
        assert dim % num_heads == 0, 'dim should be divisible by num_heads'
        self.num_heads = num_heads
        self.head_dim = dim // num_heads
        self.scale = self.head_dim ** -0.5
        self.fast_attn = hasattr(torch.nn.functional, 'scaled_dot_product_attention')  # FIXME

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.q_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.k_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = masked_linear_layer(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, self.head_dim).permute(2, 0, 3, 1, 4)
        q, k, v = qkv.unbind(0)
        q, k = self.q_norm(q), self.k_norm(k)

        if self.fast_attn:
            x = F.scaled_dot_product_attention(
                q, k, v,
                dropout_p=self.attn_drop.p,
            )
        else:
            q = q * self.scale
            attn = q @ k.transpose(-2, -1)
            attn = attn.softmax(dim=-1)
            attn = self.attn_drop(attn)
            x = attn @ v

        x = x.transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x, attn
    

class SupSupBlock(nn.Module):

    def __init__(
            self,
            dim,
            num_heads,
            mlp_ratio=4.,
            qkv_bias=False,
            qk_norm=False,
            drop=0.,
            attn_drop=0.,
            init_values=None,
            drop_path=0.,
            act_layer=nn.GELU,
            norm_layer=nn.LayerNorm,
            masked_linear_layer=MultitaskMaskLinear,
    ):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = SupSupAttention(
            dim,
            num_heads=num_heads,
            qkv_bias=qkv_bias,
            qk_norm=qk_norm,
            attn_drop=attn_drop,
            proj_drop=drop,
            norm_layer=norm_layer,
            masked_linear_layer=masked_linear_layer,
        )
        self.ls1 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path1 = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.identity = nn.Identity()

        self.norm2 = norm_layer(dim)
        self.mlp = Mlp(
            in_features=dim,
            hidden_features=int(dim * mlp_ratio),
            act_layer=act_layer,
            drop=drop,
        )
        self.ls2 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path2 = DropPath(drop_path) if drop_path > 0. else nn.Identity()

    def forward(self, x):
        _x, attn = self.attn(self.norm1(x))
        x = x + self.drop_path1(self.ls1(_x))
        x = self.identity(x)
        x = x + self.drop_path2(self.ls2(self.mlp(self.norm2(x))))
        return x


def _SupSupBlock(cls, masked_linear_layer):
    class _ParamSupSupBlock(cls):
        __init__ = partialmethod(cls.__init__, masked_linear_layer=masked_linear_layer)
    return _ParamSupSupBlock
