from copy import deepcopy

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.jit import Final

from timm.layers import Mlp, DropPath
from timm.models._registry import get_arch_name
from timm.models.vision_transformer import LayerScale, _create_vision_transformer
from timm.models.vision_transformer import default_cfgs as timm_vit_default_cfgs
from timm.models._registry import generate_default_cfgs, register_model
from timm.data import IMAGENET_INCEPTION_MEAN, IMAGENET_INCEPTION_STD, OPENAI_CLIP_MEAN, OPENAI_CLIP_STD


class Linear(nn.Linear):

    def __init__(self, n_features, out_features, bias=True, device=None, dtype=None):

        super(Linear, self).__init__(n_features, out_features, bias=bias, device=device, dtype=dtype)
    
    def forward(self, *inputs):
        module_inputs, forward_modifiers = inputs

        assert isinstance(module_inputs, tuple), "Module inputs must be a tuple"
    
        execute_op = forward_modifiers.get("execute_op", None)
        if execute_op is None:
            raise ValueError("execute_op not found in forward_modifiers")
        
        primitive = forward_modifiers.get("primitive", None)
        if primitive is None:
            raise ValueError("primitive not found in forward_modifiers")
        
        if execute_op:
            # Reuse, Adapt
            x = super().forward(*module_inputs)
        else:
            # Skip, New
            x = module_inputs[0] if len(module_inputs) == 1 else module_inputs
        
        return x, primitive

    def __repr__(self):
        torch_repr = super().__repr__()
        return f"Conditional{torch_repr}"


class SeparableValueLinear(nn.Module):

    def __init__(self, dim, num_heads, head_dim, bias=True, device=None, dtype=None):

        super().__init__()

        self.dim = dim
        self.num_heads = num_heads
        self.head_dim = head_dim
        self.qk = nn.Linear(dim, dim * 2, bias=bias, device=device, dtype=dtype)
        self.v = Linear(dim, dim, bias=bias, device=device, dtype=dtype)

        self._weights_copied = False

    def copy_weights(self, weights, bias):

        if not self._weights_copied:
            with torch.no_grad():
                qk_size = self.dim * 2
                qk_weight  = weights[:qk_size, :]
                qk_bias = bias[:qk_size] if bias is not None else None
                self.qk.weight.copy_(qk_weight)
                if qk_bias is not None:
                    self.qk.bias.copy_(qk_bias)

                v_weight = weights[qk_size:, :]
                v_bias = bias[qk_size:] if bias is not None else None
                self.v.weight.copy_(v_weight)
                if v_bias is not None:
                    self.v.bias.copy_(v_bias)

            self._weights_copied = True
    
    def forward(self, x):
        
        B, N, C = x.shape

        # Hooks on v will handle the conditional execution
        v, primitive = self.v(x)

        if primitive == "skip":
            return None, None, v, primitive
        
        v = v.reshape(B, N, self.num_heads, self.head_dim).permute(0, 2, 1, 3)

        qk = self.qk(x).reshape(B, N, 2, self.num_heads, self.head_dim).permute(2, 0, 3, 1, 4)
        q, k = qk.unbind(0)

        return q, k, v, primitive


class ArtiHippoValueAttention(nn.Module):
    fast_attn: Final[bool]

    def __init__(
            self,
            dim,
            num_heads=8,
            qkv_bias=False,
            qk_norm=False,
            attn_drop=0.,
            proj_drop=0.,
            norm_layer=nn.LayerNorm,
    ):
        super().__init__()
        assert dim % num_heads == 0, 'dim should be divisible by num_heads'
        self.num_heads = num_heads
        self.head_dim = dim // num_heads
        self.scale = self.head_dim ** -0.5
        self.fast_attn = hasattr(torch.nn.functional, 'scaled_dot_product_attention')  # FIXME

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.separable_qkv = SeparableValueLinear(dim, self.num_heads, self.head_dim, bias=qkv_bias)
        self.q_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.k_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        self.separable_qkv.copy_weights(self.qkv.weight, self.qkv.bias)
        # qkv = self.separable_qkv(x, self.qkv.weight, self.qkv.bias).reshape(B, N, 3, self.num_heads, self.head_dim).permute(2, 0, 3, 1, 4)
        # q, k, v = qkv.unbind(0)
        q, k, v, primitive = self.separable_qkv(x)

        if primitive == "skip":
            return v

        q, k = self.q_norm(q), self.k_norm(k)

        if self.fast_attn:
            x = F.scaled_dot_product_attention(
                q, k, v,
                dropout_p=self.attn_drop.p,
            )
        else:
            q = q * self.scale
            attn = q @ k.transpose(-2, -1)
            attn = attn.softmax(dim=-1)
            attn = self.attn_drop(attn)
            x = attn @ v

        x = x.transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x
    

class ArtiHippoValueBlock(nn.Module):

    def __init__(
            self,
            dim,
            num_heads,
            mlp_ratio=4.,
            qkv_bias=False,
            qk_norm=False,
            drop=0.,
            attn_drop=0.,
            init_values=None,
            drop_path=0.,
            act_layer=nn.GELU,
            norm_layer=nn.LayerNorm
    ):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = ArtiHippoValueAttention(
            dim,
            num_heads=num_heads,
            qkv_bias=qkv_bias,
            qk_norm=qk_norm,
            attn_drop=attn_drop,
            proj_drop=drop,
            norm_layer=norm_layer,
        )
        self.ls1 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path1 = DropPath(drop_path) if drop_path > 0. else nn.Identity()

        self.norm2 = norm_layer(dim)
        self.mlp = Mlp(
            in_features=dim,
            hidden_features=int(dim * mlp_ratio),
            act_layer=act_layer,
            drop=drop,
        )
        self.ls2 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path2 = DropPath(drop_path) if drop_path > 0. else nn.Identity()

    def forward(self, x):
        x = x + self.drop_path1(self.ls1(self.attn(self.norm1(x))))
        x = x + self.drop_path2(self.ls2(self.mlp(self.norm2(x))))
        return x
