from copy import deepcopy

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.jit import Final

from timm.layers import Mlp, DropPath
from timm.models._registry import get_arch_name
from timm.models.vision_transformer import LayerScale, _create_vision_transformer
from timm.models.vision_transformer import default_cfgs as timm_vit_default_cfgs
from timm.models._registry import generate_default_cfgs, register_model
from timm.data import IMAGENET_INCEPTION_MEAN, IMAGENET_INCEPTION_STD, OPENAI_CLIP_MEAN, OPENAI_CLIP_STD


class ZeroInitAttention(nn.Module):
    fast_attn: Final[bool]

    def __init__(
            self,
            dim,
            num_heads=8,
            qkv_bias=False,
            qk_norm=False,
            attn_drop=0.,
            proj_drop=0.,
            norm_layer=nn.LayerNorm,
    ):
        super().__init__()
        assert dim % num_heads == 0, 'dim should be divisible by num_heads'
        self.num_heads = num_heads
        self.head_dim = dim // num_heads
        self.scale = self.head_dim ** -0.5
        self.fast_attn = hasattr(torch.nn.functional, 'scaled_dot_product_attention')  # FIXME

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.q_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.k_norm = norm_layer(self.head_dim) if qk_norm else nn.Identity()
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

        # Prompts
        self.num_prompts = 10
        self.prompt = nn.Parameter(torch.zeros(1, self.num_prompts, dim).uniform_(-0.01, 0.01), requires_grad=True)
        self.g = nn.Parameter(torch.zeros(1, self.num_heads, 1, 1), requires_grad=True)

    def forward(self, x):
        B, N, C = x.shape

        # Dirty hack
        x = torch.cat([self.prompt.expand(B, -1, -1), x], dim=1)

        qkv = self.qkv(x).reshape(B, N+self.num_prompts, 3, self.num_heads, self.head_dim).permute(2, 0, 3, 1, 4) # BxN'x3xHxd -> 3xBxHxN'xd
        q, k, v = qkv.unbind(0) # BxHxN'xd
        # Remove the prompts from the query
        q = q[:, :, self.num_prompts:, :] # BxHxNxd
        q, k = self.q_norm(q), self.k_norm(k)

        q = q * self.scale
        A = q @ k.transpose(-2, -1) # BxHxNxd @ BxHxdxN' -> BxHxNxN'
        org_attn = A[:, :, :, self.num_prompts:]
        org_attn = org_attn.softmax(dim=-1)

        prompt_attn = A[:, :, :, :self.num_prompts].softmax(dim=-1) * self.g.tanh()
        attn = torch.cat([prompt_attn, org_attn], dim=-1)

        attn = self.attn_drop(attn)
        x = attn @ v # BxHxNxN' @ BxHxN'xd -> BxHxNxd

        x = x.transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x
    

class ZeroInitAttnBlock(nn.Module):

    def __init__(
            self,
            dim,
            num_heads,
            mlp_ratio=4.,
            qkv_bias=False,
            qk_norm=False,
            drop=0.,
            attn_drop=0.,
            init_values=None,
            drop_path=0.,
            act_layer=nn.GELU,
            norm_layer=nn.LayerNorm
    ):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = ZeroInitAttention(
            dim,
            num_heads=num_heads,
            qkv_bias=qkv_bias,
            qk_norm=qk_norm,
            attn_drop=attn_drop,
            proj_drop=drop,
            norm_layer=norm_layer,
        )
        self.ls1 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path1 = DropPath(drop_path) if drop_path > 0. else nn.Identity()

        self.norm2 = norm_layer(dim)
        self.mlp = Mlp(
            in_features=dim,
            hidden_features=int(dim * mlp_ratio),
            act_layer=act_layer,
            drop=drop,
        )
        self.ls2 = LayerScale(dim, init_values=init_values) if init_values else nn.Identity()
        self.drop_path2 = DropPath(drop_path) if drop_path > 0. else nn.Identity()

    def forward(self, x):
        x = x + self.drop_path1(self.ls1(self.attn(self.norm1(x))))
        x = x + self.drop_path2(self.ls2(self.mlp(self.norm2(x))))
        return x
