#!/usr/bin/env python3
""" SupSup Training Script

This is intended to be a lean and easily modifiable ImageNet training script that reproduces ImageNet
training results with some of the latest networks and training techniques. It favours canonical PyTorch
and standard Python style over trying to be able to 'do it all.' That said, it offers quite a few speed
and training result improvements over the usual PyTorch example scripts. Repurpose as you see fit.

This script was started from an early version of the PyTorch ImageNet example
(https://github.com/pytorch/examples/tree/master/imagenet)

NVIDIA CUDA specific speedups adopted from NVIDIA Apex examples
(https://github.com/NVIDIA/apex/tree/master/examples/imagenet)



Further hacked by  
"""
import argparse
import logging
import os
import time
from pathlib import Path
from collections import OrderedDict
from contextlib import suppress
from datetime import datetime
from functools import partial

import torch
from torch import Tensor
import torch.nn as nn
import torchvision.utils
import yaml
from torch.nn.parallel import DistributedDataParallel as NativeDDP
# << 
from torch.utils.data import RandomSampler
import torch.distributed as dist
#  >>

import numpy as np
import random

from timm import utils
from timm.data import create_loader, resolve_data_config, Mixup, FastCollateMixup, AugMixDataset
from timm.layers import convert_splitbn_model, convert_sync_batchnorm, set_fast_norm
from timm.loss import JsdCrossEntropy, SoftTargetCrossEntropy, BinaryCrossEntropy, LabelSmoothingCrossEntropy
from timm.models import create_model, safe_model_name, resume_checkpoint, load_checkpoint, model_parameters
from timm.optim import create_optimizer_v2, optimizer_kwargs
from timm.scheduler import create_scheduler_v2, scheduler_kwargs
from timm.utils import ApexScaler, NativeScaler
from timm.utils.model import unwrap_model

from artihippo.models.wrapper_supsup import SupSupWrapper
from artihippo.utils.backbone_iterators import iter_supsup_vit
from artihippo.models.modules.attn_proj_supsup_block import MultitaskMaskLinear, SupSupBlock, _MultitaskMaskLinear, _SupSupBlock

from artihippo.data import create_dataset_v2
from artihippo.config_utils import Config
from artihippo.timm_custom.data import create_loader_v2
from artihippo.timm_custom.data.distributed_sampler import RepeatedDistributedSampler

try:
    from apex import amp
    from apex.parallel import DistributedDataParallel as ApexDDP
    from apex.parallel import convert_syncbn_model
    has_apex = True
except ImportError:
    has_apex = False

has_native_amp = False
try:
    if getattr(torch.cuda.amp, 'autocast') is not None:
        has_native_amp = True
except AttributeError:
    pass

try:
    import wandb
    has_wandb = True
except ImportError:
    has_wandb = False

try:
    from functorch.compile import memory_efficient_fusion
    has_functorch = True
except ImportError as e:
    has_functorch = False

has_compile = hasattr(torch, 'compile')


_logger = logging.getLogger('train')

# The first arg parser parses out only the --config argument, this argument is used to
# load a yaml file containing key-values that override the defaults for the main parser below
config_parser = parser = argparse.ArgumentParser(description='Training Config', add_help=False)
parser.add_argument('-c', '--config', default='', type=str, metavar='FILE',
                    help='YAML config file specifying default arguments')


parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')

# Dataset parameters
group = parser.add_argument_group('Dataset parameters')
group.add_argument('--runtime-config-file', default='', type=str, metavar='RUNTIME_CONFIG',
                   help='YAML config file specifying ImageNet arguments')
group.add_argument('--benchmark', default='vdd', type=str, metavar='BENCHMARK',
                   help='Benchmark to run. Should be one of "vdd", "5-datasets"')
group.add_argument('--root_dir', metavar='ROOT',
                   help='root directory to the repository')
group.add_argument("--backbone_weights_path", default=None, type=str, help="ImageNet checkpoint containing the backbone weights.")
group.add_argument("--backbone_checkpoint_dir", default=None, type=str, help="ImageNet checkpoint containing mean calculations.")

group.add_argument('--dataset', '-d', metavar='NAME', default='',
                   help='dataset type (default: ImageFolder/ImageTar if empty)')
group.add_argument('--train-split', metavar='NAME', default='train',
                   help='dataset train split (default: train)')
group.add_argument('--val-split', metavar='NAME', default='validation',
                   help='dataset validation split (default: validation)')
group.add_argument('--dataset-download', action='store_true', default=False,
                   help='Allow download of dataset for torch/ and tfds/ datasets that support it.')
group.add_argument('--class-map', default='', type=str, metavar='FILENAME',
                   help='path to class to idx mapping file (default: "")')

# Model parameters
group = parser.add_argument_group('Model parameters')
group.add_argument('--model', default='resnet50', type=str, metavar='MODEL',
                   help='Name of model to train (default: "resnet50")')
group.add_argument('--pretrained', action='store_true', default=False,
                   help='Start with pretrained version of specified network (if avail)')
group.add_argument('--initial-checkpoint', default='', type=str, metavar='PATH',
                   help='Initialize model from this checkpoint (default: none)')
group.add_argument('--resume', default='', type=str, metavar='PATH',
                   help='Resume full model and optimizer state from checkpoint (default: none)')
group.add_argument('--no-resume-opt', action='store_true', default=False,
                   help='prevent resume of optimizer state when resuming model')
group.add_argument('--num-classes', type=int, default=None, metavar='N',
                   help='number of label classes (Model default if None)')
group.add_argument('--gp', default=None, type=str, metavar='POOL',
                   help='Global pool type, one of (fast, avg, max, avgmax, avgmaxc). Model default if None.')
group.add_argument('--img-size', type=int, default=None, metavar='N',
                   help='Image size (default: None => model default)')
group.add_argument('--in-chans', type=int, default=None, metavar='N',
                   help='Image input channels (default: None => 3)')
group.add_argument('--input-size', default=None, nargs=3, type=int,
                   metavar='N N N',
                   help='Input all image dimensions (d h w, e.g. --input-size 3 224 224), uses model default if empty')
group.add_argument('--crop-pct', default=None, type=float,
                   metavar='N', help='Input image center crop percent (for validation only)')
group.add_argument('--mean', type=float, nargs='+', default=None, metavar='MEAN',
                   help='Override mean pixel value of dataset')
group.add_argument('--std', type=float, nargs='+', default=None, metavar='STD',
                   help='Override std deviation of dataset')
group.add_argument('--interpolation', default='', type=str, metavar='NAME',
                   help='Image resize interpolation type (overrides model)')
group.add_argument('-b', '--batch-size', type=int, default=128, metavar='N',
                   help='Input batch size for training (default: 128)')
group.add_argument('-vb', '--validation-batch-size', type=int, default=None, metavar='N',
                   help='Validation batch size override (default: None)')
group.add_argument('--channels-last', action='store_true', default=False,
                   help='Use channels_last memory layout')
group.add_argument('--fuser', default='', type=str,
                   help="Select jit fuser. One of ('', 'te', 'old', 'nvfuser')")
group.add_argument('--grad-checkpointing', action='store_true', default=False,
                   help='Enable gradient checkpointing through model blocks/stages')
group.add_argument('--fast-norm', default=False, action='store_true',
                   help='enable experimental fast-norm')
group.add_argument('--model-kwargs', nargs='*', default={}, action=utils.ParseKwargs)

scripting_group = group.add_mutually_exclusive_group()
scripting_group.add_argument('--torchscript', dest='torchscript', action='store_true',
                             help='torch.jit.script the full model')
scripting_group.add_argument('--torchcompile', nargs='?', type=str, default=None, const='inductor',
                             help="Enable compilation w/ specified backend (default: inductor).")
scripting_group.add_argument('--aot-autograd', default=False, action='store_true',
                             help="Enable AOT Autograd support.")

# Optimizer parameters
group = parser.add_argument_group('Optimizer parameters')
group.add_argument('--opt', default='sgd', type=str, metavar='OPTIMIZER',
                   help='Optimizer (default: "sgd")')
group.add_argument('--opt-eps', default=None, type=float, metavar='EPSILON',
                   help='Optimizer Epsilon (default: None, use opt default)')
group.add_argument('--opt-betas', default=None, type=float, nargs='+', metavar='BETA',
                   help='Optimizer Betas (default: None, use opt default)')
group.add_argument('--momentum', type=float, default=0.9, metavar='M',
                   help='Optimizer momentum (default: 0.9)')
group.add_argument('--weight-decay', type=float, default=2e-5,
                   help='weight decay (default: 2e-5)')
group.add_argument('--clip-grad', type=float, default=None, metavar='NORM',
                   help='Clip gradient norm (default: None, no clipping)')
group.add_argument('--clip-mode', type=str, default='norm',
                   help='Gradient clipping mode. One of ("norm", "value", "agc")')
group.add_argument('--layer-decay', type=float, default=None,
                   help='layer-wise learning rate decay (default: None)')
group.add_argument('--opt-kwargs', nargs='*', default={}, action=utils.ParseKwargs)

# Learning rate schedule parameters
group = parser.add_argument_group('Learning rate schedule parameters')
group.add_argument('--sched', type=str, default='cosine', metavar='SCHEDULER',
                   help='LR scheduler (default: "step"')
group.add_argument('--sched-on-updates', action='store_true', default=False,
                   help='Apply LR scheduler step on update instead of epoch end.')
group.add_argument('--lr', type=float, default=None, metavar='LR',
                   help='learning rate, overrides lr-base if set (default: None)')
group.add_argument('--prompt_lr', type=float, default=0.001, metavar='PROMPT_LR',
                   help='learning rate for learning prompts (default: 0.001)')
group.add_argument('--lr-base', type=float, default=0.1, metavar='LR',
                   help='base learning rate: lr = lr_base * global_batch_size / base_size')
group.add_argument('--lr-base-size', type=int, default=256, metavar='DIV',
                   help='base learning rate batch size (divisor, default: 256).')
group.add_argument('--lr-base-scale', type=str, default='', metavar='SCALE',
                   help='base learning rate vs batch_size scaling ("linear", "sqrt", based on opt if empty)')
group.add_argument('--lr-noise', type=float, nargs='+', default=None, metavar='pct, pct',
                   help='learning rate noise on/off epoch percentages')
group.add_argument('--lr-noise-pct', type=float, default=0.67, metavar='PERCENT',
                   help='learning rate noise limit percent (default: 0.67)')
group.add_argument('--lr-noise-std', type=float, default=1.0, metavar='STDDEV',
                   help='learning rate noise std-dev (default: 1.0)')
group.add_argument('--lr-cycle-mul', type=float, default=1.0, metavar='MULT',
                   help='learning rate cycle len multiplier (default: 1.0)')
group.add_argument('--lr-cycle-decay', type=float, default=0.5, metavar='MULT',
                   help='amount to decay each learning rate cycle (default: 0.5)')
group.add_argument('--lr-cycle-limit', type=int, default=1, metavar='N',
                   help='learning rate cycle limit, cycles enabled if > 1')
group.add_argument('--lr-k-decay', type=float, default=1.0,
                   help='learning rate k-decay for cosine/poly (default: 1.0)')
group.add_argument('--warmup-lr', type=float, default=1e-5, metavar='LR',
                   help='warmup learning rate (default: 1e-5)')
group.add_argument('--min-lr', type=float, default=0, metavar='LR',
                   help='lower lr bound for cyclic schedulers that hit 0 (default: 0)')
group.add_argument('--epochs', type=int, default=300, metavar='N',
                   help='number of epochs to train (default: 300)')
group.add_argument('--prompt_epochs', type=int, default=20, metavar='N',
                   help='number of epochs to train (default: 300)')
group.add_argument('--epoch-repeats', type=float, default=0., metavar='N',
                   help='epoch repeat multiplier (number of times to repeat dataset epoch per train epoch).')
group.add_argument('--start-epoch', default=None, type=int, metavar='N',
                   help='manual epoch number (useful on restarts)')
group.add_argument('--decay-milestones', default=[90, 180, 270], type=int, nargs='+', metavar="MILESTONES",
                   help='list of decay epoch indices for multistep lr. must be increasing')
group.add_argument('--decay-epochs', type=float, default=90, metavar='N',
                   help='epoch interval to decay LR')
group.add_argument('--warmup-epochs', type=int, default=5, metavar='N',
                   help='epochs to warmup LR, if scheduler supports')
group.add_argument('--warmup-prefix', action='store_true', default=False,
                   help='Exclude warmup period from decay schedule.'),
group.add_argument('--cooldown-epochs', type=int, default=0, metavar='N',
                   help='epochs to cooldown LR at min_lr, after cyclic schedule ends')
group.add_argument('--patience-epochs', type=int, default=10, metavar='N',
                   help='patience epochs for Plateau LR scheduler (default: 10)')
group.add_argument('--decay-rate', '--dr', type=float, default=0.1, metavar='RATE',
                   help='LR decay rate (default: 0.1)')

# Augmentation & regularization parameters
group = parser.add_argument_group('Augmentation and regularization parameters')
group.add_argument('--no-aug', action='store_true', default=False,
                   help='Disable all training augmentation, override other train aug args')
group.add_argument('--scale', type=float, nargs='+', default=[0.08, 1.0], metavar='PCT',
                   help='Random resize scale (default: 0.08 1.0)')
group.add_argument('--ratio', type=float, nargs='+', default=[3. / 4., 4. / 3.], metavar='RATIO',
                   help='Random resize aspect ratio (default: 0.75 1.33)')
group.add_argument('--hflip', type=float, default=0.5,
                   help='Horizontal flip training aug probability')
group.add_argument('--vflip', type=float, default=0.,
                   help='Vertical flip training aug probability')
group.add_argument('--color-jitter', type=float, default=0.4, metavar='PCT',
                   help='Color jitter factor (default: 0.4)')
group.add_argument('--aa', type=str, default=None, metavar='NAME',
                   help='Use AutoAugment policy. "v0" or "original". (default: None)'),
group.add_argument('--aug-repeats', type=float, default=0,
                   help='Number of augmentation repetitions (distributed training only) (default: 0)')
group.add_argument('--aug-splits', type=int, default=0,
                   help='Number of augmentation splits (default: 0, valid: 0 or >=2)')
group.add_argument('--jsd-loss', action='store_true', default=False,
                   help='Enable Jensen-Shannon Divergence + CE loss. Use with `--aug-splits`.')
group.add_argument('--bce-loss', action='store_true', default=False,
                   help='Enable BCE loss w/ Mixup/CutMix use.')
group.add_argument('--bce-target-thresh', type=float, default=None,
                   help='Threshold for binarizing softened BCE targets (default: None, disabled)')
group.add_argument('--reprob', type=float, default=0., metavar='PCT',
                   help='Random erase prob (default: 0.)')
group.add_argument('--remode', type=str, default='pixel',
                   help='Random erase mode (default: "pixel")')
group.add_argument('--recount', type=int, default=1,
                   help='Random erase count (default: 1)')
group.add_argument('--resplit', action='store_true', default=False,
                   help='Do not random erase first (clean) augmentation split')
group.add_argument('--mixup', type=float, default=0.0,
                   help='mixup alpha, mixup enabled if > 0. (default: 0.)')
group.add_argument('--cutmix', type=float, default=0.0,
                   help='cutmix alpha, cutmix enabled if > 0. (default: 0.)')
group.add_argument('--cutmix-minmax', type=float, nargs='+', default=None,
                   help='cutmix min/max ratio, overrides alpha and enables cutmix if set (default: None)')
group.add_argument('--mixup-prob', type=float, default=1.0,
                   help='Probability of performing mixup or cutmix when either/both is enabled')
group.add_argument('--mixup-switch-prob', type=float, default=0.5,
                   help='Probability of switching to cutmix when both mixup and cutmix enabled')
group.add_argument('--mixup-mode', type=str, default='batch',
                   help='How to apply mixup/cutmix params. Per "batch", "pair", or "elem"')
group.add_argument('--mixup-off-epoch', default=0, type=int, metavar='N',
                   help='Turn off mixup after this epoch, disabled if 0 (default: 0)')
group.add_argument('--smoothing', type=float, default=0.1,
                   help='Label smoothing (default: 0.1)')
group.add_argument('--train-interpolation', type=str, default='random',
                   help='Training interpolation (random, bilinear, bicubic default: "random")')
group.add_argument('--drop', type=float, default=0.0, metavar='PCT',
                   help='Dropout rate (default: 0.)')
group.add_argument('--drop-connect', type=float, default=None, metavar='PCT',
                   help='Drop connect rate, DEPRECATED, use drop-path (default: None)')
group.add_argument('--drop-path', type=float, default=None, metavar='PCT',
                   help='Drop path rate (default: None)')
group.add_argument('--drop-block', type=float, default=None, metavar='PCT',
                   help='Drop block rate (default: None)')

# Batch norm parameters (only works with gen_efficientnet based models currently)
group = parser.add_argument_group('Batch norm parameters', 'Only works with gen_efficientnet based models currently.')
group.add_argument('--bn-momentum', type=float, default=None,
                   help='BatchNorm momentum override (if not None)')
group.add_argument('--bn-eps', type=float, default=None,
                   help='BatchNorm epsilon override (if not None)')
group.add_argument('--sync-bn', action='store_true',
                   help='Enable NVIDIA Apex or Torch synchronized BatchNorm.')
group.add_argument('--dist-bn', type=str, default='reduce',
                   help='Distribute BatchNorm stats between nodes after each epoch ("broadcast", "reduce", or "")')
group.add_argument('--split-bn', action='store_true',
                   help='Enable separate BN layers per augmentation split.')

# Model Exponential Moving Average
group = parser.add_argument_group('Model exponential moving average parameters')
group.add_argument('--model-ema', action='store_true', default=False,
                   help='Enable tracking moving average of model weights')
group.add_argument('--model-ema-force-cpu', action='store_true', default=False,
                   help='Force ema to be tracked on CPU, rank=0 node only. Disables EMA validation.')
group.add_argument('--model-ema-decay', type=float, default=0.9998,
                   help='decay factor for model weights moving average (default: 0.9998)')

# Misc
group = parser.add_argument_group('Miscellaneous parameters')
group.add_argument('--seed', type=int, default=42, metavar='S',
                   help='random seed (default: 42)')
group.add_argument('--worker-seeding', type=str, default='all',
                   help='worker seed mode (default: all)')
group.add_argument('--log-interval', type=int, default=50, metavar='N',
                   help='how many batches to wait before logging training status')
group.add_argument('--recovery-interval', type=int, default=0, metavar='N',
                   help='how many batches to wait before writing recovery checkpoint')
group.add_argument('--checkpoint-hist', type=int, default=1, metavar='N',
                   help='number of checkpoints to keep (default: 10)')
group.add_argument('-j', '--workers', type=int, default=4, metavar='N',
                   help='how many training processes to use (default: 4)')
group.add_argument('--save-images', action='store_true', default=False,
                   help='save images of input bathes every log interval for debugging')
group.add_argument('--amp', action='store_true', default=False,
                   help='use NVIDIA Apex AMP or Native AMP for mixed precision training')
group.add_argument('--amp-dtype', default='float16', type=str,
                   help='lower precision AMP dtype (default: float16)')
group.add_argument('--amp-impl', default='native', type=str,
                   help='AMP impl to use, "native" or "apex" (default: native)')
group.add_argument('--no-ddp-bb', action='store_true', default=False,
                   help='Force broadcast buffers for native DDP to off.')
group.add_argument('--pin-mem', action='store_true', default=False,
                   help='Pin CPU memory in DataLoader for more efficient (sometimes) transfer to GPU.')
group.add_argument('--no-prefetcher', action='store_true', default=False,
                   help='disable fast prefetcher')
group.add_argument('--output', default='', type=str, metavar='PATH',
                   help='path to output folder (default: none, current dir)')
group.add_argument('--experiment', default='', type=str, metavar='NAME',
                   help='name of train experiment, name of sub-folder for output')
group.add_argument('--eval-metric', default='top1', type=str, metavar='EVAL_METRIC',
                   help='Best metric (default: "top1"')
group.add_argument('--tta', type=int, default=0, metavar='N',
                   help='Test/inference time augmentation (oversampling) factor. 0=None (default: 0)')
group.add_argument("--local_rank", default=0, type=int)
group.add_argument('--use-multi-epochs-loader', action='store_true', default=False,
                   help='use the multi-epochs-loader to save time at the beginning of every epoch')
group.add_argument('--log-wandb', action='store_true', default=False,
                   help='log training and validation metrics to wandb')

# << 
group = parser.add_argument_group('Runtime settings')
group.add_argument('--wandb-project', default=None, type=str,
                   help='WandB project name')
group.add_argument('--wandb-username', default=None, type=str,
                   help='WandB username')
group.add_argument('--exp-name', default=None, type=str,
                   help='WandB run group.')
group.add_argument('--auto-scale-const',
                   type=float,
                   default=512.,
                   help='Scaling constant in Auto-scale lr: lr * total_bsz / auto_scale_const')
group.add_argument('--unit-test',
                   action='store_true',
                   default=False,
                   help='Unit Test flag for skipping full runs.')
group.add_argument('--dev-percent',
                   type=float,
                   default=0.,
                   help='Percent data used for development set')
group.add_argument('--unit-test-epochs',
                   type=int,
                   default=10,
                   help='Epochs for unit testing')
#  >>

# << 
group = parser.add_argument_group(' modified settings')
group.add_argument('--auto-scale-lr',
                   action='store_true',
                   default=False,
                   help='Auto-scale lr: lr * total_bsz / auto_scale_const')
group.add_argument('--use-deterministic',
                   action='store_true',
                   default=False,
                   help='set cuda to be deterministic')
#  >>

# << 
lll_group = parser.add_argument_group('Arguments for LLL settings')
lll_group.add_argument("--task-idx", 
                       default=None, 
                       type=int, 
                       help="Task Index for the task to train.")
lll_group.add_argument("--load-single-task", 
                       action="store_true",
                       default=False, 
                       help="Create dataset only for the current task.")
lll_group.add_argument("--offset_task_labels", 
                       action="store_true",
                       default=False, 
                       help="Add offsets to the class lables in the dataloaders. Used for single-head setting")
lll_group.add_argument("--artihippo_component", 
                       default=None, 
                       type=str, 
                       help="Component of the backbone used as ArtiHippo.")
lll_group.add_argument("--sparsity", 
                       default=0.5, 
                       type=float, 
                       help="Sparsity for SupSup.")


data_group = parser.add_argument_group('Arguments for Data settings')
data_group.add_argument('--data_root', metavar='DIR',
                    help='Root path to the dataset')
data_group.add_argument('--imagenet_root', metavar='DIR',
                    help='Root path to the ImageNet dataset')
data_group.add_argument("--cache_data", 
                       action="store_true",
                       default=False, 
                       help="Cache the dataset metadata.")
data_group.add_argument('--cache_root', default=None, 
                        type=str,
                        help='Root path to the cache')
data_group.add_argument("--repeated-sampling", 
                       action="store_true",
                       default=False, 
                       help="Draw samples with replacement.")
data_group.add_argument('--min-batch-size', default=64, 
                        type=int,
                        help='Minimum batch size for the dataloader.')
#  >>

def _parse_args():
    # Do we have a config file to parse?
    args_config, remaining = config_parser.parse_known_args()
    if args_config.config:
        with open(args_config.config, 'r') as f:
            cfg = yaml.safe_load(f)
            parser.set_defaults(**cfg)

    # The main arg parser parses the rest of the args, the usual
    # defaults will have been overridden if config file specified.
    args = parser.parse_args(remaining)

    # Cache the args as a text string to save them in the output dir later
    args_text = yaml.safe_dump(args.__dict__, default_flow_style=False)
    return args, args_text


def get_model(backbone, benchmark_config, runtime, args):

    def head_factory(backbone):
        feature_dim = backbone.num_features
        
        heads = nn.ModuleList([
            nn.Linear(feature_dim, benchmark_config.task_config[task]['num_classes']) for task in runtime.task_order
        ])

        # Copy the weights from the backbone to the head
        heads[0].load_state_dict(backbone.head.state_dict())

        return heads

    iter_func = iter_supsup_vit

    model = SupSupWrapper(backbone, iter_func, head_factory)

    return model


def main():

    dash_line = '-' * 60 + '\n'
    args, args_text = _parse_args()

    if torch.cuda.is_available():
        torch.backends.cuda.matmul.allow_tf32 = True
        torch.backends.cudnn.benchmark = True

    assert args.task_idx > 0, "ImageNet is considerd the 0th task and handled separately."

    _config = Config(
        args.benchmark, args.runtime_config_file, args.root_dir, args.exp_name, 
        wandb_username=args.wandb_username, wandb_project=args.wandb_project, 
        checkpoint_path=None)
    runtime = _config.runtime
    benchmark_config = _config.benchmark_config

    # Housekeeping
    runtime.config_dir.mkdir(parents=True, exist_ok=True)
    runtime.temp_dir.mkdir(parents=True, exist_ok=True)
    runtime.log_dir.mkdir(parents=True, exist_ok=True)
    runtime.supernet_config_dir.mkdir(parents=True, exist_ok=True)
    runtime.stats_dir.mkdir(parents=True, exist_ok=True)

    # Override the parameters
    args.img_size = runtime.img_size
    args.num_classes = benchmark_config.task_config["imagenet12"]['num_classes']
    args.running_mean_epochs = runtime.running_mean_epochs
    # args.drop_path = runtime.drop_path

    # Set up logging
    _time = datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
    log_dir = Path(runtime.log_dir, _time, "finetune")
    log_dir.mkdir(parents=True, exist_ok=True)
    logfile = str(Path(log_dir, f"supsup.log"))

    args.output = log_dir

    utils.setup_default_logging(log_path=logfile)
    
    output_dir = None

    args.prefetcher = not args.no_prefetcher
    device = utils.init_distributed_device(args)
    if args.distributed:
        _logger.info(
            'Training in distributed mode with multiple processes, 1 device per process.'
            f'Process {args.rank}, total {args.world_size}, device {args.device}.')
    else:
        _logger.info(f'Training with a single process on 1 device ({args.device}).')
    assert args.rank >= 0

    if utils.is_primary(args):
        if args.experiment:
            exp_name = args.experiment
        else:
            exp_name = '-'.join([
                datetime.now().strftime("%Y%m%d-%H%M%S"),
                safe_model_name(args.model)
            ])
        output_dir = utils.get_outdir(args.output if args.output else f'{args.root_dir}/artifacts/{args.benchmark}', exp_name)
        Path(output_dir).mkdir(parents=True, exist_ok=True)

    if utils.is_primary(args) and args.log_wandb:

        if has_wandb:
            assert args.wandb_project is not None
            assert args.wandb_username is not None
            wandb_run_id = wandb.util.generate_id()
            run = wandb.init(project=runtime.wandb["project"], entity=runtime.wandb["entity"], group=args.exp_name, name=f"{args.task_name}_finetune", config=args, id=wandb_run_id)
            run.config.update(_config.config_dict)
        else: 
            _logger.warning("You've requested to log metrics to wandb but package not found. "
                            "Metrics not being logged to wandb, try `pip install wandb`")

    # resolve AMP arguments based on PyTorch / Apex availability
    use_amp = None
    amp_dtype = torch.float16
    if args.amp:
        if args.amp_impl == 'apex':
            assert has_apex, 'AMP impl specified as APEX but APEX is not installed.'
            use_amp = 'apex'
            assert args.amp_dtype == 'float16'
        else:
            assert has_native_amp, 'Please update PyTorch to a version with native AMP (or use APEX).'
            use_amp = 'native'
            assert args.amp_dtype in ('float16', 'bfloat16')
        if args.amp_dtype == 'bfloat16':
            amp_dtype = torch.bfloat16

    # << 
    if args.use_deterministic:
        torch.manual_seed(0)
        torch.cuda.manual_seed_all(0)
        np.random.seed(0)
        random.seed(0)
        torch.backends.cudnn.deterministic = True
    else:
        #  >>
        utils.random_seed(args.seed, args.rank)

    if args.fuser:
        utils.set_jit_fuser(args.fuser)
    if args.fast_norm:
        set_fast_norm()

    in_chans = 3
    if args.in_chans is not None:
        in_chans = args.in_chans
    elif args.input_size is not None:
        in_chans = args.input_size[0]

    # Create the block function
    Linear = _MultitaskMaskLinear(MultitaskMaskLinear, args.sparsity, len(runtime.task_order))
    block_fn = _SupSupBlock(SupSupBlock, masked_linear_layer=Linear)

    backbone = create_model(
        args.model,
        pretrained=args.pretrained,
        in_chans=in_chans,
        num_classes=args.num_classes,
        drop_rate=args.drop,
        drop_path_rate=args.drop_path,
        drop_block_rate=args.drop_block,
        global_pool=args.gp,
        bn_momentum=args.bn_momentum,
        bn_eps=args.bn_eps,
        scriptable=args.torchscript,
        checkpoint_path=args.initial_checkpoint,
        pretrained_strict=False,
        block_fn=block_fn,
        **runtime.model_kwargs
    )

    backbone_state_dict_path = args.backbone_weights_path
    # Load the weights of the base checkpoint
    if utils.is_primary(args):
        _logger.info(f"Loading backbone checkpoint from {backbone_state_dict_path}.")
    state_dict = torch.load(backbone_state_dict_path, map_location="cpu")["state_dict"]
    backbone.load_state_dict(state_dict, strict=False)

    backbone.requires_grad_(False)

    _logger.info(f"Backbone trainable params: {sum([p.numel() for p in backbone.parameters() if p.requires_grad])}")

    model = get_model(backbone, benchmark_config, runtime, args)

    if utils.is_primary(args):
        _logger.info(f"{model}")

    if args.num_classes is None:
        assert hasattr(model, 'num_classes'), 'Model must have `num_classes` attr if not set on cmd line/config.'
        args.num_classes = model.num_classes  # FIXME handle model default vs config num_classes more elegantly

    if args.grad_checkpointing:
        model.set_grad_checkpointing(enable=True)

    num_classes = sum([benchmark_config.task_config[task]['num_classes'] for task in runtime.task_order])

    # Actual parameter counts
    args.mparams = sum([m.numel() for m in model.parameters()])
    # Number of parameters without the head
    num_params_head = model.backbone.num_features * num_classes + num_classes
    params_without_head = args.mparams - num_params_head
    mparams_trainable = sum([m.numel() for m in model.parameters() if m.requires_grad])
    mparams_trainable_wo_head = mparams_trainable - num_params_head

    if utils.is_primary(args):
        if args.unit_test:
            _logger.info("Running in Unit Test mode. Full run will be skipped.")
        _logger.info(f"Using img_size={args.img_size}\n{dash_line}")
        
        _logger.info(
            f'Model {safe_model_name(args.model)} created, param count:{args.mparams}. param count without head: {params_without_head}')
        _logger.info(f'{args.mparams} Params. Without Head: {params_without_head} Params\n{dash_line}')
        _logger.info(f'{mparams_trainable} Trainable Params. Without Head: {mparams_trainable_wo_head} Params\n{dash_line}')

        _logger.info(
            f'{dash_line} Auto-scaled: {"Y" if args.auto_scale_lr else "N"}, lr={args.lr}, warmup_lr={args.warmup_lr}, min_lr={args.min_lr}, scaling_const={args.auto_scale_const}\n{dash_line}'
        )
        if args.pretrained:
            _logger.info("Initializing from pretrained weights.")

    data_config = resolve_data_config(vars(args), model=model.backbone, verbose=utils.is_primary(args))

    # setup augmentation batch splits for contrastive loss or split bn
    num_aug_splits = 0
    if args.aug_splits > 0:
        assert args.aug_splits > 1, 'A split of 1 makes no sense'
        num_aug_splits = args.aug_splits

    # enable split bn (separate bn stats per batch-portion)
    if args.split_bn:
        assert num_aug_splits > 1 or args.resplit
        model = convert_splitbn_model(model, max(num_aug_splits, 2))

    # move model to GPU, enable channels last layout if set
    model.to(device=device)
    if args.channels_last:
        model.to(memory_format=torch.channels_last)

    # setup synchronized BatchNorm for distributed training
    if args.distributed and args.sync_bn:
        args.dist_bn = ''  # disable dist_bn when sync BN active
        assert not args.split_bn
        if has_apex and use_amp == 'apex':
            # Apex SyncBN used with Apex AMP
            # WARNING this won't currently work with models using BatchNormAct2d
            model = convert_syncbn_model(model)
        else:
            model = convert_sync_batchnorm(model)
        if utils.is_primary(args):
            _logger.info(
                'Converted model to use Synchronized BatchNorm. WARNING: You may have issues if using '
                'zero initialized BN layers (enabled by default for ResNets) while sync-bn enabled.')

    if args.torchscript:
        assert not use_amp == 'apex', 'Cannot use APEX AMP with torchscripted model'
        assert not args.sync_bn, 'Cannot use SyncBatchNorm with torchscripted model'
        model = torch.jit.script(model)
    elif args.torchcompile:
        # FIXME dynamo might need move below DDP wrapping? TBD
        assert has_compile, 'A version of torch w/ torch.compile() is required for --compile, possibly a nightly.'
        torch._dynamo.reset()
        model = torch.compile(model, backend=args.torchcompile)
    elif args.aot_autograd:
        assert has_functorch, "functorch is needed for --aot-autograd"
        model = memory_efficient_fusion(model)

    if not args.lr:
        global_batch_size = args.batch_size * args.world_size
        batch_ratio = global_batch_size / args.lr_base_size
        if not args.lr_base_scale:
            on = args.opt.lower()
            args.lr_base_scale = 'sqrt' if any([o in on for o in ('ada', 'lamb')]) else 'linear'
        if args.lr_base_scale == 'sqrt':
            batch_ratio = batch_ratio ** 0.5
        args.lr = args.lr_base * batch_ratio
        if utils.is_primary(args):
            _logger.info(
                f'Learning rate ({args.lr}) calculated from base learning rate ({args.lr_base}) '
                f'and global batch size ({global_batch_size}) with {args.lr_base_scale} scaling.')

    # setup automatic mixed-precision (AMP) loss scaling and op casting
    amp_autocast = suppress  # do nothing
    loss_scaler = None
    if use_amp == 'apex':
        assert device.type == 'cuda'
        model, optimizer = amp.initialize(model, optimizer, opt_level='O1')
        loss_scaler = ApexScaler()
        if utils.is_primary(args):
            _logger.info('Using NVIDIA APEX AMP. Training in mixed precision.')
    elif use_amp == 'native':
        try:
            amp_autocast = partial(torch.autocast, device_type=device.type, dtype=amp_dtype)
        except (AttributeError, TypeError):
            # fallback to CUDA only AMP for PyTorch < 1.10
            assert device.type == 'cuda'
            amp_autocast = torch.cuda.amp.autocast
        if device.type == 'cuda' and amp_dtype == torch.float16:
            # loss scaler only used for float16 (half) dtype, bfloat16 does not need it
            loss_scaler = NativeScaler()
        if utils.is_primary(args):
            _logger.info('Using native Torch AMP. Training in mixed precision.')
    else:
        if utils.is_primary(args):
            _logger.info('AMP not enabled. Training in float32.')

    # optionally resume from a checkpoint
    resume_epoch = None
    if args.resume:
        resume_epoch = resume_checkpoint(
            model,
            args.resume,
            optimizer=None if args.no_resume_opt else optimizer,
            loss_scaler=None if args.no_resume_opt else loss_scaler,
            log_info=utils.is_primary(args),
        )

    # setup exponential moving average of model weights, SWA could be used here too
    model_ema = None
    if args.model_ema:
        # Important to create EMA model after cuda(), DP wrapper, and AMP but before DDP wrapper
        model_ema = utils.ModelEmaV2(
            model, decay=args.model_ema_decay, device='cpu' if args.model_ema_force_cpu else None)
        if args.resume:
            load_checkpoint(model_ema.module, args.resume, use_ema=True)

    # setup distributed training
    if args.distributed:
        if has_apex and use_amp == 'apex':
            # Apex DDP preferred unless native amp is activated
            if utils.is_primary(args):
                _logger.info("Using NVIDIA APEX DistributedDataParallel.")
            model = ApexDDP(model, delay_allreduce=True)
        else:
            if utils.is_primary(args):
                _logger.info("Using native Torch DistributedDataParallel.")
            model = NativeDDP(model, device_ids=[device], broadcast_buffers=not args.no_ddp_bb)
        # NOTE: EMA model does not need to be wrapped by DDP

    eval_set = "val" if args.dev_percent > 0. else "test"

    # create the train and eval datasets
    datasets = create_dataset_v2(
        args.benchmark, 
        data_root=args.data_root,
        task_names=runtime.task_order,
        imagenet_root=args.imagenet_root, 
        task_idx=args.task_idx,
        load_single_task=args.load_single_task,
        cache=args.cache_data, 
        cache_root=args.cache_root,
        dev_percent=args.dev_percent, 
        offset_task_labels=args.offset_task_labels)

    # setup mixup / cutmix
    collate_fn = None
    mixup_fn = None
    mixup_active = args.mixup > 0 or args.cutmix > 0. or args.cutmix_minmax is not None
    if mixup_active:
        mixup_args = dict(
            mixup_alpha=args.mixup,
            cutmix_alpha=args.cutmix,
            cutmix_minmax=args.cutmix_minmax,
            prob=args.mixup_prob,
            switch_prob=args.mixup_switch_prob,
            mode=args.mixup_mode,
            label_smoothing=args.smoothing,
            num_classes=args.num_classes
        )
        if args.prefetcher:
            assert not num_aug_splits  # collate conflict (need to support deinterleaving in collate mixup)
            collate_fn = FastCollateMixup(**mixup_args)
        else:
            mixup_fn = Mixup(**mixup_args)

    # wrap dataset in AugMix helper
    if num_aug_splits > 1:
        dataset_train = AugMixDataset(dataset_train, num_splits=num_aug_splits)

    if args.distributed:
        args.batch_size = args.batch_size // args.world_size
        if utils.is_primary(args):
            _logger.info(f"Reducing batch size to {args.batch_size} for distributed training")

    loaders_train = []
    loaders_eval = []
    for task_idx, task_name in enumerate(runtime.task_order):

        dataset_train = datasets["train"][task_idx]
        train_transforms = benchmark_config.train_transforms[task_name]
        if task_idx == 0:
            train_transforms = benchmark_config.eval_transforms[task_name]
        
        # Set up custom sampling
        custom_sampler = None
        if args.repeated_sampling:
            total_num_samples = len(dataset_train)

            # If the total number of samples that will be drawn is less than the minimum number of batches, use repeated sampling
            if total_num_samples // (args.batch_size * args.world_size) < runtime.min_finetune_batches:
                if utils.is_primary(args):
                    _logger.info("Using repeated sampler")
                if args.distributed:
                    custom_sampler = RepeatedDistributedSampler(dataset_train, batch_size=args.batch_size, min_batches=runtime.min_finetune_batches)
                else:
                    num_samples = args.batch_size * runtime.min_finetune_batches
                    custom_sampler = RandomSampler(dataset_train, replacement=True, num_samples=num_samples)

        # create data loaders w/ augmentation pipeiine
        train_interpolation = data_config['interpolation']
        if args.no_aug or not train_interpolation:
            train_interpolation = data_config['interpolation']
        loader_train = create_loader_v2(
            task_name,
            dataset_train,
            input_size=data_config['input_size'],
            transform_list=train_transforms,
            batch_size=args.batch_size,
            is_training=True and task_idx > 0,
            use_prefetcher=args.prefetcher,
            no_aug=args.no_aug,
            re_prob=args.reprob,
            re_mode=args.remode,
            re_count=args.recount,
            re_split=args.resplit,
            scale=args.scale,
            ratio=args.ratio,
            hflip=args.hflip,
            vflip=args.vflip,
            color_jitter=args.color_jitter,
            auto_augment=args.aa,
            num_aug_repeats=args.aug_repeats,
            num_aug_splits=num_aug_splits,
            interpolation=train_interpolation,
            mean=data_config['mean'],
            std=data_config['std'],
            num_workers=args.workers,
            distributed=args.distributed,
            collate_fn=collate_fn,
            pin_memory=args.pin_mem,
            device=device,
            use_multi_epochs_loader=args.use_multi_epochs_loader,
            worker_seeding=args.worker_seeding,
            custom_sampler=custom_sampler
        )
        loaders_train.append(loader_train)

        eval_workers = args.workers
        if args.distributed and ('tfds' in args.dataset or 'wds' in args.dataset):
            # FIXME reduces validation padding issues when using TFDS, WDS w/ workers and distributed training
            eval_workers = min(2, args.workers)
        
        dataset_eval = datasets[eval_set][task_idx]
        eval_transforms = benchmark_config.eval_transforms[task_name]
        loader_eval = create_loader_v2(
            task_name,
            dataset_eval,
            input_size=data_config['input_size'],
            transform_list=eval_transforms,
            batch_size=args.validation_batch_size or args.batch_size,
            is_training=False,
            use_prefetcher=args.prefetcher,
            interpolation=data_config['interpolation'],
            mean=data_config['mean'],
            std=data_config['std'],
            num_workers=eval_workers,
            distributed=args.distributed,
            crop_pct=data_config['crop_pct'],
            pin_memory=args.pin_mem,
            device=device,
        )
        loaders_eval.append(loader_eval)

        if utils.is_primary(args):
            train_size = len(dataset_train)
            eval_size = len(dataset_eval)
            _logger.info(f"Task: {task_name}, Train Set Size: {train_size}, {eval_set} Set Size: {eval_size}")

    # setup loss function
    if args.jsd_loss:
        assert num_aug_splits > 1  # JSD only valid with aug splits set
        train_loss_fn = JsdCrossEntropy(num_splits=num_aug_splits, smoothing=args.smoothing)
    elif mixup_active:
        # smoothing is handled with mixup target transform which outputs sparse, soft targets
        if args.bce_loss:
            train_loss_fn = BinaryCrossEntropy(target_threshold=args.bce_target_thresh)
        else:
            train_loss_fn = SoftTargetCrossEntropy()
        if utils.is_primary(args):
            _logger.info(f"MixUp Active.\n{dash_line}")
    elif args.smoothing:
        if args.bce_loss:
            train_loss_fn = BinaryCrossEntropy(smoothing=args.smoothing, target_threshold=args.bce_target_thresh)
        else:
            train_loss_fn = LabelSmoothingCrossEntropy(smoothing=args.smoothing)
    else:
        train_loss_fn = nn.CrossEntropyLoss()
    train_loss_fn = train_loss_fn.to(device=device)
    validate_loss_fn = nn.CrossEntropyLoss().to(device=device)

    # setup checkpoint saver and eval metric tracking
    eval_metric = args.eval_metric

    start_epoch = 0
    if args.start_epoch is not None:
        # a specified start_epoch will always override the resume epoch
        start_epoch = args.start_epoch
    elif resume_epoch is not None:
        start_epoch = resume_epoch

    try:

        _model = unwrap_model(model)

        for task_idx, task_name in enumerate(runtime.task_order):

            _logger.info(f"-----------------\nStarting task {task_name}.")
            model.set_task_idx(task_idx)
            
            loader_train = loaders_train[task_idx]
            loader_eval = loaders_eval[task_idx]

            # Init optimizer and lr scheduler
            optimizer = create_optimizer_v2(
                model,
                **optimizer_kwargs(cfg=args),
                **args.opt_kwargs,
            )

            finetune_start = time.time()
            # If Imagenet, only calculate the importances
            if task_idx > 0:   

                updates_per_epoch = len(loader_train)
                lr_scheduler, num_epochs = create_scheduler_v2(
                    optimizer,
                    **scheduler_kwargs(args),
                    updates_per_epoch=updates_per_epoch,
                )

                if lr_scheduler is not None and start_epoch > 0:
                    if args.sched_on_updates:
                        lr_scheduler.step_update(start_epoch * updates_per_epoch)
                    else:
                        lr_scheduler.step(start_epoch)

                if utils.is_primary(args):
                    _logger.info(
                        f'Scheduled epochs: {num_epochs}. LR stepped per {"epoch" if lr_scheduler.t_in_epochs else "update"}.')

                best_metric = None
                best_epoch = None
                saver = None

                if utils.is_primary(args):
                    decreasing = True if eval_metric == 'loss' else False
                    saver = utils.CheckpointSaver(
                        model=model,
                        optimizer=optimizer,
                        args=args,
                        model_ema=model_ema,
                        amp_scaler=loss_scaler,
                        checkpoint_dir=output_dir,
                        recovery_dir=output_dir,
                        decreasing=decreasing,
                        max_history=args.checkpoint_hist
                    )
                    with open(os.path.join(output_dir, 'args.yaml'), 'w') as f:
                        f.write(args_text)

                for epoch in range(start_epoch, num_epochs):
                    if args.distributed and hasattr(loader_train.sampler, 'set_epoch'):
                        loader_train.sampler.set_epoch(epoch)

                    train_metrics = train_one_epoch(
                        epoch,
                        model,
                        loader_train,
                        optimizer,
                        train_loss_fn,
                        args,
                        lr_scheduler=lr_scheduler,
                        saver=saver,
                        output_dir=output_dir,
                        amp_autocast=amp_autocast,
                        loss_scaler=loss_scaler,
                        model_ema=model_ema,
                        mixup_fn=mixup_fn,
                    )

                    if args.distributed and args.dist_bn in ('broadcast', 'reduce'):
                        if utils.is_primary(args):
                            _logger.info("Distributing BatchNorm running means and vars")
                        utils.distribute_bn(model, args.world_size, args.dist_bn == 'reduce')

                    eval_metrics = validate(
                        model,
                        loader_eval,
                        validate_loss_fn,
                        args,
                        amp_autocast=amp_autocast,
                    )

                    if model_ema is not None and not args.model_ema_force_cpu:
                        if args.distributed and args.dist_bn in ('broadcast', 'reduce'):
                            utils.distribute_bn(model_ema, args.world_size, args.dist_bn == 'reduce')
                        ema_eval_metrics = validate(
                            model_ema.module,
                            loader_eval,
                            validate_loss_fn,
                            args,
                            amp_autocast=amp_autocast,
                            log_suffix=' (EMA)',
                        )
                        eval_metrics = ema_eval_metrics

                    if output_dir is not None:
                        lrs = [param_group['lr'] for param_group in optimizer.param_groups]
                        utils.update_summary(
                            epoch,
                            train_metrics,
                            eval_metrics,
                            filename=os.path.join(output_dir, 'summary.csv'),
                            lr=sum(lrs) / len(lrs),
                            write_header=best_metric is None,
                            log_wandb=args.log_wandb and has_wandb,
                        )

                    if saver is not None:
                        # save proper checkpoint with eval metric
                        save_metric = eval_metrics[eval_metric]
                        best_metric, best_epoch = saver.save_checkpoint(epoch, metric=save_metric)

                    if lr_scheduler is not None:
                        # step LR for next epoch
                        lr_scheduler.step(epoch + 1, eval_metrics[eval_metric])

                    if args.unit_test and epoch+1 == args.unit_test_epochs:
                        _logger.info(f"Unit Test: Exiting after {args.unit_test_epochs} epochs.")
                        break

                finetune_training_time = (time.time() - finetune_start) / 3600

                _logger.info(f"Eval Acc: {eval_metrics['top1']:.3f}")
                _logger.info(f"-----------------------\nFinished task {task_name}.\n-----------------------")

                if utils.is_primary(args):
                    _logger.info('total finetuning time = {:.4f} hours'.format(finetune_training_time))
                    if args.log_wandb and has_wandb:
                        wandb.log({f"{args.task_name}_finetune_time": finetune_training_time})

                if utils.is_primary(args):
                    # Checkpoint the model
                    checkpoint_save_path = Path(runtime.checkpoint_dir, f"checkpoint_{task_idx}.pth.tar")
                    checkpoint_save_path.parent.mkdir(exist_ok=True)
                    _logger.info(f"saving checkpoint to {checkpoint_save_path}")
                    module = model.module if args.distributed else model
                    torch.save(module.state_dict(), checkpoint_save_path)

                    # Remove the other checkpoints
                    checkpoint_dir = Path(output_dir)
                    checkpoint_files = checkpoint_dir.glob("*.pth.tar")
                    for checkpoint_file in checkpoint_files:
                        _logger.info(f"Removing file {checkpoint_file}")
                        checkpoint_file.unlink()

    except KeyboardInterrupt:
        pass

    if best_metric is not None:
        _logger.info('*** Best metric: {0} (epoch {1})'.format(best_metric, best_epoch))


def train_one_epoch(
        epoch,
        model,
        loader,
        optimizer,
        loss_fn,
        args,
        device=torch.device('cuda'),
        lr_scheduler=None,
        saver=None,
        output_dir=None,
        amp_autocast=suppress,
        loss_scaler=None,
        model_ema=None,
        mixup_fn=None,
        **kwargs
):
    if args.mixup_off_epoch and epoch >= args.mixup_off_epoch:
        if args.prefetcher and loader.mixup_enabled:
            loader.mixup_enabled = False
        elif mixup_fn is not None:
            mixup_fn.mixup_enabled = False

    second_order = hasattr(optimizer, 'is_second_order') and optimizer.is_second_order
    batch_time_m = utils.AverageMeter()
    data_time_m = utils.AverageMeter()
    losses_m = utils.AverageMeter()

    model.train()

    end = time.time()
    num_batches_per_epoch = len(loader)
    last_idx = num_batches_per_epoch - 1
    num_updates = epoch * num_batches_per_epoch
    for batch_idx, (input, target) in enumerate(loader):
        last_batch = batch_idx == last_idx
        if last_idx == 0:
            last_idx = 1
        data_time_m.update(time.time() - end)
        if not args.prefetcher:
            input, target = input.to(device), target.to(device)
            if mixup_fn is not None:
                input, target = mixup_fn(input, target)
        if args.channels_last:
            input = input.contiguous(memory_format=torch.channels_last)

        with amp_autocast():
            output = model(input, **kwargs)

            loss = loss_fn(output, target)

        if not args.distributed:
            losses_m.update(loss.item(), input.size(0))

        optimizer.zero_grad()
        if loss_scaler is not None:
            loss_scaler(
                loss, optimizer,
                clip_grad=args.clip_grad,
                clip_mode=args.clip_mode,
                parameters=model_parameters(model, exclude_head='agc' in args.clip_mode),
                create_graph=second_order
            )
        else:
            loss.backward(create_graph=second_order)
            if args.clip_grad is not None:
                utils.dispatch_clip_grad(
                    model_parameters(model, exclude_head='agc' in args.clip_mode),
                    value=args.clip_grad,
                    mode=args.clip_mode
                )
            optimizer.step()

        if model_ema is not None:
            model_ema.update(model)

        if device.type == 'cuda':
            torch.cuda.synchronize()

        num_updates += 1
        batch_time_m.update(time.time() - end)
        if last_batch or batch_idx % args.log_interval == 0:
            lrl = [param_group['lr'] for param_group in optimizer.param_groups]
            lr = sum(lrl) / len(lrl)

            if args.distributed:
                reduced_loss = utils.reduce_tensor(loss.data, args.world_size)
                losses_m.update(reduced_loss.item(), input.size(0))

            if utils.is_primary(args):
                _logger.info(
                    'Train: {} [{:>4d}/{} ({:>3.0f}%)]  '
                    'Loss: {loss.val:#.4g} ({loss.avg:#.3g})  '
                    'Time: {batch_time.val:.3f}s, {rate:>7.2f}/s  '
                    '({batch_time.avg:.3f}s, {rate_avg:>7.2f}/s)  '
                    'LR: {lr:.3e}  '
                    'Data: {data_time.val:.3f} ({data_time.avg:.3f})'.format(
                        epoch,
                        batch_idx, len(loader),
                        100. * batch_idx / last_idx,
                        loss=losses_m,
                        batch_time=batch_time_m,
                        rate=input.size(0) * args.world_size / batch_time_m.val,
                        rate_avg=input.size(0) * args.world_size / batch_time_m.avg,
                        lr=lr,
                        data_time=data_time_m)
                )

                if args.save_images and output_dir:
                    torchvision.utils.save_image(
                        input,
                        os.path.join(output_dir, 'train-batch-%d.jpg' % batch_idx),
                        padding=0,
                        normalize=True
                    )

        if saver is not None and args.recovery_interval and (
                last_batch or (batch_idx + 1) % args.recovery_interval == 0):
            saver.save_recovery(epoch, batch_idx=batch_idx)

        if lr_scheduler is not None:
            lr_scheduler.step_update(num_updates=num_updates, metric=losses_m.avg)

        end = time.time()
        # end for

    if hasattr(optimizer, 'sync_lookahead'):
        optimizer.sync_lookahead()

    return OrderedDict([('loss', losses_m.avg)])


def validate(
        model,
        loader,
        loss_fn,
        args,
        device=torch.device('cuda'),
        amp_autocast=suppress,
        log_suffix='',
        **kwargs
):
    batch_time_m = utils.AverageMeter()
    losses_m = utils.AverageMeter()
    top1_m = utils.AverageMeter()
    top5_m = utils.AverageMeter()

    model.eval()

    end = time.time()
    last_idx = len(loader) - 1
    with torch.no_grad():
        for batch_idx, (input, target) in enumerate(loader):
            last_batch = batch_idx == last_idx
            if not args.prefetcher:
                input = input.to(device)
                target = target.to(device)
            if args.channels_last:
                input = input.contiguous(memory_format=torch.channels_last)

            with amp_autocast():
                output = model(input, **kwargs)
                if isinstance(output, (tuple, list)):
                    output = output[0]

                # augmentation reduction
                reduce_factor = args.tta
                if reduce_factor > 1:
                    output = output.unfold(0, reduce_factor, reduce_factor).mean(dim=2)
                    target = target[0:target.size(0):reduce_factor]

                loss = loss_fn(output, target)
            acc1, acc5 = utils.accuracy(output, target, topk=(1, 5))

            if args.distributed:
                reduced_loss = utils.reduce_tensor(loss.data, args.world_size)
                acc1 = utils.reduce_tensor(acc1, args.world_size)
                acc5 = utils.reduce_tensor(acc5, args.world_size)
            else:
                reduced_loss = loss.data

            if device.type == 'cuda':
                torch.cuda.synchronize()

            losses_m.update(reduced_loss.item(), input.size(0))
            top1_m.update(acc1.item(), output.size(0))
            top5_m.update(acc5.item(), output.size(0))

            batch_time_m.update(time.time() - end)
            end = time.time()
            if utils.is_primary(args) and (last_batch or batch_idx % args.log_interval == 0):
                log_name = 'Test' + log_suffix
                _logger.info(
                    '{0}: [{1:>4d}/{2}]  '
                    'Time: {batch_time.val:.3f} ({batch_time.avg:.3f})  '
                    'Loss: {loss.val:>7.4f} ({loss.avg:>6.4f})  '
                    'Acc@1: {top1.val:>7.4f} ({top1.avg:>7.4f})  '
                    'Acc@5: {top5.val:>7.4f} ({top5.avg:>7.4f})'.format(
                        log_name, batch_idx, last_idx,
                        batch_time=batch_time_m,
                        loss=losses_m,
                        top1=top1_m,
                        top5=top5_m)
                )

    metrics = OrderedDict([('loss', losses_m.avg), ('top1', top1_m.avg), ('top5', top5_m.avg)])

    return metrics

if __name__ == '__main__':
    main()
