#    Copyright 2023 Rohan Taori, Ishaan Gulrajani, Tianyi Zhang, Yann Dubois, Xuechen Li
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.
import os
from os.path import exists, join, isdir
import numpy as np
import gc
import json
import math
import random
import copy
from copy import deepcopy
import logging
import json
from dataclasses import dataclass, field
from typing import Dict, Optional, Sequence, Callable, List, Tuple, Union, Any

import torch
from torch import nn
from torch.utils.data import Dataset
import utils

import transformers
from transformers import Trainer, BitsAndBytesConfig, set_seed

from peft import PeftModel
from transformers import AutoModelForCausalLM, AutoTokenizer

import warnings
from tqdm import tqdm

warnings.filterwarnings("ignore")

import transformers.integrations
import transformers.modeling_utils

IGNORE_INDEX = -100


@dataclass
class ModelArguments:
    model_name_or_path: Optional[str] = field(default="")


@dataclass
class DataArguments:
    data_path: str = field(
        default="data/select.json", metadata={"help": "Path to the training data."}
    )


@dataclass
class TrainingArguments(transformers.TrainingArguments):
    report_to: str = field(default="none")
    cache_dir: Optional[str] = field(default=None)
    optim: str = field(
        default="paged_adamw_32bit"
    )  # "paged_lion_8bit", "paged_adamw_8bit", "paged_lion_32bit", "paged_adamw_32bit"
    lr_scheduler_type: str = field(
        default="constant_with_warmup"
    )  # "constant", "constant_with_warmup", "cosine", "cosine_with_restarts", "linear"
    model_max_length: int = field(
        default=2048,
        metadata={
            "help": "Maximum sequence length. Sequences will be right padded (and possibly truncated)."
        },
    )


def _tokenize_fn(
    strings: Sequence[str], tokenizer: transformers.PreTrainedTokenizer
) -> Dict:
    """Tokenize a list of strings."""
    tokenized_list = [
        tokenizer(
            text,
            return_tensors="pt",
            padding="longest",
            max_length=tokenizer.model_max_length,
            truncation=True,
        )
        for text in strings
    ]
    input_ids = labels = [tokenized.input_ids[0] for tokenized in tokenized_list]
    input_ids_lens = labels_lens = [
        tokenized.input_ids.ne(tokenizer.pad_token_id).sum().item()
        for tokenized in tokenized_list
    ]
    return dict(
        input_ids=input_ids,
        labels=labels,
        input_ids_lens=input_ids_lens,
        labels_lens=labels_lens,
    )


def preprocess(
    sources: Sequence[str],
    targets: Sequence[str],
    tokenizer: transformers.PreTrainedTokenizer,
) -> Dict:
    """Preprocess the data by tokenizing."""
    examples = [s + t for s, t in zip(sources, targets)]
    examples_tokenized, sources_tokenized = [
        _tokenize_fn(strings, tokenizer) for strings in (examples, sources)
    ]
    input_ids = examples_tokenized["input_ids"]
    labels = copy.deepcopy(input_ids)
    # for label, source_len in zip(labels, sources_tokenized["input_ids_lens"]):
    #     label[:source_len] = IGNORE_INDEX
    return dict(input_ids=input_ids, labels=labels)


class SupervisedDataset(Dataset):
    """Dataset for supervised fine-tuning."""

    def __init__(self, data_path: str, tokenizer: transformers.PreTrainedTokenizer):
        super(SupervisedDataset, self).__init__()

        logging.warning("Loading data: {}".format(data_path))
        data_list = utils.jload(data_path)

        # Preprocess Data
        logging.warning("Processing data")
        self.tokenizer = tokenizer
        self.sources = []
        self.targets = []

        for idx in range(len(data_list)):
            data = data_list[idx]
            corpus = data["corpus"]
            if corpus != "":
                # pretrain mode
                source = f"{tokenizer.bos_token}"
                self.sources.append(source)

                target = f"{corpus}{tokenizer.eos_token}"
                self.targets.append(target)
            else:
                # instruction mode
                instruction = data["instruction"]
                conversation = data["conversation"]
                if len(conversation) == 1:
                    if instruction == "":
                        source = f"{tokenizer.bos_token}"
                    else:
                        source = f"{tokenizer.bos_token}### System:\n{instruction}\n"
                    source += (
                        f"### Human:\n{conversation[0]['input']}\n### Assistant:\n"
                    )
                    self.sources.append(source)

                    target = f"{conversation[0]['output']}{tokenizer.eos_token}"

                    self.targets.append(target)
                # else:
                # dialog mode

        del data_list
        gc.collect()

        logging.warning("there are {} data in dataset".format(len(self.sources)))

    def __len__(self):
        return len(self.sources)

    def __getitem__(self, i):
        # return dict(input_ids=self.input_ids[i], labels=self.labels[i])
        source = [self.sources[i]]
        target = [self.targets[i]]
        data_dict = preprocess(source, target, self.tokenizer)

        input_ids = data_dict["input_ids"][0]
        labels = data_dict["labels"][0]

        return dict(input_ids=input_ids, labels=labels)


@dataclass
class DataCollatorForSupervisedDataset(object):
    """Collate examples for supervised fine-tuning."""

    tokenizer: transformers.PreTrainedTokenizer

    def __call__(self, instances: Sequence[Dict]) -> Dict[str, torch.Tensor]:
        input_ids, labels = tuple(
            [instance[key] for instance in instances] for key in ("input_ids", "labels")
        )
        input_ids = torch.nn.utils.rnn.pad_sequence(
            input_ids, batch_first=True, padding_value=self.tokenizer.pad_token_id
        )
        labels = torch.nn.utils.rnn.pad_sequence(
            labels, batch_first=True, padding_value=IGNORE_INDEX
        )
        return dict(
            input_ids=input_ids,
            labels=labels,
            attention_mask=input_ids.ne(self.tokenizer.pad_token_id),
        )


def make_supervised_data_module(
    tokenizer: transformers.PreTrainedTokenizer, data_path
) -> Dict:
    """Make dataset and collator for supervised fine-tuning."""
    train_dataset = SupervisedDataset(
        tokenizer=tokenizer, data_path=data_path
    )
    data_collator = DataCollatorForSupervisedDataset(tokenizer=tokenizer)
    return dict(
        train_dataset=train_dataset, eval_dataset=None, data_collator=data_collator
    )


def train():
    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments)
    )
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    
    set_seed(42)
    all_results = []
    for check in [8296, 9272, 10248, 11224, 12200, 13176, 14640, 15616]:
        model_name_or_path = f"output/lora-steps/checkpoint-{check}"
        base_model = f"/root/paddlejob/workspace/env_run/huitingfeng/models/llama-2-7b-hf"
        
        tokenizer = AutoTokenizer.from_pretrained(
            model_name_or_path,
            trust_remote_code=True,
            model_max_length=2048
        )
        base_model = AutoModelForCausalLM.from_pretrained(
            base_model,
            trust_remote_code=True,
            low_cpu_mem_usage=True,
            torch_dtype=torch.bfloat16
        )
        model = PeftModel.from_pretrained(base_model, model_name_or_path)
        model.to("cuda:1")
        tokenizer.pad_token = tokenizer.eos_token
        data_module = make_supervised_data_module(tokenizer=tokenizer, data_path=data_args.data_path)
        losses = []
        with torch.no_grad():
            for step, batch in enumerate(tqdm(data_module["train_dataset"])):
                input_ids = batch["input_ids"].to("cuda:1")
                labels = batch["labels"].to("cuda:1")
                attention_mask = torch.ones_like(input_ids)
                attention_mask[input_ids == 0] = 0
                attention_mask.to("cuda:1")
                input_ids = input_ids.unsqueeze(0)
                labels = labels.unsqueeze(0)
                attention_mask = attention_mask.unsqueeze(0)
                result = model(input_ids=input_ids, labels=labels, attention_mask=attention_mask)
                losses.append(result.loss.detach().cpu().item())
        all_results.append(losses)

    def allocate_data(loss_lists, max_length):
        num_samples = len(loss_lists[0])
        num_lists = len(loss_lists)
        allocated_lists = [[] for _ in range(num_lists)]
        allocated_counts = [0] * num_lists

        for i in range(num_samples):
            loss_values = [loss_list[i] for loss_list in loss_lists]
            sorted_indices = np.argsort(loss_values)

            for idx in sorted_indices:
                if allocated_counts[idx] < max_length:
                    allocated_lists[idx].append(i)
                    allocated_counts[idx] += 1
                    break

        return allocated_lists

    allocated_lists = allocate_data(all_results, 800)
    
    
    with open(data_args.data_path, "r") as f:
        origin_data = json.load(f)
    for lst, check in zip(allocated_lists, [8296, 9272, 10248, 11224, 12200, 13176, 14640, 15616]):
        final_data = []
        for d in lst:
            final_data.append(origin_data[d])
        with open(f"data/{check}.json", "w") as f:
            json.dump(final_data, f, indent=4, ensure_ascii=False) 
    
if __name__ == "__main__":
    train()
    
