import re
import csv
import sys
from pathlib import Path

def parse_log_to_csv(log_path, csv_path):
    # Regex patterns (note the colon after 'took:')
    step_pattern = re.compile(r"Continuous step took:\s+([\d\.]+)\s*ms")
    ccd_pattern = re.compile(r"CCD Tested\s+(\d+)\s+triangles")
    version1_marker = re.compile(r"The next test has GIOM disabled")

    with open(log_path, "r") as f:
        lines = f.readlines()

    # Detect version
    is_version1 = any(version1_marker.search(line) for line in lines)

    rows = []
    frame_number = -1
    last_time = None
    step_triangles = []

    for line in lines:
        if "Discrete step took" in line:
            continue  # skip discrete steps

        step_match = step_pattern.search(line)
        ccd_match = ccd_pattern.search(line)

        if step_match:
            # flush previous frame if any
            if frame_number >= 0:
                if is_version1:
                    bm, nsdf = (0, 0)
                    if len(step_triangles) >= 1:
                        bm = step_triangles[0]
                    if len(step_triangles) >= 2:
                        nsdf = step_triangles[1]
                    rows.append([frame_number, last_time, bm, nsdf])
                else:
                    nsdf = step_triangles[0] if step_triangles else 0
                    rows.append([frame_number, last_time, 0, nsdf])

            # new frame
            frame_number += 1
            last_time = float(step_match.group(1))
            step_triangles = []

        elif ccd_match:
            step_triangles.append(int(ccd_match.group(1)))

    # flush last step
    if frame_number >= 0:
        if is_version1:
            bm, nsdf = (0, 0)
            if len(step_triangles) >= 1:
                bm = step_triangles[0]
            if len(step_triangles) >= 2:
                nsdf = step_triangles[1]
            rows.append([frame_number, last_time, bm, nsdf])
        else:
            nsdf = step_triangles[0] if step_triangles else 0
            rows.append([frame_number, last_time, 0, nsdf])

    # write CSV
    with open(csv_path, "w", newline="") as f:
        writer = csv.writer(f)
        writer.writerow(["frame_number", "physics_stepping_time_ms",
                         "triangles_bounding_mesh", "triangles_neural_sdf"])
        writer.writerows(rows)

    print(f"Saved {len(rows)} rows to {csv_path}")


if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("Usage: python parse_logs.py <logfile> <csvfile>")
        sys.exit(1)
    log_file = sys.argv[1]
    csv_file = sys.argv[2]
    parse_log_to_csv(log_file, csv_file)
