import pycosat
import random

def generate_clause(num_vars):
    """Generate a 3-SAT clause with 3 variables."""
    return [random.choice([i, -i]) for i in random.sample(range(1, num_vars + 1), 3)]

def generate_expression(num_vars, num_clauses):
    """Generate a 3-SAT expression consisting of a specified number of clauses."""
    return [generate_clause(num_vars) for _ in range(num_clauses)]

def check_satisfiability(expression):
    """Check the satisfiability of a 3-SAT expression, returning True or False."""
    return pycosat.solve(expression) != "UNSAT"

def clause_to_string(clause):
    """Convert a single clause into string format, e.g., [1, -2, 3] -> "(x1|~x2|x3)"."""
    clause_str = []
    for var in clause:
        if var < 0:
            clause_str.append(f"~x{-var}")
        else:
            clause_str.append(f"x{var}")
    return f"({'|'.join(clause_str)})"

def expression_to_string(expression):
    """Convert the entire 3-SAT expression into string format."""
    return "&".join([clause_to_string(clause) for clause in expression])

def generate_training_data(num_vars, num_clauses, num_samples):
    """Generate 3-SAT training data with expressions and their satisfiability labels."""
    satisfiable = []
    unsatisfiable = []
    for _ in range(num_samples):
        expr = generate_expression(num_vars, num_clauses)
        if check_satisfiability(expr):
            satisfiable.append(expression_to_string(expr))
        else:
            unsatisfiable.append(expression_to_string(expr))
    return satisfiable, unsatisfiable

if __name__ == "__main__":
    # Parameters
    num_vars = 4
    num_clauses = 20
    num_samples = 10000

    # Generate data
    satisfiable, unsatisfiable = generate_training_data(num_vars, num_clauses, num_samples)

    # Print statistics
    print(f"Number of satisfiable expressions: {len(satisfiable)}")
    print(f"Number of unsatisfiable expressions: {len(unsatisfiable)}")

    # Optional: Save or process results
    # For example, save to a file or convert to string for inspection
    satisfiable_strings = [expression_to_string(expr) for expr in satisfiable]
    unsatisfiable_strings = [expression_to_string(expr) for expr in unsatisfiable]

    # Example: Print a few samples
    print("\nExample Satisfiable Expressions:")
    for expr_str in satisfiable_strings[:3]:
        print(expr_str)

    print("\nExample Unsatisfiable Expressions:")
    for expr_str in unsatisfiable_strings[:3]:
        print(expr_str)
