"""
Bar chart curriculum generator
"""


import os
import json
import copy
from typing import List, Dict
from metachart import MetaGenerator


class BarChartGenerator(MetaGenerator):
    def __init__(self, args, chart_id):
        super().__init__(args)
        self.chart_type = args.chart_type
        self.chart_id = chart_id
        self.all_qa_data_list = []
        self.round_num = 2
        self.qa_idx = 0
        self.detailed_reasoning = args.detailed_reasoning
        self.force_ground = args.force_ground        
    


    ############################################################
    #   One-step Operator: h(list[Bar] | list[v]) → list[v]
    ############################################################
    
    def _one_step_statistics(self, chart_metadata: Dict):
        """
        Statistics: sum, mean, median, count
        """
        bar_singular = chart_metadata["bar_category"]["singular"]
        bar_plural = chart_metadata["bar_category"]["plural"]
        x_axis_title = chart_metadata["x_label"]
        y_axis_title = chart_metadata["y_label"]
        bar_list = [bar_idx for bar_idx in range(len(chart_metadata["bar_data"]))]
        target_bar_data = copy.deepcopy(chart_metadata["bar_data"])
        target_bar_label = copy.deepcopy(chart_metadata["bar_labels"])

        # Read
        x_singular = chart_metadata
        read_reason = '\n'.join([f"* {target_bar_label[bar_idx]}: {target_bar_data[bar_idx]}" for bar_idx in range(len(bar_list))]).strip()

        # Sum
        sum_answer = self._compute_data_sum(target_bar_data)
        sum_reason = f"{'+'.join([str(ddd) for ddd in target_bar_data])} = {sum_answer:.2f}"

        # Mean
        mean_answer = sum_answer / len(bar_list)
        mean_reason = f"{'+'.join([str(ddd) for ddd in target_bar_data])}/{len(bar_list)} = {sum_answer:.2f}/{len(bar_list)} = {mean_answer:.2f}"

        # Median
        median_value, median_indices = self._compute_data_median(target_bar_data)
        median_indices = [bar_list[relative_idx] for relative_idx in median_indices]

        # Median reasoning
        if len(target_bar_data) % 2 == 1:
            median_reason = f"* Sort {y_axis_title} values: {sorted(target_bar_data)}\n* The median is the middle value: {median_value}"
        else:
            sorted_data = sorted(target_bar_data)
            mid1, mid2 = sorted_data[len(sorted_data)//2-1], sorted_data[len(sorted_data)//2]
            median_reason = f"* Sort {y_axis_title} values: {sorted_data}\n* The median is the average of the two middle values: ({mid1} + {mid2})/2 = {median_value}"

        # Count
        count_answer = len(bar_list)

        # Chart QA Pool
        easy_qa_pool = {
            "one_step__statistics__sum": {
                "question": [
                    f"What is the total {y_axis_title} for all the {bar_plural} in this chart?",
                    f"For all the {bar_plural} in this chart, what is the sum of their {y_axis_title}?",
                    f"Can you help calculate the sum of {y_axis_title} for all the {bar_plural} in this chart?",
                    f"Please calculate the sum of {y_axis_title} for all the {bar_plural} in this chart.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their total {y_axis_title}" + (f":\n{sum_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": sum_answer,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
            "one_step__statistics__mean": {
                "question": [
                    f"What is the mean {y_axis_title} of all the {bar_plural} in this chart? Please round to two decimal places.",
                    f"For all the {bar_plural} in this chart, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the mean {y_axis_title} of all the {bar_plural} in this chart? Please round to two decimal places.",
                    f"Please compute the mean {y_axis_title} of all the {bar_plural} in this chart. Please round to two decimal places.",
                    f"What is the average {y_axis_title} of all the {bar_plural} in this chart? Please round to two decimal places.",
                    f"For all the {bar_plural} in this chart, what is their average {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the average {y_axis_title} of all the {bar_plural} in this chart? Please round to two decimal places.",
                    f"Please compute the average {y_axis_title} of all the {bar_plural} in this chart. Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their mean {y_axis_title}" + (f":\n{mean_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": mean_answer,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
            "one_step__statistics__median": {
                "question": [
                    f"What is the median {y_axis_title} among all the {bar_plural} in this chart?",
                    f"For all the {bar_plural} in this chart, what is the median of their {y_axis_title}?",
                    f"Can you help calculate the median {y_axis_title} of all the {bar_plural} in this chart?",
                    f"Please compute the median {y_axis_title} of all the {bar_plural} in this chart.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their median {y_axis_title}:\n{median_reason}",
                    },
                ],
                "constraint": "none",
                "answer": median_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": median_indices,
                    "answer": median_indices,
                },
            },
            "one_step__statistics__count": {
                "question": [
                    f"How many {bar_plural} are included in this chart?",
                    f"What is the number of {bar_plural} shown in this chart?",
                    f"Please help count the total number of {bar_plural} plotted in this chart.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to count how many {bar_plural} are shown in this chart" + (f": {count_answer} {bar_plural}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": count_answer,
                "mask": {
                    "step_1": bar_list,
                    "answer": bar_list,
                },
            },
        }

        return easy_qa_pool

    
    
    ############################################################
    #                     Two-step Operator
    ############################################################

    def _two_step_statistics(self, chart_metadata: Dict, bar_list: List, constraint: str):
        """
        Statistics: sum, mean, median, count
        """
        bar_list.sort()
        bar_singular = chart_metadata["bar_category"]["singular"]
        bar_plural = chart_metadata["bar_category"]["plural"]
        y_axis_title = chart_metadata["y_label"]
        target_bar_data = [chart_metadata["bar_data"][bar_idx] for bar_idx in bar_list]
        target_bar_label = [chart_metadata["bar_labels"][bar_idx] for bar_idx in bar_list]

        # Reason
        read_reason = '\n'.join([f"* {target_bar_label[bar_idx]}: {target_bar_data[bar_idx]}" for bar_idx in range(len(bar_list))]).strip()
        
        # Sum
        sum_answer = self._compute_data_sum(target_bar_data)
        sum_reason = f"{'+'.join([str(ddd) for ddd in target_bar_data])} = {sum_answer:.2f}"

        # Mean
        mean_answer = sum_answer / len(bar_list)
        mean_reason = f"{'+'.join([str(ddd) for ddd in target_bar_data])}/{len(bar_list)} = {sum_answer:.2f}/{len(bar_list)} = {mean_answer:.2f}"

        # Median
        median_value, median_indices = self._compute_data_median(target_bar_data)
        median_indices = [bar_list[relative_idx] for relative_idx in median_indices]

        # Median reasoning
        if len(target_bar_data) % 2 == 1:
            median_reason = f"* Sort their {y_axis_title} values: {sorted(target_bar_data)}\n* The median is the middle value: {median_value}"
        else:
            sorted_data = sorted(target_bar_data)
            mid1, mid2 = sorted_data[len(sorted_data)//2-1], sorted_data[len(sorted_data)//2]
            median_reason = f"* Sort their {y_axis_title} values: {sorted_data}\n* The median is the average of the two middle values: ({mid1} + {mid2})/2 = {median_value}"

        # Count
        count_answer = len(bar_list)

        # Chart QA Pool
        medium_qa_pool = {
            "two_step__statistics__sum": {
                "question": [
                    f"What is the sum of {y_axis_title} for bars {constraint}?",
                    f"For the bars {constraint}, what is the sum of their {y_axis_title}?",
                    f"Can you help calculate the sum of {y_axis_title} for bars {constraint}?",
                    f"Please compute the sum of {y_axis_title} for bars {constraint}.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the bars {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their total {y_axis_title}" + (f":\n{sum_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": sum_answer,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
            "two_step__statistics__mean": {
                "question": [
                    f"What is the mean {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"For the bars {constraint}, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the mean {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"Please compute the mean {y_axis_title} for bars {constraint}. Please round to two decimal places.",
                    f"What is the average {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"For the bars {constraint}, what is their average {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the average {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"Please compute the average {y_axis_title} for bars {constraint}. Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the bars {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their mean {y_axis_title}" + (f":\n{mean_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": mean_answer,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
            "two_step__statistics__median": {
                "question": [
                    f"What is the median value of {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"For the bars {constraint}, what is the median value of their {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the median value of {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"Please compute the median value of {y_axis_title} for bars {constraint}. Please round to two decimal places.",
                    f"What is the median of {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"For the bars {constraint}, what is the median of their {y_axis_title}? Please round to two decimal places.",
                    f"Can you help calculate the median of {y_axis_title} for bars {constraint}? Please round to two decimal places.",
                    f"Please compute the median of {y_axis_title} for bars {constraint}. Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the bars {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their median {y_axis_title}" + (f":\n{median_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": median_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": median_indices,
                    "answer": median_indices,
                },
            },
            "two_step__statistics__count": {
                "question": [
                    f"How many bars {constraint} are shown in this chart?",
                    f"What is the number of bars {constraint}?",
                    f"Please help count the number of bars {constraint}.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to find all the bars {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to count the total number of these bars" + (f": {count_answer} bars" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": count_answer,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list,
                    "answer": bar_list,
                },
            },
        }

        return medium_qa_pool


    ############################################################
    #                     Multi-step Operator
    ############################################################

    def _multi_step_threshold(self, chart_metadata: Dict):
        """
        Threshold: above / below mean, differences
        """
        bar_singular = chart_metadata["bar_category"]["singular"]
        bar_plural = chart_metadata["bar_category"]["plural"]
        x_axis_title = chart_metadata["x_label"]
        y_axis_title = chart_metadata["y_label"]
        bar_list = [i for i in range(len(chart_metadata["bar_data"]))]
        target_bar_data = copy.deepcopy(chart_metadata["bar_data"])
        target_bar_label = copy.deepcopy(chart_metadata["bar_labels"])
        
        # Sum
        value_sum = self._compute_data_sum(target_bar_data)
        sumup_str = "+".join([str(nn) for nn in target_bar_data])
        sum_str = f"{sumup_str} = {value_sum:.2f}"

        # Mean
        value_mean = self._compute_data_mean(target_bar_data)

        # Above & below count
        above_num, below_num, bar_idx = 0, 0, 0
        above_bar_idx, below_bar_idx = [], []
        above_bar_data, below_bar_data = [], []
        above_bar_label, below_bar_label = [], []
        for v in target_bar_data:
            if v > value_mean:
                above_num += 1
                above_bar_idx.append(bar_idx)
                above_bar_data.append(v)
                above_bar_label.append(target_bar_label[bar_idx])
            elif v < value_mean:
                below_num += 1
                below_bar_idx.append(bar_idx)
                below_bar_data.append(v)
                below_bar_label.append(target_bar_label[bar_idx])
            bar_idx += 1

        # Sum sublist
        above_sum = self._compute_data_sum(above_bar_data)
        below_sum = self._compute_data_sum(below_bar_data)
        above_sum_str = f"{'+'.join([str(nn) for nn in above_bar_data])} = {above_sum:.2f}"
        below_sum_str = f"{'+'.join([str(nn) for nn in below_bar_data])} = {below_sum:.2f}"

        # Mean sublist
        above_mean = above_sum/above_num
        below_mean = below_sum/below_num
        above_mean_str = f"({'+'.join([str(nn) for nn in above_bar_data])})/{above_num} = {above_sum:.2f}/{above_num} = {above_mean:.2f}"
        below_mean_str = f"({'+'.join([str(nn) for nn in below_bar_data])})/{below_num} = {below_sum:.2f}/{below_num} = {below_mean:.2f}"

        # Max/min among above-mean sublist
        max_above_mean_value = max(above_bar_data)
        min_above_mean_value = min(above_bar_data)
        max_above_value_indices = self._find_indices_in_list(target_bar_data, max_above_mean_value)
        min_above_value_indices = self._find_indices_in_list(target_bar_data, min_above_mean_value)
        max_above_mean_label = self._convert_answer_idx_to_str(target_bar_label, max_above_value_indices)
        min_above_mean_label = self._convert_answer_idx_to_str(target_bar_label, min_above_value_indices)

        # Max/min among below-mean sublist
        max_below_mean_value = max(below_bar_data)
        min_below_mean_value = min(below_bar_data)
        max_below_value_indices = self._find_indices_in_list(target_bar_data, max_below_mean_value)
        min_below_value_indices = self._find_indices_in_list(target_bar_data, min_below_mean_value)
        max_below_mean_label = self._convert_answer_idx_to_str(target_bar_label, max_below_value_indices)
        min_below_mean_label = self._convert_answer_idx_to_str(target_bar_label, min_below_value_indices)

        # Difference between the sum of above-mean bars and the sum of below-mean bars
        sum_diff = above_sum - below_sum
        mean_diff = (above_sum/above_num)-(below_sum/below_num)

        # Reason
        read_reason = '\n'.join([f"* {target_bar_label[bar_idx]}: {target_bar_data[bar_idx]}" for bar_idx in range(len(bar_list))]).strip()
        read_reason_above_mean = '\n'.join([f"* {above_bar_label[bar_idx]}: {above_bar_data[bar_idx]}" for bar_idx in range(len(above_bar_label))]).strip()
        read_reason_below_mean = '\n'.join([f"* {below_bar_label[bar_idx]}: {below_bar_data[bar_idx]}" for bar_idx in range(len(below_bar_label))]).strip()

        # Chart QA Pool
        hard_qa_pool = {
            "multi_step__threshold__above_mean__max__value": {
                "question": [
                    f"What is the highest {y_axis_title} among bars that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"What is the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"What is the highest {y_axis_title} among bars that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"What is the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the highest {y_axis_title} among these {bar_plural}" + (f":\n{max_above_mean_label}: {max_above_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "max value among bars above mean",
                "answer": max_above_mean_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": max_above_value_indices,
                    "answer": max_above_value_indices,
                },
            },
            "multi_step__threshold__above_mean__min__value": {
                "question": [
                    f"What is the lowest {y_axis_title} among bars that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"What is the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"What is the lowest {y_axis_title} among bars that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"What is the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the lowest {y_axis_title} among these {bar_plural}" + (f":\n{min_above_mean_label}: {min_above_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "min value among bars above mean",
                "answer": min_above_mean_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": min_above_value_indices,
                    "answer": min_above_value_indices,
                },
            },
            "multi_step__threshold__above_mean__max__label": {
                "question": [
                    f"Which {bar_singular} has the highest {y_axis_title} among bars that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Which {bar_singular} has the highest {y_axis_title} among bars that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the highest {y_axis_title} among these {bar_plural}" + (f":\n{max_above_mean_label}: {max_above_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "label of the max value among bars above mean",
                "answer": max_above_mean_label,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": max_above_value_indices,
                    "answer": max_above_value_indices,
                },
            },
            "multi_step__threshold__above_mean__min__label": {
                "question": [
                    f"Which {bar_singular} has the lowest {y_axis_title} among bars that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among bars that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the lowest {y_axis_title} among these {bar_plural}" + (f":\n{min_above_mean_label}: {min_above_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "label of the min value among bars above mean",
                "answer": min_above_mean_label,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": min_above_value_indices,
                    "answer": min_above_value_indices,
                },
            },
            "multi_step__threshold__below_mean__max__value": {
                "question": [
                    f"What is the highest {y_axis_title} among bars that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"What is the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"What is the highest {y_axis_title} among bars that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"What is the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the highest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the highest {y_axis_title} among these {bar_plural}" + (f":\n{max_below_mean_label}: {max_below_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "max value among bars below mean",
                "answer": max_below_mean_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": max_below_value_indices,
                    "answer": max_below_value_indices,
                },
            },
            "multi_step__threshold__below_mean__min__value": {
                "question": [
                    f"What is the lowest {y_axis_title} among bars that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"What is the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"What is the lowest {y_axis_title} among bars that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"What is the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the lowest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the lowest {y_axis_title} among these {bar_plural}" + (f":\n{min_below_mean_label}: {min_below_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "min value among bars below mean",
                "answer": min_below_mean_value,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": min_below_value_indices,
                    "answer": min_below_value_indices,
                },
            },
            "multi_step__threshold__below_mean__max__label": {
                "question": [
                    f"Which {bar_singular} has the highest {y_axis_title} among bars that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Which {bar_singular} has the highest {y_axis_title} among bars that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the highest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, which {bar_singular} has the highest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the highest {y_axis_title} among these {bar_plural}" + (f":\n{max_below_mean_label}: {max_below_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "label of the max value among bars below mean",
                "answer": max_below_mean_label,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": max_below_value_indices,
                    "answer": max_below_value_indices,
                },
            },
            "multi_step__threshold__below_mean__min__label": {
                "question": [
                    f"Which {bar_singular} has the lowest {y_axis_title} among bars that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among bars that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Which {bar_singular} has the lowest {y_axis_title} among {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, which {bar_singular} has the lowest {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to compute their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to identify the {bar_singular} with the lowest {y_axis_title} among these {bar_plural}" + (f":\n{min_below_mean_label}: {min_below_mean_value}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "label of the min value among bars below mean",
                "answer": min_below_mean_label,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": min_below_value_indices,
                    "answer": min_below_value_indices,
                },
            },
            "multi_step__threshold__above_mean__sum": {
                "question": [
                    f"What is the sum of {y_axis_title} for bars that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"What is the sum of {y_axis_title} for {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"What is the sum of {y_axis_title} for bars that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"What is the sum of {y_axis_title} for {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate the total {y_axis_title} of these {bar_plural}" + (f": {above_sum_str}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "sum of bars above mean",
                "answer": above_sum,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": above_bar_idx if self.force_ground else [],
                    "answer": above_bar_idx,
                },
            },
            "multi_step__threshold__below_mean__sum": {
                "question": [
                    f"What is the sum of {y_axis_title} for bars that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"What is the sum of {y_axis_title} for {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"What is the sum of {y_axis_title} for bars that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"What is the sum of {y_axis_title} for {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars?",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is the sum of their {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate the total {y_axis_title} of these {bar_plural}" + (f": {below_sum_str}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "sum of bars below mean",
                "answer": below_sum,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": below_bar_idx if self.force_ground else [],
                    "answer": below_bar_idx,
                },
            },
            "multi_step__threshold__above_mean__mean": {
                "question": [
                    f"What is the mean value of {y_axis_title} for bars that have their {y_axis_title} above the average {y_axis_title} of all bars? Please round to two decimal places.",
                    f"What is the mean value of {y_axis_title} for {bar_plural} that have their {y_axis_title} above the average {y_axis_title} of all bars? Please round to two decimal places.",
                    f"Among bars whose {y_axis_title} are above the average {y_axis_title} of all bars, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"Among {bar_plural} whose {y_axis_title} are above the average {y_axis_title} of all bars, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"What is the average value of {y_axis_title} for bars that have their {y_axis_title} above the mean {y_axis_title} of all bars? Please round to two decimal places.",
                    f"What is the average value of {y_axis_title} for {bar_plural} that have their {y_axis_title} above the mean {y_axis_title} of all bars? Please round to two decimal places.",
                    f"Among bars whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is their average {y_axis_title}? Please round to two decimal places.",
                    f"Among {bar_plural} whose {y_axis_title} are above the mean {y_axis_title} of all bars, what is their average {y_axis_title}? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate their average {y_axis_title}" + (f":\n({'+'.join([str(nn) for nn in above_bar_data])})/{above_num} = {above_sum:.2f}/{above_num} = {above_mean:.2f}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "mean of bars above mean",
                "answer": above_mean,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": above_bar_idx,
                    "step_4": above_bar_idx if self.force_ground else [],
                    "answer": above_bar_idx,
                },
            },
            "multi_step__threshold__below_mean__mean": {
                "question": [
                    f"What is the mean value of {y_axis_title} for bars that have their {y_axis_title} below the average {y_axis_title} of all bars? Please round to two decimal places.",
                    f"What is the mean value of {y_axis_title} for {bar_plural} that have their {y_axis_title} below the average {y_axis_title} of all bars? Please round to two decimal places.",
                    f"Among bars whose {y_axis_title} are below the average {y_axis_title} of all bars, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"Among {bar_plural} whose {y_axis_title} are below the average {y_axis_title} of all bars, what is their mean {y_axis_title}? Please round to two decimal places.",
                    f"What is the average value of {y_axis_title} for bars that have their {y_axis_title} below the mean {y_axis_title} of all bars? Please round to two decimal places.",
                    f"What is the average value of {y_axis_title} for {bar_plural} that have their {y_axis_title} below the mean {y_axis_title} of all bars? Please round to two decimal places.",
                    f"Among bars whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is their average {y_axis_title}? Please round to two decimal places.",
                    f"Among {bar_plural} whose {y_axis_title} are below the mean {y_axis_title} of all bars, what is their average {y_axis_title}? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find all the {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate their average {y_axis_title}" + (f":\n({'+'.join([str(nn) for nn in below_bar_data])})/{below_num} = {below_sum:.2f}/{below_num} = {below_mean:.2f}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "mean of bars below mean",
                "answer": below_mean,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": below_bar_idx,
                    "step_4": below_bar_idx if self.force_ground else [],
                    "answer": below_bar_idx,
                },
            },
            "multi_step__threshold__mean__sum_diff": {
                "question": [
                    f"What is the absolute difference between the total {y_axis_title} of bars above the average {y_axis_title} and those below it?",
                    f"What is the absolute value of the difference between the total {y_axis_title} for bars above the average and those below the average {y_axis_title}?",
                    f"What is the absolute difference between the total {y_axis_title} of bars above the mean {y_axis_title} and those below it?",
                    f"What is the absolute value of the difference between the total {y_axis_title} for bars above the mean {y_axis_title} and those below the mean {y_axis_title}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find the first group of {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f} and the second group of {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n(1) Above average:\n{read_reason_above_mean}\n(2) Below average:\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate the total {y_axis_title} of these two groups of {bar_plural} respectively" + (f":\n(1) Above average sum: {above_sum_str}\n(2) Below average sum: {below_sum_str}" if self.detailed_reasoning else "."),
                        "step_5": f"Fifth, I can get the absolute difference between these two groups of {bar_plural}" + (f":\n|{above_sum:.2f} - {below_sum:.2f}| = {sum_diff:.2f}." if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "sum difference bewteen bars above and below mean",
                "answer": sum_diff,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": bar_list,
                    "step_4": bar_list if self.force_ground else [],
                    "step_5": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
            "multi_step__threshold__mean__mean_diff": {
                "question": [
                    f"What is the absolute difference between the average {y_axis_title} of bars above and below the overall average? Please round to two decimal places.",
                    f"What is the absolute difference between the mean {y_axis_title} of bars above the overall average and those below it? Please round to two decimal places.",
                    f"What is the absolute difference between the average {y_axis_title} of bars whose values are above the overall average and those whose values are below it? Please round to two decimal places.",
                    f"What is the absolute difference between the average {y_axis_title} of {bar_plural} above and below the overall average? Please round to two decimal places.",
                    f"What is the absolute difference between the mean {y_axis_title} of {bar_plural} above the overall average and those below it? Please round to two decimal places.",
                    f"What is the absolute difference between the average {y_axis_title} of {bar_plural} whose values are above the overall average and those whose values are below it? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read the {y_axis_title} of each {bar_singular} in this chart" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their average {y_axis_title}" + (f": {value_sum:.2f}/{len(bar_list)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                        "step_3": f"Third, I need to find the first group of {bar_plural} whose {y_axis_title} is higher than {value_mean:.2f} and the second group of {bar_plural} whose {y_axis_title} is lower than {value_mean:.2f}" + (f":\n(1) Above average:\n{read_reason_above_mean}\n(2) Below average:\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                        "step_4": f"Fourth, I need to calculate the mean {y_axis_title} for these two groups of {bar_plural} respectively" + (f":\n(1) Above average mean: {above_mean_str}\n(2) Below average mean: {below_mean_str}" if self.detailed_reasoning else "."),
                        "step_5": f"Fifth, I can get the absolute difference between these two groups of {bar_plural}" + (f":\n|{above_mean:.2f} - {below_mean:.2f}| = {mean_diff:.2f}." if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "mean difference bewteen bars above and below mean",
                "answer": mean_diff,
                "mask": {
                    "step_1": bar_list,
                    "step_2": bar_list if self.force_ground else [],
                    "step_3": bar_list,
                    "step_4": bar_list if self.force_ground else [],
                    "step_5": bar_list if self.force_ground else [],
                    "answer": bar_list,
                },
            },
        }

        return hard_qa_pool
    