"""
Radar chart curriculum generator
"""

import os
import json
import copy
from typing import List, Dict
from metachart import MetaGenerator


class RadarChartGenerator(MetaGenerator):
    def __init__(self, args, chart_id):
        super().__init__(args)
        self.chart_type = args.chart_type
        self.chart_id = chart_id
        self.all_qa_data_list = []
        self.round_num = 2
        self.qa_idx = 0
        self.detailed_reasoning = args.detailed_reasoning
        self.force_ground = args.force_ground


    ############################################################
    #   One-step Operator: h(list[Radar] | list[v]) → list[v]
    ############################################################
    
    def _one_step_statistics(self, chart_metadata: Dict):
        """
        Statistics: sum, mean, median, count for radar charts
        """
        data_singular = chart_metadata["radar_data_category"]["singular"]
        data_plural = chart_metadata["radar_data_category"]["plural"]
        label_singular = chart_metadata["radar_label_category"]["singular"]
        label_plural = chart_metadata["radar_label_category"]["plural"]
        
        radar_data = chart_metadata["radar_data"]
        radar_labels = chart_metadata["radar_labels"]
        data_labels = chart_metadata["data_labels"]
        
        num_series = len(radar_data)
        num_axes = len(radar_labels)
        
        # Flatten all data points for overall statistics
        all_values = []
        all_indices = []
        for series_idx, series in enumerate(radar_data):
            for axis_idx, value in enumerate(series):
                all_values.append(value)
                all_indices.append((series_idx, axis_idx))
        
        # Read reason - show all data points
        read_reason = ""
        for series_idx, (series, data_label) in enumerate(zip(radar_data, data_labels)):
            read_reason += f"* {data_label}:\n"
            for axis_idx, (value, radar_label) in enumerate(zip(series, radar_labels)):
                read_reason += f"  - {radar_label}: {value}\n"
        read_reason = read_reason.strip()

        # Sum
        sum_answer = self._compute_data_sum(all_values)
        sum_reason = f"* Sum of all values: {'+'.join([str(v) for v in all_values])} = {sum_answer:.2f}"

        # Mean
        mean_answer = sum_answer / len(all_values)
        mean_reason = f"* Mean = {sum_answer:.2f}/{len(all_values)} = {mean_answer:.2f}"

        # Median
        median_value, relative_median_indices = self._compute_data_median(all_values)
        median_indices = [all_indices[idx] for idx in relative_median_indices]
        
        # Median reasoning
        if len(all_values) % 2 == 1:
            median_reason = f"* Sorted values: {sorted(all_values)}\n* The median is the middle value: {median_value}"
        else:
            sorted_values = sorted(all_values)
            mid1, mid2 = sorted_values[len(sorted_values)//2-1], sorted_values[len(sorted_values)//2]
            median_reason = f"* Sorted values: {sorted_values}\n* The median is the average of the two middle values: ({mid1} + {mid2})/2 = {median_value}"

        # Count
        count_answer = len(all_values)

        # Chart QA Pool
        easy_qa_pool = {
            "one_step__statistics__sum": {
                "question": [
                    f"What is the total sum of all values across all {data_plural} and {label_plural} in this radar chart? Please round to two decimal places.",
                    f"Calculate the sum of all data points for all {data_plural} and {label_plural} shown in this chart. Please round to two decimal places.",
                    f"What is the total of all values displayed in this radar chart for all {data_plural} and {label_plural}? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate the total sum of all these values" + (f":\n{sum_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": f"{sum_answer:.2f}",
                "mask": {
                    "step_1": all_indices,
                    "step_2": all_indices if self.force_ground else [],
                    "answer": all_indices,
                },
            },
            "one_step__statistics__mean": {
                "question": [
                    f"What is the mean value across all {data_plural} and {label_plural} in this radar chart? Please round to two decimal places.",
                    f"Calculate the average value of all data points for all {data_plural} and {label_plural} shown in this chart. Please round to two decimal places.",
                    f"What is the overall average of all values displayed in this radar chart? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate the mean of all these values" + (f":\n{mean_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": mean_answer,
                "mask": {
                    "step_1": all_indices,
                    "step_2": all_indices if self.force_ground else [],
                    "answer": all_indices,
                },
            },
            "one_step__statistics__median": {
                "question": [
                    f"What is the median value across all {data_plural} and {label_plural} in this radar chart?",
                    f"Calculate the median of all data points for all {data_plural} and {label_plural} shown in this chart.",
                    f"What is the middle value of all data points displayed in this radar chart?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to find the median of all these values" + (f":\n{median_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": median_value,
                "mask": {
                    "step_1": all_indices,
                    "step_2": median_indices,
                    "answer": median_indices,
                },
            },
            "one_step__statistics__count": {
                "question": [
                    f"How many total data points are displayed in this radar chart across all {data_plural} and {label_plural}?",
                    f"What is the total number of values shown for all {data_plural} and {label_plural} in this chart?",
                    f"Count the total number of data points in this radar chart.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to count all data points across {num_series} {data_plural} and {num_axes} {label_plural}" + (f": {num_series} × {num_axes} = {count_answer} data points" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": "none",
                "answer": count_answer,
                "mask": {
                    "step_1": all_indices,
                    "answer": all_indices,
                },
            },
        }

        return easy_qa_pool
    
    
    ############################################################
    #                     Two-step Operator
    ############################################################

    def _two_step_statistics(self, chart_metadata: Dict, target_indices: List, constraint: str):
        """
        Statistics: sum, mean, median, count for specific data points in radar charts
        """
        data_singular = chart_metadata["radar_data_category"]["singular"]
        data_plural = chart_metadata["radar_data_category"]["plural"]
        label_singular = chart_metadata["radar_label_category"]["singular"]
        label_plural = chart_metadata["radar_label_category"]["plural"]
        
        radar_data = chart_metadata["radar_data"]
        radar_labels = chart_metadata["radar_labels"]
        data_labels = chart_metadata["data_labels"]
        
        # Extract target values and create read reason
        target_values = []
        read_reason = ""
        
        # Group by series for better readability
        series_data = {}
        for series_idx, axis_idx in target_indices:
            if series_idx not in series_data:
                series_data[series_idx] = []
            series_data[series_idx].append((axis_idx, radar_data[series_idx][axis_idx], radar_labels[axis_idx]))
            target_values.append(radar_data[series_idx][axis_idx])
        
        for series_idx in sorted(series_data.keys()):
            read_reason += f"* {data_labels[series_idx]}:\n"
            for axis_idx, value, label in sorted(series_data[series_idx]):
                read_reason += f"  - {label}: {value}\n"
        read_reason = read_reason.strip()

        # Sum
        sum_answer = self._compute_data_sum(target_values)
        sum_reason = f"* Sum: {' + '.join([str(v) for v in target_values])} = {sum_answer:.2f}"

        # Mean
        mean_answer = sum_answer / len(target_values)
        mean_reason = f"* Mean: {sum_answer:.2f} / {len(target_values)} = {mean_answer:.2f}"

        # Median
        median_value, relative_median_indices = self._compute_data_median(target_values)
        median_indices = [target_indices[idx] for idx in relative_median_indices]
        
        # Median reasoning
        if len(target_values) % 2 == 1:
            median_reason = f"* Sorted values: {sorted(target_values)}\n* The median is the middle value: {median_value}"
        else:
            sorted_values = sorted(target_values)
            mid1, mid2 = sorted_values[len(sorted_values)//2-1], sorted_values[len(sorted_values)//2]
            median_reason = f"* Sorted values: {sorted_values}\n* The median is the average of the two middle values: ({mid1} + {mid2})/2 = {median_value}"

        # Count
        count_answer = len(target_values)

        # Chart QA Pool
        medium_qa_pool = {
            "two_step__statistics__sum": {
                "question": [
                    f"What is the sum of values for data points {constraint}?",
                    f"Calculate the total of all values for data points {constraint}.",
                    f"What is the combined value for data points {constraint}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the data points {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their sum" + (f":\n{sum_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": sum_answer,
                "mask": {
                    "step_1": target_indices,
                    "step_2": target_indices if self.force_ground else [],
                    "answer": target_indices,
                },
            },
            "two_step__statistics__mean": {
                "question": [
                    f"What is the mean value for data points {constraint}? Please round to two decimal places.",
                    f"Calculate the average value for data points {constraint}. Please round to two decimal places.",
                    f"What is the average of values for data points {constraint}? Please round to two decimal places.",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the data points {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their mean" + (f":\n{mean_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": mean_answer,
                "mask": {
                    "step_1": target_indices,
                    "step_2": target_indices if self.force_ground else [],
                    "answer": target_indices,
                },
            },
            "two_step__statistics__median": {
                "question": [
                    f"What is the median value for data points {constraint}?",
                    f"Calculate the median of values for data points {constraint}.",
                    f"What is the middle value for data points {constraint}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the data points {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to calculate their median" + (f":\n{median_reason}" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": median_value,
                "mask": {
                    "step_1": target_indices,
                    "step_2": median_indices,
                    "answer": median_indices,
                },
            },
            "two_step__statistics__count": {
                "question": [
                    f"How many data points {constraint} are there in this radar chart?",
                    f"What is the count of data points {constraint}?",
                    f"How many values are there for data points {constraint}?",
                ],
                "reasoning": [
                    {
                        "step_1": f"First, I need to identify the data points {constraint}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                        "step_2": f"Second, I need to count how many they are" + (f": {count_answer} data points" if self.detailed_reasoning else "."),
                    },
                ],
                "constraint": constraint,
                "answer": count_answer,
                "mask": {
                    "step_1": target_indices,
                    "step_2": target_indices,
                    "answer": target_indices,
                },
            },
        }

        return medium_qa_pool


    ############################################################
    #                     Multi-step Operator
    ############################################################

    def _multi_step_threshold(self, chart_metadata: Dict):
        """
        Multi-step threshold: complex operations on above/below mean data points
        """
        data_singular = chart_metadata["radar_data_category"]["singular"]
        data_plural = chart_metadata["radar_data_category"]["plural"]
        label_singular = chart_metadata["radar_label_category"]["singular"]
        label_plural = chart_metadata["radar_label_category"]["plural"]
        
        radar_data = chart_metadata["radar_data"]
        radar_labels = chart_metadata["radar_labels"]
        data_labels = chart_metadata["data_labels"]
        
        # Flatten all data points
        all_values = []
        all_indices = []
        for series_idx, series in enumerate(radar_data):
            for axis_idx, value in enumerate(series):
                all_values.append(value)
                all_indices.append((series_idx, axis_idx))
        
        # Read reason - show all data points
        read_reason = ""
        for series_idx, (series, data_label) in enumerate(zip(radar_data, data_labels)):
            read_reason += f"* {data_label}:\n"
            for axis_idx, (value, radar_label) in enumerate(zip(series, radar_labels)):
                read_reason += f"  - {radar_label}: {value}\n"
        read_reason = read_reason.strip()
        
        # Sum and Mean
        value_sum = self._compute_data_sum(all_values)
        value_mean = self._compute_data_mean(all_values)

        # Separate above and below mean data
        above_values, below_values = [], []
        above_indices, below_indices = [], []
        
        for idx, value in enumerate(all_values):
            if value > value_mean:
                above_values.append(value)
                above_indices.append(all_indices[idx])
            elif value < value_mean:
                below_values.append(value)
                below_indices.append(all_indices[idx])

        # Create read reasons for above and below mean groups
        read_reason_above_mean = ""
        above_series_data = {}
        for series_idx, axis_idx in above_indices:
            if series_idx not in above_series_data:
                above_series_data[series_idx] = []
            value = radar_data[series_idx][axis_idx]
            above_series_data[series_idx].append((axis_idx, value, radar_labels[axis_idx]))
        
        for series_idx in sorted(above_series_data.keys()):
            read_reason_above_mean += f"* {data_labels[series_idx]}:\n"
            for axis_idx, value, label in sorted(above_series_data[series_idx]):
                read_reason_above_mean += f"  - {label}: {value}\n"
        read_reason_above_mean = read_reason_above_mean.strip()

        read_reason_below_mean = ""
        below_series_data = {}
        for series_idx, axis_idx in below_indices:
            if series_idx not in below_series_data:
                below_series_data[series_idx] = []
            value = radar_data[series_idx][axis_idx]
            below_series_data[series_idx].append((axis_idx, value, radar_labels[axis_idx]))
        
        for series_idx in sorted(below_series_data.keys()):
            read_reason_below_mean += f"* {data_labels[series_idx]}:\n"
            for axis_idx, value, label in sorted(below_series_data[series_idx]):
                read_reason_below_mean += f"  - {label}: {value}\n"
        read_reason_below_mean = read_reason_below_mean.strip()

        # Calculations for above-mean group
        if above_values:
            above_sum = self._compute_data_sum(above_values)
            above_mean = self._compute_data_mean(above_values)
            above_sum_str = f"{' + '.join([str(v) for v in above_values])} = {above_sum:.2f}"
            above_mean_str = f"({' + '.join([str(v) for v in above_values])}) / {len(above_values)} = {above_sum:.2f} / {len(above_values)} = {above_mean:.2f}"
            
            # Max/min among above-mean group
            max_above_value = max(above_values)
            min_above_value = min(above_values)
            max_above_indices = [above_indices[i] for i, v in enumerate(above_values) if v == max_above_value]
            min_above_indices = [above_indices[i] for i, v in enumerate(above_values) if v == min_above_value]
            
            # Create labels for max/min above mean
            max_above_labels = []
            for series_idx, axis_idx in max_above_indices:
                max_above_labels.append(f"{data_labels[series_idx]} ({radar_labels[axis_idx]})")
            max_above_labels_str = ", ".join(max_above_labels)
            
            min_above_labels = []
            for series_idx, axis_idx in min_above_indices:
                min_above_labels.append(f"{data_labels[series_idx]} ({radar_labels[axis_idx]})")
            min_above_labels_str = ", ".join(min_above_labels)

        # Calculations for below-mean group
        if below_values:
            below_sum = self._compute_data_sum(below_values)
            below_mean = self._compute_data_mean(below_values)
            below_sum_str = f"{' + '.join([str(v) for v in below_values])} = {below_sum:.2f}"
            below_mean_str = f"({' + '.join([str(v) for v in below_values])}) / {len(below_values)} = {below_sum:.2f} / {len(below_values)} = {below_mean:.2f}"
            
            # Max/min among below-mean group
            max_below_value = max(below_values)
            min_below_value = min(below_values)
            max_below_indices = [below_indices[i] for i, v in enumerate(below_values) if v == max_below_value]
            min_below_indices = [below_indices[i] for i, v in enumerate(below_values) if v == min_below_value]
            
            # Create labels for max/min below mean
            max_below_labels = []
            for series_idx, axis_idx in max_below_indices:
                max_below_labels.append(f"{data_labels[series_idx]} ({radar_labels[axis_idx]})")
            max_below_labels_str = ", ".join(max_below_labels)
            
            min_below_labels = []
            for series_idx, axis_idx in min_below_indices:
                min_below_labels.append(f"{data_labels[series_idx]} ({radar_labels[axis_idx]})")
            min_below_labels_str = ", ".join(min_below_labels)

        # Differences between groups
        if above_values and below_values:
            sum_diff = abs(above_sum - below_sum)
            mean_diff = abs(above_mean - below_mean)

        # Chart QA Pool
        hard_qa_pool = {}
        
        if above_values:
            hard_qa_pool.update({
                "multi_step__threshold__above_mean__max__value": {
                    "question": [
                        f"What is the highest value among data points that have values above the overall average in this radar chart?",
                        f"Among data points with above-average values, what is the maximum value?",
                        f"What is the largest value among data points that exceed the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to identify the highest value among these data points" + (f": {max_above_value}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "max value among data points above mean",
                    "answer": max_above_value,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": above_indices,
                        "step_4": max_above_indices,
                        "answer": max_above_indices,
                    },
                },
                "multi_step__threshold__above_mean__min__value": {
                    "question": [
                        f"What is the lowest value among data points that have values above the overall average in this radar chart?",
                        f"Among data points with above-average values, what is the minimum value?",
                        f"What is the smallest value among data points that exceed the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to identify the lowest value among these data points" + (f": {min_above_value}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "min value among data points above mean",
                    "answer": min_above_value,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": above_indices,
                        "step_4": min_above_indices,
                        "answer": min_above_indices,
                    },
                },
                "multi_step__threshold__above_mean__max__label": {
                    "question": [
                        f"Which {data_singular} and {label_singular} combination has the highest value among data points that are above the overall average?",
                        f"Among data points with above-average values, which shows the maximum value?",
                        f"Which data point has the largest value among those that exceed the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to identify which data point has the highest value among these" + (f": {max_above_labels_str}: {max_above_value}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "label of max value among data points above mean",
                    "answer": max_above_labels_str,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": above_indices,
                        "step_4": max_above_indices,
                        "answer": max_above_indices,
                    },
                },
                "multi_step__threshold__above_mean__sum": {
                    "question": [
                        f"What is the sum of all values for data points that are above the overall average in this radar chart?",
                        f"Calculate the total of all above-average data points.",
                        f"What is the combined value of all data points that exceed the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to calculate their total sum" + (f": {above_sum_str}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "sum of data points above mean",
                    "answer": above_sum,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": above_indices,
                        "step_4": above_indices if self.force_ground else [],
                        "answer": above_indices,
                    },
                },
                "multi_step__threshold__above_mean__mean": {
                    "question": [
                        f"What is the average value of data points that are above the overall average in this radar chart? Please round to two decimal places.",
                        f"Calculate the mean of all above-average data points. Please round to two decimal places.",
                        f"What is the average of data points that exceed the overall mean? Please round to two decimal places.",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values higher than {value_mean:.2f}" + (f":\n{read_reason_above_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to calculate their average" + (f": {above_mean_str}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "mean of data points above mean",
                    "answer": above_mean,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": above_indices,
                        "step_4": above_indices if self.force_ground else [],
                        "answer": above_indices,
                    },
                },
            })
        
        if below_values:
            hard_qa_pool.update({
                "multi_step__threshold__below_mean__max__value": {
                    "question": [
                        f"What is the highest value among data points that have values below the overall average in this radar chart?",
                        f"Among data points with below-average values, what is the maximum value?",
                        f"What is the largest value among data points that are below the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to identify the highest value among these data points" + (f": {max_below_value}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "max value among data points below mean",
                    "answer": max_below_value,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": below_indices,
                        "step_4": max_below_indices,
                        "answer": max_below_indices,
                    },
                },
                "multi_step__threshold__below_mean__min__value": {
                    "question": [
                        f"What is the lowest value among data points that have values below the overall average in this radar chart?",
                        f"Among data points with below-average values, what is the minimum value?",
                        f"What is the smallest value among data points that are below the overall mean?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to find all data points with values lower than {value_mean:.2f}" + (f":\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to identify the lowest value among these data points" + (f": {min_below_value}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "min value among data points below mean",
                    "answer": min_below_value,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": below_indices,
                        "step_4": min_below_indices,
                        "answer": min_below_indices,
                    },
                },
            })

        if above_values and below_values:
            hard_qa_pool.update({
                "multi_step__threshold__mean__sum_diff": {
                    "question": [
                        f"What is the absolute difference between the total sum of above-average data points and below-average data points?",
                        f"Calculate the absolute difference between the sum of data points above the mean and those below the mean.",
                        f"What is the absolute value of the difference between the total of above-average and below-average data points?",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to separate data points into above-average and below-average groups" + (f":\n(1) Above average:\n{read_reason_above_mean}\n(2) Below average:\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to calculate the sum for each group" + (f":\n(1) Above average sum: {above_sum_str}\n(2) Below average sum: {below_sum_str}" if self.detailed_reasoning else "."),
                            "step_5": f"Fifth, I need to find the absolute difference" + (f": |{above_sum:.2f} - {below_sum:.2f}| = {sum_diff:.2f}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "sum difference between above and below mean data points",
                    "answer": sum_diff,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": all_indices,
                        "step_4": all_indices if self.force_ground else [],
                        "step_5": all_indices if self.force_ground else [],
                        "answer": all_indices,
                    },
                },
                "multi_step__threshold__mean__mean_diff": {
                    "question": [
                        f"What is the absolute difference between the average of above-average data points and the average of below-average data points? Please round to two decimal places.",
                        f"Calculate the absolute difference between the mean of data points above the overall mean and those below it. Please round to two decimal places.",
                        f"What is the absolute value of the difference between the averages of above-mean and below-mean data points? Please round to two decimal places.",
                    ],
                    "reasoning": [
                        {
                            "step_1": f"First, I need to read all values for each {data_singular} across all {label_plural}" + (f":\n{read_reason}" if self.detailed_reasoning else "."),
                            "step_2": f"Second, I need to compute the overall average" + (f": {value_sum:.2f} / {len(all_values)} = {value_mean:.2f}" if self.detailed_reasoning else "."),
                            "step_3": f"Third, I need to separate data points into above-average and below-average groups" + (f":\n(1) Above average:\n{read_reason_above_mean}\n(2) Below average:\n{read_reason_below_mean}" if self.detailed_reasoning else "."),
                            "step_4": f"Fourth, I need to calculate the average for each group" + (f":\n(1) Above average mean: {above_mean_str}\n(2) Below average mean: {below_mean_str}" if self.detailed_reasoning else "."),
                            "step_5": f"Fifth, I need to find the absolute difference" + (f": |{above_mean:.2f} - {below_mean:.2f}| = {mean_diff:.2f}" if self.detailed_reasoning else "."),
                        },
                    ],
                    "constraint": "mean difference between above and below mean data points",
                    "answer": mean_diff,
                    "mask": {
                        "step_1": all_indices,
                        "step_2": all_indices if self.force_ground else [],
                        "step_3": all_indices,
                        "step_4": all_indices if self.force_ground else [],
                        "step_5": all_indices if self.force_ground else [],
                        "answer": all_indices,
                    },
                },
            })

        return hard_qa_pool