# Bounding CVaR via Distributional Discrepancies

This project implements methods for bounding Conditional Value at Risk (CVaR) using distributional discrepancies. It provides tools for comparing different sampling methods and their computational efficiency.

## Features

- CVaR estimation and bounding algorithms
- Truncated Gaussian Mixture Model (GMM) sampling
- Normal distribution approximations
- Parallel computation support
- Visualization tools for convergence analysis
- Performance benchmarking
- Thomas bounds comparison
- Monte Carlo probability analysis
- Cross-platform compatibility (all paths use current directory)

## Installation

### Using pip

```bash
pip install -r requirements.txt
```

### Using virtual environment (recommended)

```bash
# Create virtual environment
python3 -m venv .venv

# Activate virtual environment
# On Windows:
.venv\Scripts\activate
# On macOS/Linux:
source .venv/bin/activate

# Install dependencies
pip install -r requirements.txt
```

### Development installation

```bash
pip install -e ".[dev]"
```

## Usage

### Basic CVaR Estimation

```python
import numpy as np
from dist_dicrepancy_bounds.bounds import cvar_estimator

# Generate sample data
samples = np.random.normal(0, 1, 1000)

# Calculate CVaR with alpha = 0.1
cvar = cvar_estimator(samples, alpha=0.1)
print(f"CVaR (α=0.1): {cvar}")
```

### Running Simulations

The project includes several simulation scripts that can be run directly:

```bash
# Activate virtual environment
source .venv/bin/activate

# Run comparison with Thomas bounds
python dist_dicrepancy_bounds/comparison_to_thomas.py

# Run comprehensive GMM vs Normal simulations
python dist_dicrepancy_bounds/distributional_discrepancies_cvar_bounds_simulations.py

# Run Monte Carlo probability analysis
python dist_dicrepancy_bounds/stochastic_bound_probability.py
```

### Output

All scripts generate plots and analysis results in the `./dist_disc_plots/` directory:
- `cvar_bounds_comparison.png` - Comparison of Thomas vs Stochastic bounds
- `sample_histograms.png` - Distribution histograms
- `sample_cdfs.png` - Cumulative distribution functions
- `cvar_convergence_with_bounds.png` - CVaR convergence analysis
- `sampling_time_comparison.png` - Performance comparisons
- `cvar_computation_time_comparison.png` - Computation time analysis

## Project Structure

```
bounding_cvar_via_distributional_discrepancies/
├── dist_dicrepancy_bounds/                          # Main package directory
│   ├── bounds.py                                    # Core CVaR functions
│   ├── comparison_to_thomas.py                      # Thomas bounds comparison
│   ├── distributional_discrepancies_cvar_bounds_simulations.py  # GMM vs Normal simulations
│   ├── stochastic_bound_probability.py              # Monte Carlo probability analysis
│   ├── bounded_actions_plot.py                      # Bounded actions visualization
│   ├── lower_and_upper_F_plots.py                  # F-function plots
│   └── __init__.py                                 # Package initialization
├── dist_disc_plots/                                 # Generated plots and outputs
├── .venv/                                          # Virtual environment
├── requirements.txt                                 # Python dependencies
├── setup.py                                        # Package setup
├── pyproject.toml                                  # Modern Python packaging
└── README.md                                       # This file
```

## Dependencies

- **numpy**: Numerical computing
- **scipy**: Scientific computing and statistics
- **matplotlib**: Plotting and visualization
- **joblib**: Parallel processing

## Development

### Code Formatting

```bash
black dist_dicrepancy_bounds/
```

### Type Checking

```bash
mypy dist_dicrepancy_bounds/
```

### Linting

```bash
flake8 dist_dicrepancy_bounds/
```

### Cache Management

The project uses relative paths for all cache directories:
- `./__pycache__/` - Python bytecode cache
- `./.cache` - General cache files
- `./.pytest_cache/` - pytest cache
- `./.mypy_cache/` - mypy type checking cache

## Recent Results

The project has been successfully tested with the following results:

### Comparison to Thomas Bounds
- **Coverage Rate**: 100% for all tested distributions
- **Distributions Tested**: Truncated Normal, Beta(2,2), Beta(0.5,0.5), Beta(2,5), Beta(5,2), Beta(10,2), Beta(2,10), Laplace(0,1)
- **Sample Sizes**: 100 to 1000 samples
- **Result**: Both Thomas and Stochastic bounds provide excellent coverage

### GMM vs Normal Approximation Performance
- **GMM Sampling**: 3.20x slower than Normal approximation
- **CVaR Computation**: GMM 2.79x slower than Normal bounds
- **KS Distance**: 0.0224 between GMM and Normal distributions
- **Truncation Impact**: 6.03x slower for GMM, 11.20x slower for Normal

### Monte Carlo Probability Analysis
- **Normal Distributions**: 0.00% probability of CVaR falling outside bounds
- **Binomial Distributions**: 0.00% probability of CVaR falling outside bounds
- **Beta Distributions**: 0.00% probability of CVaR falling outside bounds
- **Result**: Perfect coverage across all distribution types

## License

MIT License - see LICENSE file for details.

## Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Ensure code quality (formatting, linting, type checking)
5. Submit a pull request

## Citation

If you use this code in your research, please cite:
