import torch
import random
import time, sys
import argparse
import logging
import numpy as np
import ipdb as pdb
import os, pwd, yaml
import pytorch_lightning as pl
from torch.utils.data import Subset
from torch._utils import _accumulate
from torch.utils.data import DataLoader, random_split
from fssm.modules.stationarypred2step import StationaryProcessPred2Step
from fssm.modules.nonstationarypred2step import NonStationaryProcessPred2Step
from fssm.tools.utils import load_yaml, setup_seed
from fssm.datasets.sim_dataset import StationaryDataset
from pytorch_lightning.callbacks.early_stopping import EarlyStopping
from pytorch_lightning.callbacks import ModelCheckpoint
from pytorch_lightning.loggers import TensorBoardLogger

import warnings
warnings.filterwarnings('ignore')

def main(args):
    pl.seed_everything(args.seed)
    assert args.exp is not None, "FATAL: "+__file__+": You must specify an exp configs file (e.g., *.yaml)"
    current_user = pwd.getpwuid(os.getuid()).pw_name
    script_dir = os.path.dirname(__file__)
    rel_path = os.path.join('../fssm/configs',
                            '%s.yaml'%args.exp)
    abs_file_path = os.path.join(script_dir, rel_path)
    cfg = load_yaml(abs_file_path)

    # update configs dic fromm cmd line
    if args.epoch >= 0:
        cfg['VAE']['EPOCHS'] = args.epoch


    # load dataset
    data = StationaryDataset(directory=cfg['ROOT'],
                             transition=cfg['DATASET'])
    num_validation_samples = cfg['VAE']['N_VAL_SAMPLES']
    indices = list(range(len(data)))
    lengths = [len(data) - num_validation_samples, num_validation_samples]
    train_data, val_data = [Subset(data, indices[offset - length: offset]) for offset, length in
                            zip(_accumulate(lengths), lengths)]
    train_loader = DataLoader(train_data,
                              batch_size=cfg['VAE']['TRAIN_BS'], 
                              pin_memory=cfg['VAE']['PIN'],
                              num_workers=cfg['VAE']['CPU'],
                              drop_last=False,
                              shuffle=True)

    val_loader = DataLoader(val_data,
                            batch_size=cfg['VAE']['VAL_BS'], 
                            pin_memory=cfg['VAE']['PIN'],
                            num_workers=cfg['VAE']['CPU'],
                            shuffle=False)

    # define logfile
    log_dir = os.path.join(cfg["LOG"], current_user, args.exp.split('_')[-1])
    if args.logfolder_suffixmodel is None:
        model_name = args.model
    else:
        model_name = args.model + "_" + args.logfolder_suffixmodel

    # version name
    if args.model == 'stationary':
        version_name = f"lr{cfg['VAE']['LR']}_beta{cfg['VAE']['BETA']}_gamma{cfg['VAE']['GAMMA']}"
    elif args.model == 'nonstationary':
        version_name = f"lr{cfg['VAE']['LR']}_beta{cfg['VAE']['BETA']}_gamma{cfg['VAE']['GAMMA']}_delta{float(cfg['VAE']['DELTA'])}_epsilon{cfg['VAE']['EPSILON']}"
    if args.logfolder_suffix:
        version_name = version_name + "_" + args.logfolder_suffix


    model_path = os.path.join(log_dir, model_name, version_name)
    if not os.path.exists(model_path):
        print(f"cound not find {model_name}_{version_name}!")
        sys.exit()

    model_file = model_path + '/checkpoints/' + os.listdir(model_path + '/checkpoints/')[cfg['PREDICTOR']['PRE_IDX']]
    if args.model == 'stationary':
        model = StationaryProcessPred2Step(pretrain_path= model_file,
                                           input_dim=cfg['VAE']['INPUT_DIM'],
                                           length=cfg['VAE']['LENGTH'],
                                           z_dim=cfg['VAE']['LATENT_DIM'],
                                           lag=cfg['VAE']['LAG'],
                                           hidden_dim=cfg['VAE']['ENC']['HIDDEN_DIM'],
                                           trans_prior=cfg['VAE']['TRANS_PRIOR'],
                                           predict_mode=cfg['PREDICTOR']['PREDICT_MODE'],
                                           lr=cfg['VAE']['LR'],
                                           pred_lr=cfg['PREDICTOR']['LR'],
                                           beta=cfg['VAE']['BETA'],
                                           gamma=cfg['VAE']['GAMMA'],
                                           decoder_dist=cfg['VAE']['DEC']['DIST'],
                                           correlation=cfg['MCC']['CORR']
                                            )
    elif args.model == 'nonstationary':
        model = NonStationaryProcessPred2Step(pretrain_path=model_file,
                                           input_dim=cfg['VAE']['INPUT_DIM'],
                                           length=cfg['VAE']['LENGTH'],
                                           z_dim=cfg['VAE']['LATENT_DIM'],
                                           lag=cfg['VAE']['LAG'],
                                           hidden_dim=cfg['VAE']['ENC']['HIDDEN_DIM'],
                                           trans_prior=cfg['VAE']['TRANS_PRIOR'],
                                           predict_mode=cfg['PREDICTOR']['PREDICT_MODE'],
                                           predict_witha=cfg['PREDICTOR']['PRE_WITHA'],
                                           lstm_layer=cfg['PREDICTOR']['LSTM_LAYER'],
                                           lr=cfg['VAE']['LR'],
                                           pred_lr=cfg['PREDICTOR']['LR'],
                                           beta=cfg['VAE']['BETA'],
                                           gamma=cfg['VAE']['GAMMA'],
                                           delta=cfg['VAE']['DELTA'],
                                           epsilon=cfg['VAE']['EPSILON'],
                                           a_distribution=cfg['VAE']['DEC']['A_DIST'],
                                           decoder_dist=cfg['VAE']['DEC']['DIST'],
                                           correlation=cfg['MCC']['CORR']
                                           )

    if not torch.cuda.is_available():
        cfg['VAE']['GPU'] = None
        train_loader.num_workers = 0
        val_loader.num_workers = 0

    model_name = model_name + "_pretrained"
    if args.model == 'stationary':
        version_name = version_name + f"_predlr{cfg['PREDICTOR']['LR']}_predmode{cfg['PREDICTOR']['PREDICT_MODE']}"
    if args.model == 'nonstationary':
        version_name = version_name + f"_predlr{cfg['PREDICTOR']['LR']}_predlstmlayer{cfg['PREDICTOR']['LSTM_LAYER']}_predwitha{cfg['PREDICTOR']['PRE_WITHA']}"
    logger_TB = TensorBoardLogger(log_dir, name=model_name, version=version_name)
    log_dir = os.path.join(log_dir, model_name, version_name)
    logfile_name = f"{version_name}_{int(time.time())}.log"
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)
    else:
        print(f"{model_name}_{version_name} exist!")
        sys.exit()

    logger = logging.getLogger()
    logger.setLevel(logging.INFO)
    output_file_handler = logging.FileHandler(os.path.join(log_dir, logfile_name), mode='w')
    logger.addHandler(output_file_handler)

    logging.info("######### Configuration #########")
    logging.info("#################################")

    trainer = pl.Trainer(logger=logger_TB,
                         default_root_dir=log_dir,
                         gpus=cfg['VAE']['GPU'],
                         auto_select_gpus=True,
                         max_epochs=cfg['VAE']['EPOCHS'],
                         progress_bar_refresh_rate = 0,
                        )

    # Train the model
    trainer.fit(model, train_loader, val_loader)
    logging.info("end")

if __name__ == "__main__":
    argparser = argparse.ArgumentParser(description=__doc__)
    argparser.add_argument('-e', '--exp', type=str)
    argparser.add_argument('-s', '--seed', type=int, default=770)
    argparser.add_argument('-m', '--model', type=str, default='stationary')
    argparser.add_argument('--epoch', type=int, default=-1)
    args = argparser.parse_args()
    main(args)
