

import argparse
import builtins
import math
import os
import random
import shutil
import time
import warnings
import pprint
import logging

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torchvision.models as tvmodels

import methods
import models
from datasets import *
from losses import *
from utils import *


model_names = ['preactresnet18', 'resnet18', 'resnet34']


parser = argparse.ArgumentParser(description='PyTorch SSL for CIFAR')
parser.add_argument('--mode', default='ssl', type=str, help='training mode', choices=['ssl', 'sup', 'lnl'])
parser.add_argument('-r', '--noise-rate', metavar='NOISERATE', default=0.0, type=float, help='noise rate')
parser.add_argument('-t', '--noise-type', metavar='NOISETYPE', default='symmetric', type=str, help='noise type', choices=['symmetric', 'asymmetric'])
parser.add_argument('--tau', default=0.0, type=float, help='tau')
parser.add_argument('--mu', default=10.0, type=float, help='mu')
parser.add_argument('--lamb', default=1.0, type=float, help='lamb')

parser.add_argument('-d', '--dataset', metavar='DATASET', default='cifar10')
parser.add_argument('--img-size', default=64, type=int, help='image size')
parser.add_argument('--data-path', metavar='DATAPATH', default='/home/zhouxiong/database/')
parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet50',
                    help='model architecture: ' + ' | '.join(model_names) + ' (default: resnet50)')
parser.add_argument('--feat-dim', type=int, default=4096)
parser.add_argument('--hidden-dim', type=int, default=4096)
parser.add_argument('--need-projector', default=True, action='store_true')

# training settings
parser.add_argument('-j', '--workers', default=32, type=int, metavar='N',
                    help='number of data loading workers (default: 32)')
parser.add_argument('--epochs', default=200, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=1536, type=int, metavar='N',
                    help='mini-batch size (default: 512), this is the total '
                        'batch size of all GPUs on the current node when '
                        'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lars', action='store_true', help='Use LARS')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial (base) learning rate', dest='lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float, metavar='W', help='weight decay (default: 0.)', dest='weight_decay')


# others
parser.add_argument('-p', '--print-freq', default=20, type=int, metavar='N', help='print frequency (default: 20)')
parser.add_argument('--resume', default='', type=str, metavar='PATH', help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true', help='evaluate model on validation set')
parser.add_argument('--world-size', default=-1, type=int, help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int, help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str, help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str, help='distributed backend')
parser.add_argument('--seed', default=0, type=int, help='seed for initializing training. ')
parser.add_argument('--gpu', default=0, type=int, help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')

# additional configs:
parser.add_argument('--save', default='', type=str, help='save path')
parser.add_argument('--save-freq', default=100, type=int, metavar='N', help='save frequency (default: 50)')
parser.add_argument('--val-freq', default=100, type=int, metavar='N', help='validation frequency (default: 25)')
parser.add_argument('--version', default='', type=str)


best_acc1 = 0.0

def get_logger(args):
    head = '%(asctime)-15s %(message)s'
    logging.basicConfig(filename=os.path.join(args.save, 'log.txt'), format=head)
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)
    console = logging.StreamHandler()
    logging.getLogger('').addHandler(console)
    return logger


def get_local_dataloader(args):
    if args.dataset.lower() in ['cifar10', 'cifar100', 'svhn']:
        if args.mode == 'lnl':
            train_set = eval('Noisy' + args.dataset.upper()+'_TwoCrops')(
                True, os.path.join(args.data_path, args.dataset.upper()),
                noise_rate=args.noise_rate, is_asym=args.noise_type=='asymmetric'
            )
        elif args.mode in ['ssl', 'sup']:
            train_set = eval(args.dataset.upper()+'_TwoCrops')(
                True, os.path.join(args.data_path, args.dataset.upper())
            )

    if args.dataset.lower().startswith('imagenet'):
        train_set = ImageNet_TwoCrops(True, nori_prefix=args.dataset.lower(), spatial_size=args.img_size)
    elif args.dataset.lower() == 'domainnet':
        train_set = DomainNet_TwoCrops(True, nori_prefix=args.dataset.lower(), spatial_size=args.img_size)

    if args.dataset.lower() in ['svhn', 'cifar10']:
        args.num_classes = 10
    elif args.dataset.lower() in ['cifar100', 'imagenet100']:
        args.num_classes = 100
    elif args.dataset.lower() == 'imagenet':
        args.num_classes = 1000
    elif args.dataset.lower() == 'domainnet':
        args.num_classes = 345

    sampler = None
    batch_size = args.batch_size
    if args.distributed:
        batch_size = batch_size // dist.get_world_size()
        sampler = torch.utils.data.distributed.DistributedSampler(train_set)
    dataloader = torch.utils.data.DataLoader(
        train_set, batch_size=batch_size, 
        num_workers=args.workers, sampler=sampler,
        shuffle=not args.distributed, drop_last=args.distributed,
        pin_memory=True, persistent_workers=True
    )
    return dataloader


def get_val_dataloader(args):
    if args.dataset.lower() == 'cifar10':
        val_set = CIFAR10(False, os.path.join(args.data_path, args.dataset.upper()))
    elif args.dataset.lower() == 'cifar100':
        val_set = CIFAR100(False, os.path.join(args.data_path, args.dataset.upper()))
    elif args.dataset.lower() == 'svhn':
        val_set = SVHN(False, os.path.join(args.data_path, args.dataset.upper()))
    elif args.dataset.lower().startswith('imagenet'):
        val_set = ImageNet(False, nori_prefix=args.dataset, spatial_size=args.img_size)
    elif args.dataset.lower().startswith('domainnet'):
        val_set = DomainNet(False, nori_prefix=args.dataset, spatial_size=args.img_size)
    dataloader = torch.utils.data.DataLoader(
        val_set, batch_size=256, shuffle=False,
        num_workers=args.workers, pin_memory=True
    )
    return dataloader


def main():
    global logger
    args = parser.parse_args()
    root = '/data/SPECTRAL/' + args.mode.upper()
    if args.mode == 'lnl':
        root = os.path.join(root, '_'.join([args.dataset, args.noise_type, str(args.noise_rate)]))
    else:
        root = os.path.join(root, args.dataset)
    args.save = os.path.join(root, '_'.join([str(i) for i in [args.arch, args.epochs, 'spectral'+args.version, args.mu, args.tau]]))
    os.makedirs(args.save, exist_ok=True)

    logger = get_logger(args)
    logger.info('\n' + pprint.pformat(args))

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        torch.backends.cudnn.benchmark = False
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')
    
    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely disable data ta parallelism.')
    
    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])
    
    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()

    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    logger = get_logger(args)
    args.gpu = gpu

    # supress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:
        def print_pass(*args):
            pass
        builtins.print = print_pass
    
    if args.gpu is not None:
        logger.info("Use GPU: {} for training".format(args.gpu))
    
    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
        torch.distributed.barrier()
    
    # loading data
    train_loader = get_local_dataloader(args)
    val_loader = get_val_dataloader(args)

    # create model
    logger.info("=> creating model with '{}'".format(args.arch))
    if args.arch in model_names:
        model = methods.Base(models.__dict__[args.arch], need_projector=args.need_projector, dim=args.feat_dim, hidden_dim=args.hidden_dim)
    else:
        model = methods.Base(tvmodels.__dict__[args.arch], need_projector=args.need_projector, dim=args.feat_dim, hidden_dim=args.hidden_dim)

    classifier = nn.Sequential(
        nn.Linear(args.feat_dim, args.num_classes)
    )


    if args.distributed:
        # Apply SyncBN
        model = torch.nn.SyncBatchNorm.convert_sync_batchnorm(model)
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourves based on the total number of GPUs we have
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])

            classifier.cuda(args.gpu)
            classifier = torch.nn.parallel.DistributedDataParallel(classifier, device_ids=[args.gpu])

        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)

            classifier.cuda(args.gpu)
            classifier = torch.nn.parallel.DistributedDataParallel(classifier)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
        classifier = classifier.cuda(args.gpu)
    else:
        # AllGather implementation (batch shuffle, queue update, etc.) in
        # this code only supports DistributedDataParallel.
        raise NotImplementedError("Only DistributedDataParallel and one gpu is supported.")

    print(model) # print model after SyncBatchNorm


    # define loss function (criterion) and optimizer
    if args.mode == 'ssl':
        if args.version == 'v1':
            criterion = SpectralContrastiveLossV1(tau=args.tau, mu=args.mu).cuda(args.gpu)
        elif args.version == 'v2':
            criterion = SpectralContrastiveLossV2(tau=args.tau, mu=args.mu).cuda(args.gpu)
        else:
            criterion = SpectralContrastiveLoss(tau=args.tau, mu=args.mu).cuda(args.gpu)
    elif args.mode in ['sup', 'lnl']:
        criterion = SupervisedSpectralContrastiveLoss(tau=args.tau, mu=args.mu).cuda(args.gpu)


    # infer learning rate before changing batch size
    init_lr = args.lr * args.batch_size / 256

    optimizer = torch.optim.SGD(model.parameters(), init_lr, momentum=args.momentum, weight_decay=args.weight_decay)
    cls_optimizer = torch.optim.SGD(classifier.parameters(), 0.1, momentum=args.momentum)
    
    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            logger.info("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            best_acc1 = checkpoint['best_acc1']
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            classifier.load_state_dict(checkpoint['classifier'])
            logger.info("=> loaded checkpoint '{}' (epoch {})".format(args.resume, checkpoint['epoch']))
        else:
            logger.info("=> no checkpoint found at '{}'".format(args.resume))


    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_loader.sampler.set_epoch(epoch)
        
        adjust_learning_rate(optimizer, init_lr, epoch, args)

        train(train_loader, model, classifier, criterion, optimizer, cls_optimizer, epoch, args, logger)

        if (epoch + 1) % args.val_freq == 0:
            # evaluate on validation set
            acc1 = validate(val_loader, model, classifier, args, logger)

            # remember best acc@1 and save checkpoint
            is_best = acc1 > best_acc1
            best_acc1 = max(acc1, best_acc1)

        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0):
            if (epoch + 1) % args.save_freq == 0:
                save_checkpoint({
                    'epoch': epoch + 1,
                    'arch': args.arch,
                    'state_dict': model.state_dict(),
                    'best_acc1': best_acc1,
                    'optimizer' : optimizer.state_dict(),
                    'classifier': classifier.state_dict(),
                }, is_best, filename=f'checkpoint-{epoch+1}.pth.tar', path=args.save)


def train(train_loader, model, classifier, criterion, optimizer, cls_optimizer, epoch, args, logger=None):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    cls_losses = AverageMeter('CLS_Loss', ':.4e')
    loss_part1 = AverageMeter('Part1', ':.4e')
    loss_part2 = AverageMeter('Part2', ':.4e')
    if args.version == 'v1':
        loss_part3 = AverageMeter('Part3', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    if args.version == 'v1':
        progress = ProgressMeter(
            len(train_loader),
            [batch_time, data_time, losses, loss_part1, loss_part2, loss_part3, cls_losses, top1, top5],
            prefix="Epoch: [{}]".format(epoch))
    else:
        progress = ProgressMeter(
            len(train_loader),
            [batch_time, data_time, losses, loss_part1, loss_part2, cls_losses, top1, top5],
            prefix="Epoch: [{}]".format(epoch))
    
    
    classifier.train()
    model.train()

    end = time.time()
    for i, (inps, target) in enumerate(train_loader):
        data_time.update(time.time() - end)


        x1, x2 = inps
        x1, x2 = x1.cuda(non_blocking=True), x2.cuda(non_blocking=True)
        target = target.cuda(non_blocking=True)

        q, k = model(x1, x2)
        if args.mode == 'ssl':
            loss, d_loss = criterion(q, k)
        elif args.mode in ['sup', 'lnl']:
            loss, d_loss = criterion(q, k, target)
        
        logits = classifier(q.detach())
        
        cls_loss = F.cross_entropy(logits, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(logits, target, topk=(1, 5))
        losses.update(loss.item(), x1.size(0))
        loss_part1.update(d_loss['part1'].item(), x1.size(0))
        loss_part2.update(d_loss['part2'].item(), x1.size(0))
        if args.version == 'v1':
            loss_part3.update(d_loss['part3'].item(), x1.size(0))

        cls_losses.update(cls_loss.item(), x1.size(0))
        top1.update(acc1[0], x1.size(0))
        top5.update(acc5[0], x1.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        cls_optimizer.zero_grad()
        cls_loss.backward()
        cls_optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            msg = progress.display(i)
            if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank == 0):
                logger.info(msg + '\t Best-Acc1 {:.4}'.format(best_acc1))


def validate(val_loader, model, classifier, args, logger=None):
    batch_time = AverageMeter('Time', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(val_loader),
        [batch_time, losses, top1, top5],
        prefix='Test: ')

    # switch to evaluate mode
    model.eval()
    classifier.eval()

    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            images = images.cuda(args.gpu, non_blocking=True)
            target = target.cuda(args.gpu, non_blocking=True)

            output = model(images, images)[0]

            logits = classifier(output)

            loss = F.cross_entropy(logits, target)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(logits, target, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                msg = progress.display(i)
                if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank == 0):
                    logger.info(msg)

        # TODO: this should also be done with the ProgressMeter
        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank == 0):
            logger.info(' * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'
              .format(top1=top1, top5=top5))

    return top1.avg


def adjust_learning_rate(optimizer, init_lr, epoch, args):
    """Decay the learning rate based on schedule"""
    cur_lr = init_lr * 0.5 * (1. + math.cos(math.pi * epoch / args.epochs))
    for param_group in optimizer.param_groups:
            param_group['lr'] = cur_lr


if __name__ == '__main__':
    main()