
import argparse
import builtins
import math
import os
import random
import shutil
import time
import warnings
import pprint
import logging
import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.datasets as datasets

import methods
import models
import torchvision.models as tvmodels
from datasets import *
from losses import *
from utils import *
from main import adjust_learning_rate, get_logger
from methods.utils import spectral_projector


model_names = ['preactresnet18', 'resnet18', 'resnet34', 'resnet50']


parser = argparse.ArgumentParser(description='PyTorch SSL for CIFAR')
parser.add_argument('-d', '--dataset', metavar='DATASET', default='domainnet')
parser.add_argument('--img-size', default=64, type=int, help='image size')
parser.add_argument('--data-path', metavar='DATAPATH', default='/home/zhouxiong/database/')
parser.add_argument('--source', default='clipart', type=str)
parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet50', choices=model_names,
                    help='model architecture: ' + ' | '.join(model_names) + ' (default: resnet18)')
parser.add_argument('--feat-dim', type=int, default=4096)
parser.add_argument('--hidden-dim', type=int, default=4096)
parser.add_argument('--need-projector', default=True, action='store_true')

parser.add_argument('-j', '--workers', default=16, type=int, metavar='N',
                    help='number of data loading workers (default: 32)')
parser.add_argument('--epochs', default=100, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=2048, type=int, metavar='N',
                    help='mini-batch size (default: 512), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lars', default=True, action='store_true', help='Use LARS')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial (base) learning rate', dest='lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-5, type=float, metavar='W', help='weight decay (default: 0.)', dest='weight_decay')

parser.add_argument('-p', '--print-freq', default=20, type=int, metavar='N', help='print frequency (default: 20)')
parser.add_argument('--resume', default='', type=str, metavar='PATH', help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true', help='evaluate model on validation set')
parser.add_argument('--world-size', default=-1, type=int, help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int, help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str, help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str, help='distributed backend')
parser.add_argument('--seed', default=0, type=int, help='seed for initializing training. ')
parser.add_argument('--gpu', default=0, type=int, help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')

# additional configs:
parser.add_argument('--pretrained', default='', type=str, help='path to simsiam pretrained checkpoint')
parser.add_argument('--save', default='', type=str, help='save path')
parser.add_argument('--save-freq', default=100, type=int, metavar='N', help='save frequency (default: 50)')
parser.add_argument('--val-freq', default=100, type=int, metavar='N', help='validation frequency (default: 5)')
parser.add_argument('--original', default=False, action='store_true')
parser.add_argument('--version', default='', type=str)


best_acc1 = 0.0


def get_local_dataloader(args, source):
    if args.dataset == 'domainnet':
        train_set = DomainNet(True, nori_prefix=source, spatial_size=args.img_size, unsupervised_transform=True)
    args.num_classes = 345
    sampler = None
    batch_size = args.batch_size
    if args.distributed:
        batch_size = batch_size // dist.get_world_size()
        sampler = torch.utils.data.distributed.DistributedSampler(train_set)
    dataloader = torch.utils.data.DataLoader(
        train_set, batch_size=batch_size, 
        num_workers=args.workers, sampler=sampler,
        shuffle=not args.distributed, drop_last=args.distributed,
        pin_memory=True, persistent_workers=True
    )
    return dataloader

def get_val_dataloader(args, target):
    val_set = DomainNet(False, nori_prefix=target, spatial_size=args.img_size, unsupervised_transform=False)
    args.num_classes = 345
    dataloader = torch.utils.data.DataLoader(
        val_set, batch_size=512, shuffle=False,
        num_workers=args.workers, pin_memory=True
    )
    return dataloader

class square_loss(nn.Module):
    def __init__(self):
        super().__init__()
    
    def forward(self, logits, labels):
        device = logits.device
        r = logits.size(1)
        label_one_hot = F.one_hot(labels, r).float().to(device)
        label_one_hot = label_one_hot - 1./ r * torch.ones_like(logits, device=device)
        loss = F.mse_loss(logits, label_one_hot)
        return loss

def main():
    global loger
    args = parser.parse_args()
    if args.save == '':
        args.save = '/'.join(args.pretrained.split('/')[:-1])
    args.save = os.path.join(args.save, 'uda+' + args.source)
    os.makedirs(args.save, exist_ok=True)

    logger = get_logger(args)
    logger.info('\n' + pprint.pformat(args))

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')
    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely disable data parallelism.')
    
    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])
    
    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)

def main_worker(gpu, ngpus_per_node, args):
    logger = get_logger(args)
    args.gpu = gpu
    # supress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:
        def print_pass(*args):
            pass
        builtins.print = print_pass
    
    if args.gpu is not None:
        logger.info("Use GPU: {} for training".format(args.gpu))
    
    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
        torch.distributed.barrier()
    
    # loading_data
    source_loader = get_local_dataloader(args, args.source)
    target_loaders = [get_val_dataloader(args, target) for target in ['clipart', 'painting', 'real', 'sketch']]
    # create model
    logger.info("==>creating model with '{}".format(args.arch))
    model = tvmodels.__dict__[args.arch](num_classes=args.num_classes)
    
    prev_dim = model.fc.weight.shape[1]
    if args.need_projector:
        model.fc = spectral_projector(prev_dim, args.feat_dim, args.hidden_dim)
        classifier = nn.Sequential(nn.Linear(args.feat_dim, args.num_classes, bias=False))
    else:
        model.fc = nn.Identity()
        classifier = nn.Sequential(nn.Linear(prev_dim, args.num_classes, bias=False))

    # load from pre-trained, before DistributedDataParallel constructor
    if args.pretrained:
        if os.path.isfile(args.pretrained):
            logger.info("=> loading checkpoint '{}".format(args.pretrained))
            checkpoint = torch.load(args.pretrained, map_location='cpu')

            # rename pre-trained keys
            state_dict = checkpoint['state_dict']

            for k in list(state_dict.keys()):
                flag = True
                if k.startswith('encoder'):
                    # remove prefix
                    state_dict[k[len("encoder."):]] = state_dict[k]
                elif k.startswith('module.encoder'):
                    state_dict[k[len("module.encoder."):]] = state_dict[k]
                elif k.startswith('online_encoder') and not k.startswith('online_encoder.fc'):
                    # remove prefix
                    state_dict[k[len("online_encoder."):]] = state_dict[k]
                elif k.startswith('module.encoder') and not k.startswith('module.encoder.fc'):
                    # remove prefix
                    state_dict[k[len("module.encoder."):]] = state_dict[k]
                elif k.startswith('module.online_encoder') and not k.startswith('module.online_encoder.fc'):
                    # remove prefix
                    state_dict[k[len("module.online_encoder."):]] = state_dict[k]
                else:
                    flag = False
                if flag: del state_dict[k]
            args.start_epoch = 0
            msg = model.load_state_dict(state_dict, strict=False)
            logger.info(msg.missing_keys)


            # classifier_state_dict = checkpoint['classifier']
            # for k in list(classifier_state_dict.keys()):
            #     if k.startswith('module'):
            #         classifier_state_dict[k[len("module."):]] = classifier_state_dict[k]
            #     del classifier_state_dict[k]

            # classifier.load_state_dict(classifier_state_dict)
            # assert set(msg.missing_keys) == {"fc.weight", "fc.bias"}

            logger.info("=> loaded pre-trained model '{}'".format(args.pretrained))
        else:
            logger.info("=> no checkpoint found at '{}'".format(args.pretrained))

    # infer learning rate before changing batch size
    init_lr = args.lr * args.batch_size / 256


    if args.distributed:
        # Apply SyncBN
        model = torch.nn.SyncBatchNorm.convert_sync_batchnorm(model)
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourves based on the total number of GPUs we have
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])

            classifier.cuda(args.gpu)
            classifier = torch.nn.parallel.DistributedDataParallel(classifier, device_ids=[args.gpu])

        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)

            classifier.cuda(args.gpu)
            classifier = torch.nn.parallel.DistributedDataParallel(classifier)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
        classifier = classifier.cuda(args.gpu)
    else:
        # AllGather implementation (batch shuffle, queue update, etc.) in
        # this code only supports DistributedDataParallel.
        raise NotImplementedError("Only DistributedDataParallel and one gpu is supported.")

    print(model) # print model after SyncBatchNorm

    # define loss function (criterion) and optimizer
    # criterion = nn.CrossEntropyLoss().cuda(args.gpu)
    criterion = nn.CrossEntropyLoss()
    # criterion = square_loss()

    optimizer = torch.optim.SGD(
        [
            # {'params': model.parameters()},
            {'params': classifier.parameters()}
        ],
        init_lr, momentum=args.momentum, weight_decay=args.weight_decay
    )

    if args.lars:
        print('=> Use LARS optimizer')
        from LARC import LARC
        optimizer = LARC(optimizer=optimizer, trust_coefficient=.001, clip=False)

    for epoch in range(args.start_epoch, args.epochs):
        source_loader.sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, init_lr, epoch, args)
        # train for one epoch
        train(source_loader, model, classifier, criterion, optimizer, epoch, args, logger)

        # evaluate on valiataion set
        if (epoch + 1) % args.val_freq ==0:
            acc1s = []
            for target_loader in target_loaders:
                acc1 = validate(target_loader, model, classifier, criterion, args, logger)
                acc1s.append(acc1)
            
            if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                    and args.rank == 0):
                logger.info('clipart: {}\t painting: {}\t real: {}\t sketch: {}'.format(acc1s[0], acc1s[1], acc1s[2], acc1s[3]))
        

def train(train_loader, model, classifier, criterion, optimizer, epoch, args, logger=None):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}]".format(epoch))
    
    """
    Switch to eval mode:
    Under the protocol of linear classification on frozen features/models,
    it is not legitimate to change any part of the pre-trained model.
    BatchNorm in train mode may revise running mean/std (even if it receives
    no gradient), which are part of the model parameters too.
    """
    # model.train()
    model.eval()

    end = time.time()
    for i, (images, target) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)
        images = images.cuda(args.gpu, non_blocking=True)
        target = target.cuda(args.gpu, non_blocking=True)

        # compute output
        output = classifier(model(images))
        loss = criterion(output, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            msg = progress.display(i)
            if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank == 0):
                logger.info(msg)

def validate(val_loader, model, classifier, criterion, args, logger=None):
    batch_time = AverageMeter('Time', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(val_loader),
        [batch_time, losses, top1, top5],
        prefix='Test: ')

    # switch to evaluate mode
    model.eval()

    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            images = images.cuda(args.gpu, non_blocking=True)
            target = target.cuda(args.gpu, non_blocking=True)

            # compute output
            output = classifier(model(images))
            loss = criterion(output, target)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(output, target, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                progress.display(i)

    return top1.avg


if __name__ == '__main__':
    main()