#!/usr/bin/env python3

"""
Script to extract concept descriptions and examples from the demonstrations module
and generate detailed documentation.
"""

import os
import sys
import inspect
from importlib import import_module

# Add the parent directory to the path so we can import frame modules
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), "..")))

OUTPUT_DIR = "source/frame/knowledge_base"
os.makedirs(OUTPUT_DIR, exist_ok=True)


def generate_demonstrations_docs():
    """Generate documentation for the demonstrations module."""
    # Import the demonstrations module
    try:
        demos = import_module("frame.knowledge_base.demonstrations")
    except ImportError:
        print("Error: Could not import demonstrations module")
        sys.exit(1)

    # Get all concepts from the module
    concepts = {}

    # Inspect the module for concepts
    for name, obj in inspect.getmembers(demos):
        # Skip private/special members and functions
        if name.startswith("_") or inspect.isfunction(obj) or inspect.ismodule(obj):
            continue

        # Check if it's a Concept instance
        if hasattr(obj, "name") and hasattr(obj, "description"):
            # Skip imported concepts from other modules
            if getattr(obj, "is_imported", False):
                continue

            # Group by category using comments in the module
            category = "Other Concepts"

            if hasattr(obj, "example_structure") and hasattr(
                obj.example_structure, "concept_type"
            ):
                concept_type = str(obj.example_structure.concept_type).split(".")[-1]
                if concept_type not in concepts:
                    concepts[concept_type] = []
                concepts[concept_type].append((name, obj))
            else:
                if category not in concepts:
                    concepts[category] = []
                concepts[category].append((name, obj))

    # Get module docstring for overview
    module_doc = inspect.getdoc(demos) or "No module documentation available."

    # Generate RST content
    rst_content = f"""Demonstrations
==============

{module_doc}

.. contents:: Table of Contents
   :local:
   :depth: 2

"""

    # Add each concept by category
    for category, concept_list in concepts.items():
        # Skip empty categories
        if not concept_list:
            continue

        rst_content += f"\n{category}\n" + "-" * len(category) + "\n\n"

        for i, (name, concept) in enumerate(sorted(concept_list, key=lambda x: x[0])):
            rst_content += f"``{name}``\n" + "~" * (len(name) + 4) + "\n\n"
            rst_content += f"**Description**: {concept.description}\n\n"

            # Add component types if available
            if hasattr(concept, "example_structure") and hasattr(
                concept.example_structure, "component_types"
            ):
                component_types = concept.example_structure.component_types
                if component_types:
                    types_str = ", ".join(
                        [str(t).split(".")[-1] for t in component_types]
                    )
                    rst_content += f"**Types**: {types_str}\n\n"

            # Add arity if available
            if hasattr(concept, "example_structure") and hasattr(
                concept.example_structure, "input_arity"
            ):
                rst_content += f"**Arity**: {concept.example_structure.input_arity}\n\n"

            # Add symbolic definition if available
            if hasattr(concept, "symbolic_definition"):
                rst_content += "This concept is formally defined in the FRAME expression language.\n\n"

            # Show computational implementation availability
            if hasattr(concept, "computational_implementation"):
                rst_content += "The concept has a computational implementation that can execute on concrete values.\n\n"

            # Only add separation line if not the last item
            if i < len(concept_list) - 1:
                rst_content += "----\n\n"

    # Write the RST file
    output_file = os.path.join(OUTPUT_DIR, "demonstrations.rst")
    with open(output_file, "w") as f:
        f.write(rst_content)

    print(f"Generated: {output_file}")
    return output_file


if __name__ == "__main__":
    generate_demonstrations_docs()
