#!/usr/bin/env python
"""
Script to automatically generate RST documentation files for all interestingness modules
and update the main interestingness.rst file accordingly.
"""

import os
import glob

# Paths
DOCS_DIR = "source/frame"
INTERESTINGNESS_DIR = "../frame/interestingness"
RST_OUTPUT_DIR = f"{DOCS_DIR}/interestingness"


def ensure_dir(directory):
    """Create directory if it doesn't exist."""
    if not os.path.exists(directory):
        os.makedirs(directory)


def get_module_names(directory, exclude_init=True):
    """Get module names from a directory."""
    modules = glob.glob(f"{directory}/*.py")
    if exclude_init:
        modules = [m for m in modules if not os.path.basename(m).startswith("__")]
    return sorted([os.path.splitext(os.path.basename(m))[0] for m in modules])


def create_interestingness_rst(module_name, output_dir):
    """Create an RST file for an interestingness module."""
    # Convert module_name to a title case for display
    display_name = " ".join(word.capitalize() for word in module_name.split("_"))

    rst_content = f"""{display_name} Interestingness Metric
{'=' * (len(display_name) + 25)}

.. automodule:: frame.interestingness.{module_name}
   :members:
   :undoc-members:
   :show-inheritance:
"""

    # Write the RST file
    output_file = f"{output_dir}/{module_name}.rst"
    with open(output_file, "w") as f:
        f.write(rst_content)

    print(f"Created: {output_file}")


def update_interestingness_rst():
    """Update the main interestingness.rst file."""
    # Get module names
    interest_modules = get_module_names(INTERESTINGNESS_DIR)

    # Build the interestingness.rst content
    content = """Interestingness Metrics
======================

FRAME interestingness metrics are used to evaluate the quality and importance of concepts and conjectures.

.. toctree::
   :maxdepth: 1
   :caption: Contents:

"""

    # Add module paths
    for module in interest_modules:
        content += f"   interestingness/{module}\n"

    # Write the updated interestingness.rst file
    with open(f"{DOCS_DIR}/interestingness.rst", "w") as f:
        f.write(content)

    print(f"Updated: {DOCS_DIR}/interestingness.rst")


def main():
    """Main entry point."""
    # Ensure directory exists
    ensure_dir(RST_OUTPUT_DIR)

    # Generate RST files for interestingness modules
    for module in get_module_names(INTERESTINGNESS_DIR):
        create_interestingness_rst(module, RST_OUTPUT_DIR)

    # Update the main interestingness.rst file
    update_interestingness_rst()

    print("\nDone! Interestingness documentation has been updated.")


if __name__ == "__main__":
    main()
