from typing import Dict, Set, Optional, Callable, Any, Tuple, List
from dataclasses import dataclass
from enum import Enum
import re

from frame.environments.ground_truth_types import GroundTruthEntity, EntityType, is_prime

NT_CALCULATIONS = {
    # add ==============================================================
    "add_zero": GroundTruthEntity(
        canonical_name="add_zero",
        entity_type=EntityType.CONCEPT,
        description="Addition of zero",
        computational_implementation=lambda a: a + 0,
        discovered_names={"specialized_(add_at_0_to_zero)", "specialized_(add_at_1_to_zero)"}
    ),
    "add_one": GroundTruthEntity(
        canonical_name="add_one",
        entity_type=EntityType.CONCEPT,
        description="Addition of one",
        computational_implementation=lambda a: a + 1,
        discovered_names={"specialized_(add_at_0_to_one)", "specialized_(add_at_1_to_one)"}
    ),
    "add_two": GroundTruthEntity(
        canonical_name="add_two",
        entity_type=EntityType.CONCEPT,
        description="Addition of two",
        computational_implementation=lambda a: a + 2,
        discovered_names={"specialized_(add_at_0_to_two)", "specialized_(add_at_1_to_two)"}
    ),
    "add_with_doubled_summand": GroundTruthEntity(
        canonical_name='add_with_doubled_summand',
        entity_type=EntityType.CONCEPT,
        description='Add with first summand doubled',
        computational_implementation=lambda a, b: a*2 + b,
        discovered_names={'compose_(double_with_add_output_to_input_map={0: 0})',
                          'compose_(double_with_add_output_to_input_map={0: 1})'}
    ),
    "sum_of_doubles": GroundTruthEntity(
        canonical_name='sum_of_doubles',
        entity_type=EntityType.CONCEPT,
        description='Add with both summands doubled',
        computational_implementation=lambda a, b: a*2 + b*2,
        discovered_names={'compose_(double_with_add_with_doubled_summand_output_to_input_map={0: 1})'}
    ),
    "add_with_doubled_incr_summand": GroundTruthEntity(
        canonical_name='add_with_doubled_incr_summand',
        entity_type=EntityType.CONCEPT,
        description='Add with first summand doubled + 1',
        computational_implementation=lambda a, b: a*2 + 1 + b,
        discovered_names={'compose_(double_add_one_with_add_output_to_input_map={0: 0})',
                          'compose_(double_add_one_with_add_output_to_input_map={0: 1})'}
    ),
    "sum_of_doubles_incr": GroundTruthEntity(
        canonical_name='sum_of_doubles_incr',
        entity_type=EntityType.CONCEPT,
        description='Add with both summands doubled + 1',
        computational_implementation=lambda a, b: a*2 + b*2 + 2,
        discovered_names={'compose_(double_add_one_with_add_with_doubled_incr_summand_output_to_input_map={0: 1})'}
    ),
    "sum_of_even_odd": GroundTruthEntity(
        canonical_name='sum_of_even_odd',
        entity_type=EntityType.CONCEPT,
        description='Add with one summand doubled, one summand doubled and incr.',
        computational_implementation=lambda a, b: a*2 + 1 + b*2 , # Note(_; 5/6): not accurate to pr
        discovered_names={'compose_(double_add_one_with_add_with_doubled_summand_output_to_input_map={0: 1})',
                          'compose_(double_with_add_with_doubled_incr_summand_output_to_input_map={0: 1})'}
    ),
    # multiply ==============================================================
    "multiply_zero": GroundTruthEntity(
        canonical_name="multiply_zero",
        entity_type=EntityType.CONCEPT,
        description="Multiplication of zero",
        computational_implementation=lambda a: a * 0,
        discovered_names={"specialized_(multiply_at_0_to_zero)", "specialized_(multiply_at_1_to_zero)"}
    ),
    "multiply_one": GroundTruthEntity(
        canonical_name="multiply_one",
        entity_type=EntityType.CONCEPT,
        description="Multiplication of one",
        computational_implementation=lambda a: a * 1,
        discovered_names={"specialized_(multiply_at_0_to_one)", "specialized_(multiply_at_1_to_one)"}
    ),
    "double_add_one": GroundTruthEntity(
        canonical_name="double_add_one",
        entity_type=EntityType.CONCEPT,
        description="Double and add one",
        computational_implementation=lambda a: 2*a + 1,
        discovered_names={"compose_(double_with_add_one_output_to_input_map={0: 0})"}
    ),
    "multiply_with_doubled_factor": GroundTruthEntity(
        canonical_name='multiply_with_doubled_factor',
        entity_type=EntityType.CONCEPT,
        description='multiply with first factor doubled ',
        computational_implementation=lambda a, b: (a*2) * b,
        discovered_names={'compose_(double_with_multiply_output_to_input_map={0: 0})',
                          'compose_(double_with_multiply_output_to_input_map={0: 1})'}
    ),
    "multiply_with_doubled_incr_factor": GroundTruthEntity(
        canonical_name='multiply_with_doubled_incr_factor',
        entity_type=EntityType.CONCEPT,
        description='multiply with first factor doubled + 1',
        computational_implementation=lambda a, b: (a*2 + 1) * b,
        discovered_names={'compose_(double_add_one_with_multiply_output_to_input_map={0: 0})',
                          'compose_(double_add_one_with_multiply_output_to_input_map={0: 1})'}
    ),
    "product_of_doubles_incr": GroundTruthEntity(
        canonical_name='product_of_doubles_incr',
        entity_type=EntityType.CONCEPT,
        description='multiply with both factor doubled + 1',
        computational_implementation=lambda a, b: (a*2 + 1) * (b*2 + 1),
        discovered_names={'compose_(double_add_one_with_multiply_with_doubled_incr_factor_output_to_input_map={0: 1})'}
    ),
    # powers ===========================================================
    "power_zero": GroundTruthEntity(
        canonical_name="power_zero",
        entity_type=EntityType.CONCEPT,
        description="Power of zero",
        computational_implementation=lambda a: a ** 0,
        discovered_names={"specialized_(power_at_1_to_zero)"}
    ),
    "power_one": GroundTruthEntity(
        canonical_name="power_one",
        entity_type=EntityType.CONCEPT,
        description="Power of one",
        computational_implementation=lambda a: a ** 1,
        discovered_names={"specialized_(power_at_1_to_one)"}
    ),
    "add_one_square": GroundTruthEntity(
        canonical_name="add_one_square",
        entity_type=EntityType.CONCEPT,
        description='add one then square',
        computational_implementation=lambda a: (a+1)**2,
        discovered_names={"compose_(add_one_with_square_output_to_input_map={0: 0})"}
    )      
}

# Note(_; 5/6): all operations/calcs composed with eq. all of these should have 2 discovered names
NT_EQ_CALCULATIONS = {
    "eq_sum": GroundTruthEntity(
        canonical_name="eq_sum",
        entity_type=EntityType.CONCEPT,
        description='a + b == c',
        computational_implementation=lambda a,b,c: a+b==c,
        discovered_names={"compose_(add_with_eq_output_to_input_map={0: 0})",
                         "compose_(add_with_eq_output_to_input_map={0: 1})" }
    ),
    "eq_sum_three_numbers": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers",
        entity_type=EntityType.CONCEPT,
        description='a + b + c == d',
        computational_implementation=lambda a,b,c,d: a+b+c==d,
        discovered_names={"compose_(add_three_numbers_with_eq_output_to_input_map={0: 0})",
                         "compose_(add_three_numbers_with_eq_output_to_input_map={0: 1})"}
    ),
    "eq_sum_of_doubles": GroundTruthEntity(
        canonical_name="eq_sum_of_doubles",
        entity_type=EntityType.CONCEPT,
        description='final arg is sum of doubles of first two',
        computational_implementation=lambda a,b,c: a*2 + b*2 == c,
        discovered_names={"compose_(sum_of_doubles_with_eq_output_to_input_map={0: 0})",
                          "compose_(sum_of_doubles_with_eq_output_to_input_map={0: 1})"}
    ),
    "eq_sum_of_doubles_incr": GroundTruthEntity(
        canonical_name="eq_sum_of_doubles_incr",
        entity_type=EntityType.CONCEPT,
        description='final arg is sum of doubles of first two incremented',
        computational_implementation=lambda a,b,c: a*2 + b*2 == c,
        discovered_names={"compose_(sum_of_doubles_incr_with_eq_output_to_input_map={0: 0})",
                          "compose_(sum_of_doubles_incr_with_eq_output_to_input_map={0: 1})"}
    ),
    "eq_product": GroundTruthEntity(
        canonical_name="eq_product",
        entity_type=EntityType.CONCEPT,
        description='a*b == c',
        computational_implementation=lambda a,b,c: a*b==c,
        discovered_names={"compose_(multiply_with_eq_output_to_input_map={0: 0})",
                         "compose_(multiply_with_eq_output_to_input_map={0: 1})" }
    ),
    "eq_product_doubled_factor": GroundTruthEntity(
        canonical_name="eq_product_doubled_factor",
        entity_type=EntityType.CONCEPT,
        description='2a*b == c',
        computational_implementation=lambda a,b,c: 2*a*b==c,
        discovered_names={"compose_(multiply_with_doubled_factor_with_eq_output_to_input_map={0: 0})",
                         "compose_(multiply_with_doubled_factor_with_eq_output_to_input_map={0: 1})" }
    ),
    "eq_product_doubled_incr_factors": GroundTruthEntity(
        canonical_name="eq_product_doubled_incr_factors",
        entity_type=EntityType.CONCEPT,
        description='eq_product_doubled_incr_factors',
        computational_implementation=lambda a,b,c: (2*a + 1) * (2*b + 1) == c,
        discovered_names={"compose_(product_of_doubles_incr_with_eq_output_to_input_map={0: 0})",
                         "compose_(product_of_doubles_incr_with_eq_output_to_input_map={0: 1})" }
    ),
    "eq_power": GroundTruthEntity(
      canonical_name="eq_power",
      entity_type=EntityType.CONCEPT,
      description="power is equal to final arg",
      computational_implementation=lambda a,b,c: a**b == c,
      discovered_names={"compose_(power_with_eq_output_to_input_map={0: 0})",
                        "compose_(power_with_eq_output_to_input_map={0: 1})"}
    ),
    "eq_sum_two_powers": GroundTruthEntity(
      canonical_name="eq_sum_two_powers",
      entity_type=EntityType.CONCEPT,
      description="Sum of the two powers is equal to final arg",
      computational_implementation=lambda a,b,c,d,n: a**b + c**d == n,
      discovered_names={"compose_(sum_two_powers_with_eq_output_to_input_map={0: 0})",
                        "compose_(sum_two_powers_with_eq_output_to_input_map={0: 1})"}
    ),
    "eq_product_add": GroundTruthEntity(
        canonical_name="eq_product_add",
        entity_type=EntityType.CONCEPT,
        description='a*q + r == b',
        computational_implementation=lambda a, q, r, b: a*q + r ==b,
        discovered_names={"compose_(product_add_with_eq_output_to_input_map={0: 0})",
                          "compose_(product_add_with_eq_output_to_input_map={0: 1})"}
    ),
}

NT_PREDICATE_CALCULATIONS = {
    # concept manipulations
    # sums =========================================================================
    "is_sum_of_evens": GroundTruthEntity(
        canonical_name='is_sum_of_evens',
        entity_type=EntityType.CONCEPT,
        description='Can be written as sum of two evens',
        computational_implementation=lambda a: a % 2 == 0, 
        discovered_names={'exists_(sum_of_doubles_indices_[0, 1])',
                          'exists_(eq_sum_of_doubles_indices_[0, 1])'}
    ),
    "is_sum_of_odds": GroundTruthEntity(
        canonical_name='is_sum_of_odds',
        entity_type=EntityType.CONCEPT,
        description='Can be written as sum of two odds',
        computational_implementation=lambda a: a % 2 == 0,
        discovered_names={'exists_(sum_of_doubles_incr_indices_[0, 1])',
                          'exists_(eq_sum_of_doubles_incr_indices_[0, 1])'}
    ),
    "is_sum_of_even_odd": GroundTruthEntity(
        canonical_name='is_sum_of_even_odd',
        entity_type=EntityType.CONCEPT,
        description='Can be written as sum of an even and an odd',
        computational_implementation=lambda a: a % 2 == 0, 
        discovered_names={'exists_(sum_of_even_odd_indices_[0, 1])'}
    ),
    # primes =========================================================================
    # -- no prime square --
    "is_prime_square": GroundTruthEntity(
        canonical_name="is_prime_square",
        entity_type=EntityType.CONCEPT,
        description='Is prime and square',
        computational_implementation=lambda n: False,
        discovered_names={"compose_(is_prime_with_is_square_shared_vars={0: 0})",
                          "compose_(is_square_with_is_prime_shared_vars={0: 0})"}
    ),
    # ---
    # -- infinitely many primes --
    "is_prime_gt": GroundTruthEntity(
        canonical_name="is_prime_gt",
        entity_type=EntityType.CONCEPT,
        description='a > b and a is prime',
        computational_implementation=lambda a, b: a > b and is_prime(a),
        discovered_names={"compose_(gt_with_is_prime_shared_vars={0: 0})",
                          "compose_(is_prime_with_gt_shared_vars={0: 0})",
                          "compose_(is_prime_with_lt_shared_vars={0: 1})"}
    ),
    "lt_is_prime": GroundTruthEntity(
        canonical_name="lt_is_prime",
        entity_type=EntityType.CONCEPT,
        description='a < b and b is prime',
        computational_implementation=lambda a, b: a < b and is_prime(b),
        discovered_names={"compose_(lt_with_is_prime_shared_vars={1: 0})"}
    ),    
    # ---
    # -- twin prime conjecture --
    "is_prime_minus_two": GroundTruthEntity(
        canonical_name="is_prime_minus_two",
        entity_type=EntityType.CONCEPT,
        description='Result of adding two is prime',
        computational_implementation=lambda n: is_prime(n+2),
        discovered_names={"compose_(add_two_with_is_prime_output_to_input_map={0: 0})"}
    ),
    "is_twin_prime": GroundTruthEntity(
        canonical_name="is_twin_prime",
        entity_type=EntityType.CONCEPT,
        description='n and n+2 is prime',
        computational_implementation=lambda n: is_prime(n) and is_prime(n+2),
        discovered_names={"compose_(is_prime_minus_two_with_is_prime_shared_vars={0: 0})",
                          "compose_(is_prime_with_is_prime_minus_two_shared_vars={0: 0})"}
    ),    
    "is_twin_prime_gt": GroundTruthEntity(
        canonical_name="is_twin_prime_gt",
        entity_type=EntityType.CONCEPT,
        description='a and a+2 is prime and a >b',
        computational_implementation=lambda m, n: m > n and is_prime(m) and is_prime(m+2),
        discovered_names={"compose_(gt_with_is_twin_prime_shared_vars={0: 0})",
                          "compose_(is_twin_prime_with_gt_shared_vars={0: 0})",
                          "compose_(is_twin_prime_with_lt_shared_vars={0: 1})"}
    ),
    "lt_is_twin_prime": GroundTruthEntity(
        canonical_name="lt_is_twin_prime",
        entity_type=EntityType.CONCEPT,
        description='b and b+2 is prime and a < b',
        computational_implementation=lambda m, n: m < n and is_prime(n) and is_prime(n+2),
        discovered_names={"compose_(lt_with_is_twin_prime_shared_vars={1: 0})"}
    ),      
    # ---
    # -- goldbach strong --
    "eq_sum_first_summand_prime": GroundTruthEntity(
        canonical_name="eq_sum_first_summand_prime",
        entity_type=EntityType.CONCEPT,
        description='a + b == c and a is prime',
        computational_implementation=lambda a, b,c: is_prime(a) and a+b==c,
        discovered_names={"compose_(is_prime_with_eq_sum_shared_vars={0: 0})",
                          "compose_(is_prime_with_eq_sum_shared_vars={0: 1})"}
    ),
    "eq_sum_of_primes": GroundTruthEntity(
        canonical_name="eq_sum_of_primes",
        entity_type=EntityType.CONCEPT,
        description='a + b == c and a and b are prime',
        computational_implementation=lambda a,b,c: is_prime(a) and is_prime(b) and a+b==c,
        discovered_names={"compose_(is_prime_with_eq_sum_first_summand_prime_shared_vars={0: 1})",
                          "compose_(eq_sum_first_summand_prime_is_prime_shared_vars={1: 0})"
                          }
    ),
    "goldbach_strong_q": GroundTruthEntity(
        canonical_name="goldbach_strong_q",
        entity_type=EntityType.CONCEPT,
        description='for a given number there exist two primes that sum to it',
        computational_implementation=lambda a: a %2 == 0, #TODO(_; 5/6): include??
        discovered_names={"exists_(eq_sum_of_primes_indices_[0, 1])"}
    ),
    # -- goldbach weak -- 
    "eq_sum_three_numbers_prime_summand1": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers_prime_summand1",
        entity_type=EntityType.CONCEPT,
        description='a + b + c == d and a is prime',
        computational_implementation=lambda a,b,c,d: is_prime(a) and a+b+c==d,
        discovered_names={"compose_(is_prime_with_eq_sum_three_numbers_shared_vars={0: 0})",
                          "compose_(is_prime_with_eq_sum_three_numbers_shared_vars={0: 1})", 
                          "compose_(is_prime_with_eq_sum_three_numbers_shared_vars={0: 2})"}
    ),
    "eq_sum_three_numbers_prime_summand1_prime_summand2": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers_prime_summand1_prime_summand2",
        entity_type=EntityType.CONCEPT,
        description='a + b + c == d and a is prime and b is prime',
        computational_implementation=lambda a,b,c,d: is_prime(a) and is_prime(b) and a+b+c==d,
        discovered_names={"compose_(is_prime_with_eq_sum_three_numbers_prime_summand1_shared_vars={0: 1})", 
                          "compose_(is_prime_with_eq_sum_three_numbers_prime_summand1_shared_vars={0: 2})"}
    ),
    "eq_sum_three_primes": GroundTruthEntity(
        canonical_name="eq_sum_three_primes",
        entity_type=EntityType.CONCEPT,
        description='a + b + c == d and a, b, c are prime',
        computational_implementation=lambda a,b,c,d: is_prime(a) and is_prime(b) and is_prime(c) and a+b+c==d,
        discovered_names={"compose_(is_prime_with_eq_sum_three_numbers_prime_summand1_prime_summand2_shared_vars={0: 2})"}
    ),
    "goldbach_weak_q": GroundTruthEntity(
        canonical_name="goldbach_weak_q",
        entity_type=EntityType.CONCEPT,
        description='for all numbers bigger than 5 there exists three primes that sum to it',
        computational_implementation=lambda d: True, #Note(_; 5/5) include??
        discovered_names={"exists_(eq_sum_three_primes_indices_[0, 1, 2])"}
    ),
    # ---
    # -- bertrand postulate -- 
    "double_gt": GroundTruthEntity(
        canonical_name="double_gt",
        entity_type=EntityType.CONCEPT,
        description='first input doubled gt second input',
        computational_implementation=lambda a, b: a*2 > b,
        discovered_names={"compose_(double_with_lt_output_to_input_map={0: 1})",
                          "compose_(double_with_gt_output_to_input_map={0: 0})"
                        }
    ),
    # --   
    "gt_and_lt_double": GroundTruthEntity(
        canonical_name="gt_and_lt_double",
        entity_type=EntityType.CONCEPT,
        description='b < a < 2*b',
        computational_implementation=lambda a, b: a >b and 2*b >a,
        discovered_names={"compose_(gt_with_double_gt_shared_vars={0: 1, 1: 0})"}
    ),
    "lt_and_double_gt": GroundTruthEntity(
        canonical_name="lt_and_double_gt",
        entity_type=EntityType.CONCEPT,
        description='a < b < 2*a',
        computational_implementation=lambda a, b: a < b and 2*a > b,
        discovered_names={"compose_(lt_with_double_gt_shared_vars={0: 0, 1: 1})",
                          "compose_(double_gt_with_lt_shared_vars={0: 0, 1: 1})",
                          "compose_(double_gt_with_gt_shared_vars={1: 0, 0: 1})"} 
    ),
    # -- 
    "is_prime_gt_and_lt_double": GroundTruthEntity(
        canonical_name="is_prime_gt_and_lt_double",
        entity_type=EntityType.CONCEPT,
        description='b < a < 2*b and a is prime',
        computational_implementation=lambda a, b: is_prime(a) and b < a and b < 2*a, 
        discovered_names={"compose_(gt_and_lt_double_with_is_prime_shared_vars={0: 0})",
                          "compose_(is_prime_with_gt_and_lt_double_shared_vars={0: 0})",
                          "compose_(is_prime_with_lt_and_double_gt_shared_vars={0: 1})"} #Note(_; 5/6): compose switches order
    ),
    "is_prime_lt_and_double_gt": GroundTruthEntity(
        canonical_name="is_prime_lt_and_double_gt",
        entity_type=EntityType.CONCEPT,
        description='a < b < 2*a and b is prime',
        computational_implementation=lambda a, b: is_prime(a) and b < a and b < 2*a, 
        discovered_names={"compose_(lt_and_double_gt_with_is_prime_shared_vars={1: 0})"}
    ),
    # -- 
    "bertrand_postulate_q": GroundTruthEntity(
        canonical_name="bertrand_postulate_q",
        entity_type=EntityType.CONCEPT,
        description='exists b s.t. a < b < 2*a and b is prime',
        computational_implementation=lambda a: a > 1,
        discovered_names={"exists_(is_prime_gt_and_lt_double_indices_[0])",
                          "exists_(is_prime_lt_and_double_gt_indices_[1])"}
    ),
    # ----
    # -- legendre conjecture --
    "square_lt": GroundTruthEntity(
        canonical_name="square_lt",
        entity_type=EntityType.CONCEPT,
        description='a^2 < b',
        computational_implementation=lambda a, b: b > a**2,
        discovered_names={"compose_(square_with_gt_output_to_input_map={0: 1})",
                          "compose_(square_with_lt_output_to_input_map={0: 0})"}
    ),
    "add_one_square_gt": GroundTruthEntity(
        canonical_name="add_one_square_gt",
        entity_type=EntityType.CONCEPT,
        description='(a+1)^2 > b',
        computational_implementation=lambda a, b: b < (a+1)**2,
        discovered_names={"compose_(add_one_square_with_lt_output_to_input_map={0: 1})",
                          "compose_(add_one_square_with_gt_output_to_input_map={0: 0})"}
    ),
    # --
    "square_lt_and_add_one_square_gt": GroundTruthEntity(
        canonical_name="square_lt_and_add_one_square_gt",
        entity_type=EntityType.CONCEPT,
        description='a^2 < b < (a+1)^2',
        computational_implementation=lambda a,b: a**2 < b and (a+1)**2 > b,
        discovered_names={"compose_(add_one_square_gt_with_square_lt_shared_vars={0: 0, 1: 1})",
                          "compose_(square_lt_with_add_one_square_gt_shared_vars={0: 0, 1: 1})"}
    ),
    # --
    "is_prime_square_lt_and_add_one_square_gt": GroundTruthEntity(
        canonical_name="is_prime_square_lt_and_add_one_square_gt",
        entity_type=EntityType.CONCEPT,
        description='a^2 < b < (a+1)^2 and b is prime',
        computational_implementation=lambda a,b: a**2 < b and (a+1)**2 > b and is_prime(b),
        discovered_names={"compose_(square_lt_and_add_one_square_gt_with_is_prime_shared_vars={1: 0})"}
    ),
    "is_prime_gt_square_and_lt_add_one_square": GroundTruthEntity(
        canonical_name="is_prime_gt_square_and_lt_add_one_square",
        entity_type=EntityType.CONCEPT,
        description='b^2 < a < (b+1)^2 and a is prime',
        computational_implementation=lambda a,b: b**2 < a and (b+1)**2 > a and is_prime(a),
        discovered_names={"compose_(is_prime_with_square_lt_and_add_one_square_gt_shared_vars={0: 1})"}
    ),
    # --
    "legendre_conjecture_q": GroundTruthEntity(
        canonical_name="legendre_conjecture_q",
        entity_type=EntityType.CONCEPT,
        description='exists p is prime, greater than square of arg and less than square of arg + 1',
        computational_implementation=lambda n:n>0,
        discovered_names={"exists_(is_prime_gt_square_and_lt_add_one_square_indices_[0])",
                          "exists_(is_prime_square_lt_and_add_one_square_gt_indices_[1])"}
    ),
    # ----
    # -- fermat's little thm
    "is_mod_power": GroundTruthEntity(
        canonical_name="is_mod_power",
        entity_type=EntityType.CONCEPT,
        description='take the power, then check if eq to mod',
        computational_implementation=lambda a,b,c,d: a**b >= d and a**b % c == d,
        discovered_names={"compose_(power_with_is_mod_output_to_input_map={0: 2})"}
    ),
    # --
    "is_mod_power_matched_base_res": GroundTruthEntity(
        canonical_name="is_mod_power_matched_base_res",
        entity_type=EntityType.CONCEPT,
        description='lambda a,b,c: a**b % c == a',
        computational_implementation=lambda a,b,c: a**b % c == a,
        discovered_names={"matched_(is_mod_power_indices_[0, 3])"}
    ),
    "is_mod_power_matched_pwr_mod": GroundTruthEntity(
        canonical_name="is_mod_power_matched_pwr_mod",
        entity_type=EntityType.CONCEPT,
        description='lambda a,b,c: a**b % b == c',
        computational_implementation=lambda a,b,c: a**b >= c and a**b % b == c,
        discovered_names={"matched_(is_mod_power_indices_[1, 2])"}
    ),
    # --
    "is_mod_power_matched_base_res_pwr_mod": GroundTruthEntity(
        canonical_name="is_mod_power_matched_base_res_pwr_mod",
        entity_type=EntityType.CONCEPT,
        description='lambda a,b: a**b % b == a',
        computational_implementation=lambda a,b: a**b % b == a,
        discovered_names={"matched_(is_mod_power_matched_base_res_indices_[1, 2])",
                          "matched_(is_mod_power_matched_pwr_mod_indices_[0, 2])"}
    ),
    # --


    # -- 2 is only even prime --
    "is_even_prime": GroundTruthEntity(
        canonical_name="is_even_prime",
        entity_type=EntityType.CONCEPT,
        description='Is prime and even',
        computational_implementation=lambda n: n == 2,
        discovered_names={"compose_(is_even_with_is_prime_shared_vars={0: 0})",
                          "compose_(is_prime_with_is_even_shared_vars={0: 0})"}
    ),
    "is_prime_ne_two": GroundTruthEntity(
        canonical_name="is_prime_ne_two",
        entity_type=EntityType.CONCEPT,
        description='Is prime and not two',
        computational_implementation=lambda n: n != 2 and is_prime(n),
        discovered_names={"compose_(ne_two_with_is_prime_shared_vars={0: 0})",
                          "compose_(is_prime_with_ne_two_shared_vars={0: 0})"}
    ),
    # divides =========================================================================
    # -- divides reflexivity --
    "divides_self": GroundTruthEntity(
        canonical_name='divides_self',
        entity_type=EntityType.CONCEPT,
        description='n | n',
        computational_implementation=lambda a: a != 0,
        discovered_names={"matched_(divides_indices_[0, 1])"}
    ),
    # -- all positives divisor of zero --
    "is_divisor_of_zero": GroundTruthEntity(
        canonical_name="is_divisor_of_zero",
        entity_type=EntityType.CONCEPT,
        description="n | 0",
        computational_implementation=lambda a: a != 0,
        discovered_names={"specialized_(divides_at_1_to_zero)"}
    ),
    # -- divides each other equals --
    "divides_each_other": GroundTruthEntity(
        canonical_name="divides_each_other",
        entity_type=EntityType.CONCEPT,
        description='a | b ^ b | a',
        computational_implementation=lambda a, b: a == b,
        discovered_names={"compose_(divides_with_divides_shared_vars={0: 1, 1: 0})",
                          "compose_(divides_with_divides_shared_vars={1: 0, 0: 1})"}
    ),
    # --
    # -- divisor leq multiple --
    "divides_and_multiple_gt_zero": GroundTruthEntity(
        canonical_name='divides_and_multiple_gt_zero',
        entity_type=EntityType.CONCEPT,
        description='n | m and m != 0',
        computational_implementation=lambda a, b: a!=0 and b!=0 and b%a==0,
        discovered_names={"compose_(divides_with_gt_zero_shared_vars={1: 0})"}
    ),
    "multiple_gt_zero_and_divides": GroundTruthEntity(
        canonical_name="multiple_gt_zero_and_divides",
        entity_type=EntityType.CONCEPT,
        description='n | m and m != 0',
        computational_implementation=lambda b, a: a!=0 and b!=0 and b%a==0,
        discovered_names={"compose_(gt_zero_with_divides_shared_vars={0: 1})"}
    ),
    # ---
    # -- common divisor two numbers --
    "divides_both_numbers": GroundTruthEntity(
        canonical_name="divides_both_numbers",
        entity_type=EntityType.CONCEPT,
        description='first arg divides both second and third arg',
        computational_implementation=lambda x, y, z: x != 0 and y%x==0 and z%x==0,
        discovered_names={"compose_(divides_with_divides_shared_vars={0: 0})"}
    ),
    "divides_sum_two_products": GroundTruthEntity(
        canonical_name="divides_sum_two_products",
        entity_type=EntityType.CONCEPT,
        description='Sum the products of two pairs of numbers',
        computational_implementation=lambda a,y,b,z,x: x != 0 and (a*y + b*z) % x == 0,
        discovered_names={"compose_(sum_two_products_with_divides_output_to_input_map={0: 1})"}
    ),
    # ---
    # -- squares have odd number divisors / non-squares have even number divisors
    "odd_number_divisors": GroundTruthEntity(
        canonical_name="odd_number_divisors",
        entity_type=EntityType.CONCEPT,
        description="odd_number_divisors",
        computational_implementation=lambda a: a**0.5 == int(a**0.5),
        discovered_names={"compose_(tau_with_is_odd_output_to_input_map={0: 0})"}
    ),
    "is_square_and_odd_number_divisors": GroundTruthEntity(
        canonical_name="is_square_and_odd_number_divisors",
        entity_type=EntityType.CONCEPT,
        description="is_square_and_odd_number_divisors",
        computational_implementation=lambda a: a**0.5 == int(a**0.5),
        discovered_names={"compose_(odd_number_divisors_with_is_square_shared_vars={0: 0})",
                          "compose_(is_square_with_odd_number_divisors_shared_vars={0: 0})"}
    ),
    "even_number_divisors": GroundTruthEntity(
        canonical_name="even_number_divisors",
        entity_type=EntityType.CONCEPT,
        description="even_number_divisors",
        computational_implementation=lambda a: a**0.5 != int(a**0.5),
        discovered_names={"compose_(tau_with_is_even_output_to_input_map={0: 0})"}
    ),
    "not_square_and_even_number_divisors": GroundTruthEntity(
        canonical_name="not_square_and_even_number_divisors",
        entity_type=EntityType.CONCEPT,
        description="not_square_and_even_number_divisors",
        computational_implementation=lambda a: a**0.5 != int(a**0.5),
        discovered_names={"compose_(even_number_divisors_with_not_square_shared_vars={0: 0})",
                          "compose_(not_square_with_even_number_divisors_shared_vars={0: 0})"}
    ),
    # --
    # powers ====================================================================
    # -- even square divis by 4 --
    "is_even_square": GroundTruthEntity(
        canonical_name="is_even_square",
        entity_type=EntityType.CONCEPT,
        description="is_even_square",
        computational_implementation=lambda a: a % 4 == 0,
        discovered_names={"compose_(is_even_with_is_square_shared_vars={0: 0})",
                          "compose_(is_square_with_is_even_shared_vars={0: 0})"}
    ),
    "divis_by_four": GroundTruthEntity(
        canonical_name="divis_by_four",
        entity_type=EntityType.CONCEPT,
        description='divis by 4',
        computational_implementation=lambda a: a % 4 == 0,
        discovered_names={"specialized_(divides_at_0_to_four)"}
    ),
    # ---
    # -- pythagorean triples --
    "is_pythagorean_triple_zzz": GroundTruthEntity(
        canonical_name="is_pythagorean_triple_zzz",
        entity_type=EntityType.CONCEPT,
        description="is_pythagorean_triple_zzz",
        computational_implementation=lambda a,b,c: a**2 + b**2 == c**2,
        discovered_names={"specialized_(sum_two_powers_eq_power_match_exp_at_1_to_two)",
                          "specialized_(power_eq_sum_two_powers_match_exp_at_1_to_two)"
                          }
    ),
    "is_pythagorean_triple_zz": GroundTruthEntity(
        canonical_name="is_pythagorean_triple_zz",
        entity_type=EntityType.CONCEPT,
        description="is_pythagorean_triple_zz",
        computational_implementation=lambda a,b,c: a**2 + b**2 == c**2 and a != 0,
        discovered_names={"compose_(is_pythagorean_triple_zzz_with_gt_zero_shared_vars={0: 0})",
                          "compose_(is_pythagorean_triple_zzz_with_gt_zero_shared_vars={1: 0})",
                          "compose_(is_pythagorean_triple_zzz_with_gt_zero_shared_vars={2: 0})",
                          }
    ),
    "is_pythagorean_triple_z": GroundTruthEntity(
        canonical_name="is_pythagorean_triple_z",
        entity_type=EntityType.CONCEPT,
        description="is_pythagorean_triple_z",
        computational_implementation=lambda a,b,c: a**2 + b**2 == c**2 and a != 0 and b!= 0,
        discovered_names={ "compose_(is_pythagorean_triple_zz_with_gt_zero_shared_vars={1: 0})",
                          "compose_(is_pythagorean_triple_zz_with_gt_zero_shared_vars={2: 0})",
                          }
    ),
    "is_pythagorean_triple": GroundTruthEntity(
        canonical_name="is_pythagorean_triple",
        entity_type=EntityType.CONCEPT,
        description="is_pythagorean_triple",
        computational_implementation=lambda a,b,c: a**2 + b**2 == c**2 and a != 0 and b!= 0 and c!=0,
        discovered_names={ "compose_(is_pythagorean_triple_z_with_gt_zero_shared_vars={2: 0})"
                          }
    ),
    "is_pythagorean_triple_gt": GroundTruthEntity(
        canonical_name="is_pythagorean_triple_gt",
        entity_type=EntityType.CONCEPT,
        description="is_pythagorean_triple_gt",
        computational_implementation=lambda a,b,c: a**2 + b**2 == c**2 and a != 0 and b!= 0 and c!=0,
        discovered_names={ "compose_(is_pythagorean_triple_with_gt_shared_vars={2: 0})",
                          "compose_(is_pythagorean_triple_with_gt_shared_vars={1: 0})",
                          "compose_(is_pythagorean_triple_with_gt_shared_vars={0: 0})",
                          "compose_(is_pythagorean_triple_with_lt_shared_vars={2: 1})",
                          "compose_(is_pythagorean_triple_with_lt_shared_vars={1: 1})",
                          "compose_(is_pythagorean_triple_with_lt_shared_vars={0: 1})",
                          }
    ),

    # -- fermats last thm --
    "power_eq_sum_two_powers": GroundTruthEntity(
      canonical_name="power_eq_sum_two_powers",
      entity_type=EntityType.CONCEPT,
      description="power is equal to sum of two powers",
      computational_implementation=lambda a,b,c,d,m,n: a**b == c**d + m**n,
      discovered_names={"compose_(power_with_eq_sum_two_powers_output_to_input_map={0: 4})"}
    ),
    "sum_two_powers_eq_power": GroundTruthEntity(
      canonical_name="sum_two_powers_eq_power",
      entity_type=EntityType.CONCEPT,
      description="sum of two powers is equal to power",
      computational_implementation=lambda a,b,c,d,m,n: a**b + c**d == m**n,
      discovered_names={"compose_(sum_two_powers_with_eq_power_output_to_input_map={0: 2})"}
    ),
    # --
    "power_eq_sum_two_powers_match_exp": GroundTruthEntity(
      canonical_name="power_eq_sum_two_powers_match_exp",
      entity_type=EntityType.CONCEPT,
      description="power is equal to sum of two powers, all exponents held equal",
      computational_implementation=lambda a,n,b,c: a**n == b**n + c**n,
      discovered_names={"matched_(power_eq_sum_two_powers_indices_[1, 3, 5])"}
    ),
    "sum_two_powers_eq_power_match_exp": GroundTruthEntity(
      canonical_name="sum_two_powers_eq_power_match_exp",
      entity_type=EntityType.CONCEPT,
      description="sum of two powers is equal to power, all exponents held equal",
      computational_implementation=lambda a,n,b,c: a**n + b**n == c**n,
      discovered_names={"matched_(sum_two_powers_eq_power_indices_[1, 3, 5])"}
    ),
    # --
    "not_power_eq_sum_two_powers_match_exp": GroundTruthEntity(
      canonical_name="not_power_eq_sum_two_powers_match_exp",
      entity_type=EntityType.CONCEPT,
      description="power is not equal to sum of two powers, all exponents held equal",
      computational_implementation=lambda a,n,b,c: a**n != b**n + c**n,
      discovered_names={"not_(power_eq_sum_two_powers_match_exp)"}
    ),
    "fermats_last_thm_true_1_zz": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_1_zz",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0",
        computational_implementation=lambda a,n,b,c: a**n != b**n + c**n and a != 0,
        discovered_names={"compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={0: 0})",
                          "compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={2: 0})",
                          "compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_true_1_z": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_1_z",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0 and b!=0",
        computational_implementation=lambda a,n,b,c: a**n != b**n + c**n and a != 0 and b != 0,
        discovered_names={"compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={2: 0})",
                          "compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_true_1": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_1",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0 and b!=0 and c !=0",
        computational_implementation=lambda a,n,b,c: a**n != b**n + c**n and a != 0 and b != 0 and c != 0,
        discovered_names={"compose_(not_power_eq_sum_two_powers_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),
    "not_sum_two_powers_eq_power_match_exp": GroundTruthEntity(
      canonical_name="not_sum_two_powers_eq_power_match_exp",
      entity_type=EntityType.CONCEPT,
      description="sum of two powers not equal to power, all exponents held equal",
      computational_implementation=lambda a,n,b,c: a**n + b**n != c**n,
      discovered_names={"not_(sum_two_powers_eq_power_match_exp)"}
    ),
    "fermats_last_thm_true_2_zz": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_2_zz",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0",
        computational_implementation=lambda a,n,b,c: a**n + b**n != c**n and a != 0,
        discovered_names={"compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={0: 0})",
                          "compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={2: 0})",
                          "compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_true_2_z": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_2_z",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0 and b!=0",
        computational_implementation=lambda a,n,b,c: a**n + b**n != c**n and a != 0 and b != 0,
        discovered_names={"compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={2: 0})",
                          "compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_true_2": GroundTruthEntity(
        canonical_name="fermats_last_thm_true_2",
        entity_type=EntityType.CONCEPT,
        description="c^n != a^n + b^n, negate to get fermat's last thm, a!=0 and b!=0 and c !=0",
        computational_implementation=lambda a,n,b,c: a**n + b**n != c**n and a != 0 and b != 0 and c != 0,
        discovered_names={"compose_(not_sum_two_powers_eq_power_match_exp_with_gt_zero_shared_vars={3: 0})"}
    ),    
    # --
    "fermats_last_thm_false_zzz": GroundTruthEntity(
      canonical_name="fermats_last_thm_false_zzz",
      entity_type=EntityType.CONCEPT,
      description="c^n = a^n + b^n and n > 2, negate to get fermat's last thm",
      computational_implementation=lambda a,n,b,c: a**n == b**n + c**n and n>2,
      discovered_names={"compose_(gt_two_with_sum_two_powers_eq_power_match_exp_shared_vars={0: 1})",
                        "compose_(gt_two_with_power_eq_sum_two_powers_match_exp_shared_vars={0: 1})",
                        "compose_(sum_two_powers_eq_power_match_exp_with_gt_two_shared_vars={1: 0})",
                        "compose_(power_eq_sum_two_powers_match_exp_with_gt_two_shared_vars={1: 0})"}
    ),
    "fermats_last_thm_false_zz": GroundTruthEntity(
      canonical_name="fermats_last_thm_false_zz",
      entity_type=EntityType.CONCEPT,
      description="c^n = a^n + b^n and n > 2, negate to get fermat's last thm",
      computational_implementation=lambda a,n,b,c: a**n == b**n + c**n and n>2 and a!= 0,
      discovered_names={"compose_(fermats_last_thm_false_zzz_with_gt_zero_shared_vars={0: 0})",
                        "compose_(fermats_last_thm_false_zzz_with_gt_zero_shared_vars={2: 0})",
                        "compose_(fermats_last_thm_false_zzz_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_false_z": GroundTruthEntity(
      canonical_name="fermats_last_thm_false_z",
      entity_type=EntityType.CONCEPT,
      description="c^n = a^n + b^n and n > 2, negate to get fermat's last thm",
      computational_implementation=lambda a,n,b,c: a**n == b**n + c**n and n>2 and a!= 0 and b!= 0,
      discovered_names={"compose_(fermats_last_thm_false_zz_with_gt_zero_shared_vars={2: 0})",
                        "compose_(fermats_last_thm_false_zz_with_gt_zero_shared_vars={3: 0})"}
    ),
    "fermats_last_thm_false": GroundTruthEntity(
      canonical_name="fermats_last_thm_false",
      entity_type=EntityType.CONCEPT,
      description="c^n = a^n + b^n and n > 2, negate to get fermat's last thm",
      computational_implementation=lambda a,n,b,c: a**n == b**n + c**n and n>2 and a!= 0 and b!= 0 and c!= 0, 
      discovered_names={"compose_(fermats_last_thm_false_z_with_gt_zero_shared_vars={3: 0})"}
    ),
    # ---
    # -- euler's sum of powers conjecture --
    "eq_sum_three_numbers_fourth_power_summand1": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers_fourth_power_summand1",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b + c == d",
        computational_implementation=lambda a,b,c,d: a**4 + b + c == d,
        discovered_names={"compose_(fourth_power_with_eq_sum_three_numbers_output_to_input_map={0: 0})", 
                          "compose_(fourth_power_with_eq_sum_three_numbers_output_to_input_map={0: 1})",
                          "compose_(fourth_power_with_eq_sum_three_numbers_output_to_input_map={0: 2})", 
                          "compose_(fourth_power_with_eq_sum_three_numbers_output_to_input_map={0: 3})"}
    ),
    "eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c == d",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c == d,
        discovered_names={"compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_output_to_input_map={0: 1})",
                          "compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_output_to_input_map={0: 2})", 
                          "compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_output_to_input_map={0: 3})"}
    ),
    "eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2_fourth_power_summand3": GroundTruthEntity(
        canonical_name="eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2_fourth_power_summand3",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d,
        discovered_names={"compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2_output_to_input_map={0: 2})",
                          "compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2_output_to_input_map={0: 3})"}
    ),
    "sum_of_fourth_powers_equals_fourth_power_zzzz": GroundTruthEntity(
        canonical_name="sum_of_fourth_powers_equals_fourth_power_zzzz",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d^4",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d**4,
        discovered_names={"compose_(fourth_power_with_eq_sum_three_numbers_fourth_power_summand1_fourth_power_summand2_fourth_power_summand3_output_to_input_map={0: 3})"}
    ),
    "sum_of_fourth_powers_equals_fourth_power_zzz": GroundTruthEntity(
        canonical_name="sum_of_fourth_powers_equals_fourth_power_zzz",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d^4 and a > 0",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d**4 and a > 0,
        discovered_names={"compose_(sum_of_fourth_powers_equals_fourth_power_zzzz_with_gt_zero_shared_vars={0: 0})"}
    ),
    "sum_of_fourth_powers_equals_fourth_power_zz": GroundTruthEntity(
        canonical_name="sum_of_fourth_powers_equals_fourth_power_zz",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d^4 and a > 0 and b > 0",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d**4 and a > 0 and b > 0,
        discovered_names={"compose_(sum_of_fourth_powers_equals_fourth_power_zzz_with_gt_zero_shared_vars={1: 0})"}
    ),
    "sum_of_fourth_powers_equals_fourth_power_z": GroundTruthEntity(
        canonical_name="sum_of_fourth_powers_equals_fourth_power_z",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d^4 and a > 0 and b > 0 and c > 0",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d**4 and a > 0 and b > 0 and c > 0,
        discovered_names={"compose_(sum_of_fourth_powers_equals_fourth_power_zz_with_gt_zero_shared_vars={2: 0})"}
    ),
    "sum_of_fourth_powers_equals_fourth_power": GroundTruthEntity(
        canonical_name="sum_of_fourth_powers_equals_fourth_power",
        entity_type=EntityType.CONCEPT,
        description="a^4 + b^4 + c^4 == d^4 and a > 0 and b > 0 and c > 0 and d > 0",
        computational_implementation=lambda a,b,c,d: a**4 + b**4 + c**4 == d**4 and a > 0 and b > 0 and c > 0 and d > 0,
        discovered_names={"compose_(sum_of_fourth_powers_equals_fourth_power_z_with_gt_zero_shared_vars={3: 0})"}
    ),
    # number calculations =====================================================
    "gt_two_even": GroundTruthEntity(
        canonical_name="gt_two_even",
        entity_type=EntityType.CONCEPT,
        description='is even number greater than two',
        computational_implementation=lambda a: a > 2 and a % 2 == 0,
        discovered_names={"compose_(gt_two_with_is_even_shared_vars={0: 0})",
                          "compose_(is_even_with_gt_two_shared_vars={0: 0})"}
    ),
    "is_0_mod_3": GroundTruthEntity(
        canonical_name="is_0_mod_3",
        entity_type=EntityType.CONCEPT,
        description="0 modulo 3 predicate",
        computational_implementation=lambda a: a % 3 == 0,
        discovered_names={"specialized_(divides_at_0_to_three)"}
    ),
}