"""Specification file for FunSearch interestingness functions."""

import re
from typing import Dict, Any, Optional, List, Union
import math
import yaml
import os
import logging

# Note(_; 4/11): The default implementation starts with essentially a random policy.
class InterestingnessSpec:
    """Specification for interestingness function to use with FunSearch."""
    
    @staticmethod
    def get_function_signature() -> str:
        """
        Returns the function signature for the interestingness function.
        
        Returns:
            A string containing the function signature
        """
        return (
            "def calculate_interestingness(entity_id: str, graph: KnowledgeGraph) -> float:"
        )
    
    @staticmethod
    def get_function_header() -> str:
        """
        Returns the full header (signature + docstring) for the interestingness function.
        
        Returns:
            A string containing the function header
        """
        signature = InterestingnessSpec.get_function_signature()
        docstring = '''\n    """Calculate the interestingness score for a given entity.\n    \n    Args:\n        entity_id: The ID of the entity to score.\n        graph: The knowledge graph containing the entity.\n        \n    Returns:\n        A float value representing the interestingness score (higher is more interesting).\n    """'''
        return signature + docstring
    
    @staticmethod
    def get_default_function() -> str:
        """
        Returns a default implementation of the interestingness function.
        
        Returns:
            A string containing the full function implementation
        """
        header = InterestingnessSpec.get_function_header()
        implementation = """
    # Random implementation
    try:
        return 0.0
        
    except Exception as e:
        # Log the error if possible, or just return 0
        # print(f"Error scoring entity {entity_id}: {e}") # Avoid print in prod
        return 0.0
"""
        return header + implementation
    
    @staticmethod
    def get_imports() -> str:
        """
        Returns the import statements needed for the interestingness function.
        Now loads from the centralized YAML file.
        
        Returns:
            A string containing the import statements, or an empty string if loading fails.
        """
        # Path relative to this script's location
        # Use inspect to get the path relative to where InterestingnessSpec is defined
        # Assume 'prompt_yaml_path' is injected into the global scope when exec'd
        # Default to a relative path attempt if not found (might still fail)
        try:
            yaml_path = prompt_yaml_path 
        except NameError:
            logging.error("'prompt_yaml_path' not found in global scope for get_imports. Attempting default relative path.")
            # Fallback attempt (likely won't work reliably in exec context)
            yaml_path = os.path.join(os.path.dirname(__file__), '..', 'configs', 'prompts', 'interestingness_prompt.yaml')
        
        try:
            with open(yaml_path, 'r') as f:
                prompt_config = yaml.safe_load(f)
            imports_str = prompt_config['required_imports'] # Use direct access, raises KeyError if missing
            return imports_str.strip() + '\n' # Ensure trailing newline
        except FileNotFoundError:
            logging.error(f"Prompt config file not found at {yaml_path} when loading imports.")
            raise # Re-raise the error
        except yaml.YAMLError as e:
            logging.error(f"Error parsing prompt config YAML ({yaml_path}): {e}")
            raise RuntimeError(f"Failed to parse YAML config: {yaml_path}") from e # Raise a runtime error
        except KeyError:
            logging.error(f"Key 'required_imports' not found in {yaml_path}")
            raise # Re-raise the KeyError
        except Exception as e:
            logging.error(f"Unexpected error loading imports from {yaml_path}: {e}")
            raise RuntimeError(f"Unexpected error loading imports from {yaml_path}") from e # Raise a runtime error
    
    @staticmethod
    def validate_function(function_code: str) -> bool:
        """
        Validates that the function code adheres to the expected signature.
        
        Args:
            function_code: The Python code string to validate
            
        Returns:
            True if the function code is valid, False otherwise
        """
        # Check if the function code contains the expected function name
        if "def calculate_interestingness(" not in function_code:
            return False
        
        # Check parameters using a regex pattern
        pattern = r"def\s+calculate_interestingness\s*\(\s*entity_id\s*:\s*str\s*,\s*graph\s*:\s*KnowledgeGraph\s*\)\s*->\s*float\s*:"
        if not re.search(pattern, function_code):
            return False
        
        return True
    
    @staticmethod
    def create_full_function_code() -> str:
        """
        Creates a complete function code with imports, function signature, and implementation.
        
        Returns:
            A string containing the complete function code
        """
        imports = InterestingnessSpec.get_imports()
        function_code = InterestingnessSpec.get_default_function()
        
        return imports + function_code
