"""Algorithms for learning interestingness functions.

This module contains implementations of various algorithms for learning
interestingness functions from the DSL primitives.

Supported algorithms:
- LLM-based (One-shot) generation
- (Planned) FunSearch
"""

# Import necessary modules
from typing import List, Dict, Any, Callable, Optional, Tuple
import random
import numpy as np

from frame.knowledge_base.knowledge_graph import KnowledgeGraph
from frame.interestingness.learning.dsl_primitives import ALL_PRIMITIVES
from frame.interestingness.learning.algorithms.one_shot_llm import (
    OneShotLLMGenerator,
    prepare_prompt,
)

# This is a placeholder file for future algorithm implementations.
# Any new algorithm for learning interestingness functions should be 
# implemented here as a class with standardized interfaces.

class InterestingnessLearner:
    """Base class for all interestingness function learning algorithms."""
    
    def __init__(self, 
                 knowledge_graph: KnowledgeGraph,
                 evaluation_metric: Callable = None,
                 seed: Optional[int] = None):
        """
        Initialize the interestingness learner.
        
        Args:
            knowledge_graph: The knowledge graph to learn from
            evaluation_metric: Function to evaluate candidate interestingness functions
            seed: Random seed for reproducibility
        """
        self.knowledge_graph = knowledge_graph
        self.evaluation_metric = evaluation_metric
        self.random_state = np.random.RandomState(seed)
        
    def learn(self, **kwargs) -> Callable:
        """
        Learn an interestingness function.
        
        Returns:
            A callable that takes (entity_id, graph) and returns an interestingness score
        """
        raise NotImplementedError("Subclasses must implement learn()")
        
    def evaluate(self, candidate_function: Callable) -> float:
        """
        Evaluate a candidate interestingness function.
        
        Args:
            candidate_function: A callable taking (entity_id, graph) -> float
            
        Returns:
            A score for the function, higher is better
        """
        if self.evaluation_metric is None:
            raise ValueError("No evaluation metric provided")
        return self.evaluation_metric(candidate_function, self.knowledge_graph)

# Placeholder for a future FunSearch implementation
# class FunSearchLearner(InterestingnessLearner):
#     """Learn interestingness functions using FunSearch."""
#     ...

# Placeholder for a future Genetic Programming implementation  
# class GeneticProgrammingLearner(InterestingnessLearner):
#     """Learn interestingness functions using Genetic Programming."""
#     ...

__all__ = [
    'OneShotLLMGenerator',
    'prepare_prompt',
]
