"""Base classes for REPL commands."""

from typing import List, Dict, Any, Optional
from abc import ABC, abstractmethod
from dataclasses import dataclass
from frame.repl.core.exceptions import (
    CommandError, InvalidArgumentError, EntityNotFoundError,
    RuleApplicationError, ValidationError
)

from frame.repl.interface.protocols import Command, CommandRegistry, REPLContext

@dataclass
class CommandResult:
    """Result of command execution."""
    success: bool
    message: str
    data: Optional[Any] = None
    error: Optional[Exception] = None

class BaseCommand(ABC):
    """Base class for REPL commands."""
    
    @property
    def name(self) -> str:
        """Get the command name."""
        return self._get_command_name()
    
    def _get_command_name(self) -> str:
        """Get the command name from the class name."""
        return self.__class__.__name__.lower().replace('command', '')
    
    @property
    def help(self) -> str:
        """Get help text for the command."""
        # Use docstring if available, otherwise a default message
        return self.__doc__ or "No help available for this command."
    
    @abstractmethod
    def execute(self, *args, **kwargs) -> CommandResult:
        """Execute the command."""
        pass
    
    @abstractmethod
    def get_completions(self, *args, **kwargs) -> List[str]:
        """Get command completions."""
        pass
    
    def validate_args(self, args: tuple, kwargs: dict) -> None:
        """Validate command arguments.
        
        Args:
            args: Positional arguments
            kwargs: Keyword arguments
            
        Raises:
            InvalidArgumentError: If arguments are invalid
            ValidationError: If validation fails
        """
        pass
    
    def handle_error(self, error: Exception) -> CommandResult:
        """Handle command errors.
        
        Args:
            error: The exception that occurred
            
        Returns:
            CommandResult with error information
        """
        if isinstance(error, CommandError):
            return CommandResult(
                success=False,
                message=str(error),
                error=error
            )
        else:
            # For unexpected errors, provide a generic message
            return CommandResult(
                success=False,
                message=f"An unexpected error occurred in command '{self.name}': {str(error)}",
                error=error
            )
    
    def safe_execute(self, *args, **kwargs) -> CommandResult:
        """Execute the command with error handling.
        
        Args:
            *args: Positional arguments
            **kwargs: Keyword arguments
            
        Returns:
            CommandResult with execution result
        """
        try:
            # Validate arguments first
            self.validate_args(args, kwargs)
            
            # Execute the command
            return self.execute(*args, **kwargs)
            
        except Exception as e:
            return self.handle_error(e)

class BaseCommandRegistry:
    """Registry for managing available commands."""
    
    def __init__(self):
        """Initialize an empty command registry."""
        self._commands: Dict[str, Command] = {}
    
    def register(self, command: Command) -> None:
        """Register a new command.
        
        Args:
            command: Command instance to register
        """
        self._commands[command.name] = command
    
    def get_command(self, name: str) -> Optional[Command]:
        """Get a command by name.
        
        Args:
            name: Name of the command to retrieve
            
        Returns:
            Command instance if found, None otherwise
        """
        return self._commands.get(name.lower())
    
    def get_all_commands(self) -> Dict[str, Command]:
        """Get all registered commands.
        
        Returns:
            Dictionary mapping command names to Command instances
        """
        return dict(self._commands)
    
    def get_completions(self, context: REPLContext, partial: str) -> List[str]:
        """Get completions for partial command input.
        
        This handles both command name completion and argument completion
        for known commands.
        
        Args:
            context: Current REPL context
            partial: Partial input to complete
            
        Returns:
            List of possible completions
        """
        # Split into command and argument parts
        parts = partial.strip().split()
        
        # If no parts or just starting to type command
        if not parts:
            return list(self._commands.keys())
            
        # If completing command name
        if len(parts) == 1:
            return [name for name in self._commands.keys()
                   if name.startswith(parts[0].lower())]
                   
        # If completing command arguments
        command = self.get_command(parts[0])
        if command:
            # Pass remaining partial input to command's completion handler
            remaining = ' '.join(parts[1:])
            return command.get_completions(context, remaining)
            
        return [] 