"""Custom exceptions for the REPL system."""

class REPLError(Exception):
    """Base class for all REPL-related errors."""
    def __init__(self, message: str, command: str = None):
        self.message = message
        self.command = command
        super().__init__(self.message)

class CommandError(REPLError):
    """Base class for command-specific errors."""
    pass

class InvalidArgumentError(CommandError):
    """Raised when command arguments are invalid."""
    def __init__(self, message: str, command: str, arg_name: str = None):
        self.arg_name = arg_name
        super().__init__(message, command)

class EntityNotFoundError(CommandError):
    """Raised when an entity is not found in the knowledge graph."""
    def __init__(self, entity_name: str, command: str):
        self.entity_name = entity_name
        super().__init__(f"Entity '{entity_name}' not found", command)

class RuleApplicationError(CommandError):
    """Raised when a rule cannot be applied."""
    def __init__(self, message: str, command: str, rule_name: str = None):
        self.rule_name = rule_name
        super().__init__(message, command)

class ValidationError(CommandError):
    """Raised when command validation fails."""
    def __init__(self, message: str, command: str, field: str = None):
        self.field = field
        super().__init__(message, command)

class StateError(REPLError):
    """Raised when there's an error with the REPL state."""
    pass

class ConfigurationError(REPLError):
    """Raised when there's an error with REPL configuration."""
    pass

class PermissionError(REPLError):
    """Raised when a command requires permissions that aren't available."""
    pass 