"""Protocol definitions for the REPL system."""

from dataclasses import dataclass
from typing import Any, Dict, List, Optional, Protocol

@dataclass
class CommandResult:
    """Result of executing a command."""
    success: bool
    message: str = None
    data: Any = None
    error: Any = None

class Command(Protocol):
    """Protocol for command objects."""
    
    @property
    def name(self) -> str:
        """Get the command name."""
        ...
        
    @property
    def help(self) -> str:
        """Get help text for the command."""
        ...
        
    def execute(self, context: Any, *args) -> CommandResult:
        """Execute the command."""
        ...
        
    def get_completions(self, context: Any, partial: str) -> List[str]:
        """Get possible completions."""
        ...

class CommandRegistry(Protocol):
    """Protocol for command registry."""
    
    def register(self, command: Any) -> None:
        """Register a command."""
        ...
        
    def get_command(self, name: str) -> Optional[Any]:
        """Get a command by name."""
        ...
        
    def get_all_commands(self) -> Dict[str, Any]:
        """Get all registered commands."""
        ...
        
    def get_completions(self, context: Any, partial: str) -> List[str]:
        """Get completions for partial input."""
        ...

class REPLContext(Protocol):
    """Protocol for REPL context."""
    
    @property
    def graph(self) -> Any:
        """Get the current knowledge graph."""
        ...
        
    @property
    def available_rules(self) -> List[Any]:
        """Get available production rules."""
        ...
        
    def get_command_registry(self) -> CommandRegistry:
        """Get the command registry."""
        ...
        
    def get_concepts(self) -> List[Any]:
        """Get all available concepts."""
        ...
        
    def get_conjectures(self) -> List[Any]:
        """Get all available conjectures."""
        ...
        
    def get_entity_by_name(self, name: str) -> Optional[Any]:
        """Get an entity by its name."""
        ...
        
    def get_rule_by_name(self, name: str) -> Optional[Any]:
        """Get a rule by its name."""
        ... 