# frame/utils/proof_cache.py (New File)
import sqlite3
import threading
import time
import logging
import ast # For safe literal evaluation
from typing import Optional, Tuple, Any, Set, Literal
from datetime import datetime

logger = logging.getLogger(__name__)

ProofStatus = Literal["Proven", "Disproven", "Unknown"]

class EpisodicCache:
    _local = threading.local() # Thread-local storage for connections

    def __init__(self, db_path: str):
        self.db_path = db_path
        # Initialize DB connection and cursor for the main thread/process if needed,
        # but prioritize creating connections per thread/process access.
        self._ensure_connection() # Create initial connection for setup
        self._create_tables()

    def _ensure_connection(self) -> sqlite3.Connection:
        """Ensures a connection exists for the current thread/process."""
        if not hasattr(self._local, 'connection') or self._local.connection is None:
            try:
                # Connect with a slightly longer timeout for waiting on locks
                self._local.connection = sqlite3.connect(self.db_path, timeout=0.5) # Increased to 0.5 second timeout
                try:
                    self._local.connection.execute("PRAGMA journal_mode=WAL;") # Enable WAL for better concurrency
                    logger.debug(f"Set journal_mode=WAL for connection in thread/process: {threading.get_ident()}")
                except sqlite3.Error as wal_e:
                    # If setting WAL mode fails due to lock, assume another process set it.
                    if "database is locked" in str(wal_e):
                        logger.debug(f"Could not set WAL mode (likely already set by another process): {wal_e}")
                    else:
                        raise # Re-raise other WAL setting errors
                self._local.connection.execute("PRAGMA foreign_keys = ON;")
                # logger.debug(f"SQLite connection established for thread/process: {threading.get_ident()}")
            except sqlite3.Error as e:
                logger.error(f"Failed to connect to EpisodicCache DB at {self.db_path}: {e}")
                raise
        return self._local.connection

    def _create_tables(self):
        conn = self._ensure_connection()
        try:
            with conn: # Use connection as context manager for transactions
                conn.execute("""
                    CREATE TABLE IF NOT EXISTS entities (
                        entity_name TEXT PRIMARY KEY,
                        entity_type TEXT NOT NULL CHECK(entity_type IN ('Concept', 'Conjecture')),
                        proof_status TEXT NOT NULL DEFAULT 'Unknown' CHECK(proof_status IN ('Proven', 'Disproven', 'Unknown')),
                        last_checked TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                    )
                """)
                conn.execute("""
                    CREATE TABLE IF NOT EXISTS examples (
                        example_id INTEGER PRIMARY KEY AUTOINCREMENT,
                        entity_name TEXT NOT NULL,
                        is_positive INTEGER NOT NULL CHECK(is_positive IN (0, 1)),
                        example_value_repr TEXT NOT NULL,
                        FOREIGN KEY (entity_name) REFERENCES entities (entity_name) ON DELETE CASCADE,
                        UNIQUE (entity_name, is_positive, example_value_repr)
                    )
                """)
                # Add index for faster example lookups
                conn.execute("""
                    CREATE INDEX IF NOT EXISTS idx_examples_lookup
                    ON examples (entity_name, is_positive)
                """)
        except sqlite3.Error as e:
            logger.error(f"Failed to create EpisodicCache tables: {e}")
            raise

    def get_proof_status(self, entity_name: str) -> ProofStatus:
        conn = self._ensure_connection()
        try:
            cursor = conn.execute("SELECT proof_status FROM entities WHERE entity_name = ?", (entity_name,))
            result = cursor.fetchone()
            return result[0] if result else "Unknown"
        except sqlite3.Error as e:
            # Check for specific lock error
            if "database is locked" in str(e):
                logger.info(f"CACHE LOCK encountered getting proof status for '{entity_name}': {e}")
            else:
                logger.warning(f"Failed to get proof status for '{entity_name}': {e}")
            return "Unknown" # Default to Unknown on error

    def update_proof_status(self, entity_name: str, entity_type: Literal["Concept", "Conjecture"], status: ProofStatus):
        conn = self._ensure_connection()
        start_time = time.time()
        success = False
        try:
            with conn: # Transaction ensures atomicity
                conn.execute("""
                    INSERT INTO entities (entity_name, entity_type, proof_status, last_checked)
                    VALUES (?, ?, ?, ?)
                    ON CONFLICT(entity_name) DO UPDATE SET
                        proof_status = excluded.proof_status,
                        entity_type = excluded.entity_type, -- Update type in case it changes (e.g., Conjecture -> Theorem)
                        last_checked = CURRENT_TIMESTAMP
                """, (entity_name, entity_type, status, datetime.now()))
            success = True
        except sqlite3.Error as e:
            # Check for specific lock error
            if "database is locked" in str(e):
                logger.info(f"CACHE LOCK encountered updating proof status for '{entity_name}' to '{status}': {e}")
            else:
                logger.warning(f"Failed to update proof status for '{entity_name}' to '{status}': {e}")
        finally:
            duration = time.time() - start_time
            # Log duration at INFO level
            if success:
                logger.info(f"Cache: Updated proof status for '{entity_name}' to '{status}' in {duration:.4f}s")
            else:
                logger.info(f"Cache: Proof status update attempt for '{entity_name}' took {duration:.4f}s (failed, possibly due to lock)")

    def add_example(self, entity_name: str, example_value: Tuple[Any, ...]):
        """Adds a verified positive example to the cache."""
        self._add_example_internal(entity_name, example_value, is_positive=True)

    def add_nonexample(self, entity_name: str, example_value: Tuple[Any, ...]):
        """Adds a verified negative example (non-example) to the cache."""
        self._add_example_internal(entity_name, example_value, is_positive=False)

    def _add_example_internal(self, entity_name: str, example_value: Tuple[Any, ...], is_positive: bool):
        """Internal helper to add examples/non-examples to the cache."""
        conn = self._ensure_connection()
        # Ensure the entity exists in the entities table first (or add it as Unknown)
        # Note: A Concept might not have a proof status, but we still need the entity entry for FK constraint.
        if self.get_proof_status(entity_name) == "Unknown": # Check if entity exists
             self.update_proof_status(entity_name, "Concept", "Unknown") # Add Concept as Unknown status if missing

        start_time = time.time()
        success = False
        value_repr = "<serialization_failed>" # Default in case repr fails
        try:
            value_repr = repr(example_value)
            is_pos_int = 1 if is_positive else 0
            with conn: # Transaction
                # Use INSERT OR IGNORE to avoid errors if the exact example already exists
                conn.execute("""
                    INSERT OR IGNORE INTO examples (entity_name, is_positive, example_value_repr)
                    VALUES (?, ?, ?)
                """, (entity_name, is_pos_int, value_repr))
            success = True
        except sqlite3.Error as e:
            # Check for specific lock error
            if "database is locked" in str(e):
                logger.info(f"CACHE LOCK encountered adding example for '{entity_name}' ({value_repr}): {e}") # Changed to INFO
            else:
                logger.warning(f"Failed to add example for '{entity_name}' ({value_repr}): {e}")
        except Exception as e_repr: # Catch potential errors during repr()
             logger.warning(f"Failed to serialize example value for caching: {example_value}. Error: {e_repr}")
        finally:
            duration = time.time() - start_time
            # Log duration at INFO level
            log_prefix = "positive" if is_positive else "negative"
            if success:
                logger.info(f"Cache: Added {log_prefix} example for '{entity_name}': {value_repr} in {duration:.4f}s")
            else:
                logger.info(f"Cache: Add {log_prefix} example attempt for '{entity_name}' ({value_repr}) took {duration:.4f}s (failed, possibly due to lock)")

    def get_examples(self, entity_name: str) -> Tuple[Set[Tuple[Any, ...]], Set[Tuple[Any, ...]]]:
        """Gets all cached positive and negative examples for an entity."""
        conn = self._ensure_connection()
        positive_examples = set()
        negative_examples = set()
        try:
            cursor = conn.execute("""
                SELECT is_positive, example_value_repr FROM examples WHERE entity_name = ?
            """, (entity_name,))
            for is_pos_int, value_repr in cursor.fetchall():
                try:
                    # Use ast.literal_eval for safer evaluation than eval()
                    value = ast.literal_eval(value_repr)
                    if not isinstance(value, tuple): # Ensure it deserialized to a tuple
                         raise TypeError("Deserialized value is not a tuple")
                    if is_pos_int == 1:
                        positive_examples.add(value)
                    else:
                        negative_examples.add(value)
                except (ValueError, SyntaxError, TypeError) as e_eval:
                    logger.warning(f"Failed to deserialize cached example '{value_repr}' for '{entity_name}': {e_eval}")
            # logger.debug(f"Retrieved {len(positive_examples)} pos / {len(negative_examples)} neg examples for '{entity_name}'")
        except sqlite3.Error as e:
            # Check for specific lock error
            if "database is locked" in str(e):
                logger.info(f"CACHE LOCK encountered getting examples for '{entity_name}': {e}")
            else:
                logger.warning(f"Failed to get examples for '{entity_name}': {e}")
        return positive_examples, negative_examples

    def close(self):
        """Closes the connection for the current thread/process."""
        if hasattr(self._local, 'connection') and self._local.connection:
            # logger.debug(f"Closing SQLite connection for thread/process: {threading.get_ident()}")
            self._local.connection.close()
            self._local.connection = None