import typing
import copy
import os
from parglare.parser import LRStackNode
from frame.tools.grammar import Grammar
from frame.tools.z3_runtime import Z3DslRuntime
from collections import namedtuple
from abc import ABC, abstractmethod

Z3DslRunResult = namedtuple(
    "Z3DslRunResult",
    [
        "proved",
        "counter_example",
        "timed_out",
        "smt2" 
    ],
)

def get_error_code_str(context: LRStackNode):
    error_context = context.input_str[max(0, context.start_position-10):context.end_position+10]
    error_code_str = context.input_str[context.start_position:context.end_position]
    return f"\"{error_context}\" at \"{error_code_str}\""

class Z3ComposableDslParseNode(ABC):
    def __init__(self):
        self._children : typing.List[typing.Optional['Z3ComposableDslParseNode']] = []
        self.parent = None
        self._program_code = None
        self._parent_executable : typing.Optional[Z3Program] = None

    @property
    def children(self) -> typing.List[typing.Optional['Z3ComposableDslParseNode']]:
        return copy.copy(self._children)

    def add_child(self, child: 'Z3ComposableDslParseNode'):
        assert isinstance(child, Z3ComposableDslParseNode) or child is None
        self._children.append(child)
        if child is not None:
            child.parent = self

    def dsl(self) -> str:
        return self._program_code
    
    def smt2(self) -> str:
        raise NotImplementedError("This method should be implemented in the subclass")
    
    def return_pred(self) -> str:
        raise NotImplementedError("This method should be implemented in the subclass")
    
    def return_expr(self) -> str:
        raise NotImplementedError("This method should be implemented in the subclass")

    def __str__(self):
        return self.dsl()
    
    def __repr__(self):
        return self.__str__()

    @property
    def parent_program(self) -> typing.Optional['Z3Program']:
        return self._parent_executable

    @property
    def params(self) -> int:
        return 0

    def run() -> Z3DslRunResult:
        raise NotImplementedError("This method should be implemented in the subclass")

    @abstractmethod
    def parse(context, nodes):
        pass

class LeafNode(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LeafNode':
        if isinstance(nodes, (str, int)):
            if isinstance(nodes, str):
                if nodes.startswith("b_"):
                    is_bounded = True
                elif nodes.startswith("x_") or nodes.startswith("f_") or nodes.startswith("p_"):
                    is_bounded = False
                leaf_node = LeafNode(nodes, is_bounded)
                leaf_node._program_code = context.input_str[context.start_position:context.end_position]
                return leaf_node
            else:
                leaf_node = LeafNode(nodes, True)
                leaf_node._program_code = context.input_str[context.start_position:context.end_position]
                return leaf_node
        else:
            raise Exception("Compiler Bug: Invalid leaf node")

    def __init__(self, value: typing.Union[str, int], is_bounded: bool = False):
        super().__init__()
        assert len(self._children) == 0, "Leaf node should not have any children"
        assert isinstance(value, (str, int)), "Value should be a string or an integer"
        assert value is not None, "Value should not be None"
        self.value = value
        self.is_bounded = is_bounded
        if isinstance(value, int):
            self.is_bounded = True
    
    @property
    def params(self) -> int:
        return 0

    def add_child(self, child):
        raise Exception("Compiler Bug: Leaf node should not have any children")

    def smt2(self) -> str:
        assert len(self._children) == 0, "Leaf node should not have any children"
        if isinstance(self.value, str):
            assert len(self.value) > 0, "Leaf node value should not be empty"
            return self.value
        elif isinstance(self.value, int):
            return str(self.value)
        else:
            raise Exception("Compiler Bug: Invalid leaf node value")

    def return_expr(self) -> str:
        assert len(self._children) == 0, "Leaf node should not have any children"
        if isinstance(self.value, str):
            assert len(self.value) > 0, "Leaf node value should not be empty"
            return self.value
        elif isinstance(self.value, int):
            return str(self.value)
        else:
            raise Exception("Compiler Bug: Invalid leaf node value")
    
    def return_pred(self) -> str:
        assert len(self._children) == 0, "Leaf node should not have any children"
        if isinstance(self.value, str):
            assert len(self.value) > 0, "Leaf node value should not be empty"
            return self.value
        elif isinstance(self.value, int):
            return str(self.value)
        else:
            raise Exception("Compiler Bug: Invalid leaf node value")

class UnboundedParams(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'UnboundedParams':
        assert isinstance(nodes, list), "Compiler Bug: Invalid unbounded params"
        assert len(nodes) == 2, "Compiler Bug: Invalid unbounded params"
        number_of_params = nodes[1]
        assert isinstance(number_of_params, LeafNode), "Compiler Bug: Invalid unbounded params"
        assert number_of_params.is_bounded, "Compiler Bug: Invalid unbounded params"
        assert isinstance(number_of_params.value, int), "Compiler Bug: Invalid number of params"
        unbounded_params = UnboundedParams(number_of_params.value)
        unbounded_params._program_code = context.input_str[context.start_position:context.end_position]
        return unbounded_params
    
    def __init__(self, number_of_params: int):
        super().__init__()
        assert len(self._children) == 0, "Unbounded params should not have any children"
        assert isinstance(number_of_params, int), "Compiler Bug: Number of params should be an integer"
        assert number_of_params >= 0, "Compiler Bug: Number of params should be greater than or equal to 0"
        self._number_of_params = number_of_params
    
    @property
    def params(self) -> int:
        return self._number_of_params
    
    def add_child(self, child):
        raise Exception("Compiler Bug: Unbounded params should not have any children")

class BoundedParams(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'BoundedParams':
        assert isinstance(nodes, list), "Compiler Bug: Invalid bounded params"
        assert len(nodes) == 3, "Compiler Bug: Invalid bounded params"
        number_of_params = nodes[2]
        assert isinstance(number_of_params, LeafNode), "Compiler Bug: Invalid bounded params"
        assert number_of_params.is_bounded, "Compiler Bug: Invalid bounded params"
        assert isinstance(number_of_params.value, int), "Compiler Bug: Invalid number of params"
        bounded_params = BoundedParams(number_of_params.value)
        bounded_params._program_code = context.input_str[context.start_position:context.end_position]
        return bounded_params
    
    def __init__(self, number_of_params: int):
        super().__init__()
        assert len(self._children) == 0, "Bounded params should not have any children"
        assert isinstance(number_of_params, int), "Compiler Bug: Number of params should be an integer"
        assert number_of_params >= 0, "Compiler Bug: Number of params should be greater than or equal to 0"
        self._number_of_params = number_of_params
    
    @property
    def params(self) -> int:
        return self._number_of_params
    
    def add_child(self, child):
        raise Exception("Compiler Bug: Bounded params should not have any children")

class FunctionDeclarations(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'FunctionDeclarations':
        if isinstance(nodes, list):
            if len(nodes) == 0:
                return None
            assert len(nodes) >= 1, "Compiler Bug: Invalid function declaration"
            assert isinstance(nodes[0], FunctionDeclaration), "Compiler Bug: Invalid function declaration"
            function_declarations = FunctionDeclarations()
            function_declarations._program_code = context.input_str[context.start_position:context.end_position]
            function_declarations.add_child(nodes[0])
            if len(nodes) > 1:
                assert len(nodes) == 2, "Compiler Bug: Invalid function declaration"
                func_decls = nodes[1]
                if isinstance(func_decls, FunctionDeclarations):
                    for child in func_decls.children:
                        assert isinstance(child, FunctionDeclaration), "Compiler Bug: Invalid function declaration"
                        function_declarations.add_child(child)
                all_function_names = set([child.function_name for child in function_declarations.children])
                if len(all_function_names) != len(function_declarations.children):
                    raise Exception(f"User Error: Duplicate function names found at {get_error_code_str(context)}")
            return function_declarations
        elif isinstance(nodes, FunctionDeclaration):
            function_declarations = FunctionDeclarations()
            function_declarations._program_code = context.input_str[context.start_position:context.end_position]
            function_declarations.add_child(nodes)
            return function_declarations
        else:
            raise Exception("Compiler Bug: Invalid function declaration")
    
    def smt2(self):
        if len(self._children) == 0:
            return ""
        else:
            fun_decls = []
            for child in self._children:
                assert isinstance(child, FunctionDeclaration), "Compiler Bug: Invalid function declaration"
                assert isinstance(child.sub_program, Z3Program), "Compiler Bug: Invalid function declaration"
                assert isinstance(child.sub_program.name, str), "Compiler Bug: Invalid function declaration"
                fun_name = child.sub_program.name
                sub_program_expr = child.sub_program.return_expr()
                assert isinstance(sub_program_expr, str), "Compiler Bug: Invalid function declaration"
                assert len(sub_program_expr) > 0, "Compiler Bug: Invalid function declaration"
                num_params = child.sub_program.params
                parameter_list = " ".join([f"(x_{idx} Int)" for idx in range(num_params)])
                decl_template = f"(define-fun {fun_name} ({parameter_list}) Int {sub_program_expr})"
                fun_decls.append(decl_template)
            return "\n".join(fun_decls)

class PredicateDeclarations(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'PredicateDeclarations':
        if isinstance(nodes, list):
            if len(nodes) == 0:
                return None
            assert len(nodes) >= 1, "Compiler Bug: Invalid predicate declaration"
            assert isinstance(nodes[0], PredicateDeclaration), "Compiler Bug: Invalid predicate declaration"
            predicate_declarations = PredicateDeclarations()
            predicate_declarations._program_code = context.input_str[context.start_position:context.end_position]
            predicate_declarations.add_child(nodes[0])
            if len(nodes) > 1:
                assert len(nodes) == 2, "Compiler Bug: Invalid predicate declaration"
                pred_decls = nodes[1]
                if isinstance(pred_decls, PredicateDeclarations):
                    for child in pred_decls.children:
                        assert isinstance(child, PredicateDeclaration), "Compiler Bug: Invalid predicate declaration"
                        predicate_declarations.add_child(child)
                all_predicate_names = set([child.predicate_name for child in predicate_declarations.children])
                if len(all_predicate_names) != len(predicate_declarations.children):
                    raise Exception(f"User Error: Duplicate predicate names found at {get_error_code_str(context)}")
            return predicate_declarations
        elif isinstance(nodes, PredicateDeclaration):
            predicate_declarations = PredicateDeclarations()
            predicate_declarations._program_code = context.input_str[context.start_position:context.end_position]
            predicate_declarations.add_child(nodes)
            return predicate_declarations
        else:
            raise Exception("Compiler Bug: Invalid predicate declaration")
    
    def smt2(self):
        if len(self._children) == 0:
            return ""
        else:
            pred_decls = []
            for child in self._children:
                assert isinstance(child, PredicateDeclaration), "Compiler Bug: Invalid predicate declaration"
                assert isinstance(child.sub_program, Z3Program), "Compiler Bug: Invalid predicate declaration"
                assert isinstance(child.sub_program.name, str), "Compiler Bug: Invalid predicate declaration"
                pred_name = child.sub_program.name
                sub_program_pred = child.sub_program.return_pred()
                assert isinstance(sub_program_pred, str), "Compiler Bug: Invalid predicate declaration"
                assert len(sub_program_pred) > 0, "Compiler Bug: Invalid predicate declaration"
                num_params = child.sub_program.params
                parameter_list = " ".join([f"(x_{idx} Int)" for idx in range(num_params)])
                decl_template = f"(define-fun {pred_name} ({parameter_list}) Bool {sub_program_pred})"
                pred_decls.append(decl_template)
            return "\n".join(pred_decls)

class Expr(Z3ComposableDslParseNode):
    def parse(context, nodes):
        assert isinstance(nodes, list), "Compiler Bug: Invalid expr"
        assert len(nodes) == 1, "Compiler Bug: Invalid expr"
        if isinstance(nodes[0], str):
            if nodes[0] == "None":
                return None
            else:
                raise Exception(f"Compiler Bug: Unexpected variable name at {get_error_code_str(context)}")
        assert isinstance(nodes[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid expr"
        expr = Expr()
        expr._program_code = context.input_str[context.start_position:context.end_position]
        expr.add_child(nodes[0])
        return expr

    def return_expr(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid expr"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid expr"
        if isinstance(self._children[0], LeafNode):
            if self._children[0].value == "None":
                error_code_str = self._children[0]._program_code
                raise Exception("Runtime Error: The arithmetic program {} returns None".format(error_code_str))
        return self._children[0].return_expr()

class PredExpr(Z3ComposableDslParseNode):
    def parse(context, nodes):
        assert isinstance(nodes, list), "Compiler Bug: Invalid pred expr"
        assert len(nodes) == 1, "Compiler Bug: Invalid pred expr"
        if isinstance(nodes[0], str):
            if nodes[0] == "None":
                return None
            else:
                raise Exception(f"Compiler Bug: Unexpected variable name at {get_error_code_str(context)}")
        assert isinstance(nodes[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid pred expr"
        pred_expr = PredExpr()
        pred_expr._program_code = context.input_str[context.start_position:context.end_position]
        pred_expr.add_child(nodes[0])
        return pred_expr

    def return_pred(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid pred expr"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid pred expr"
        if isinstance(self._children[0], LeafNode):
            if self._children[0].value == "None":
                error_code_str = self._children[0]._program_code
                raise Exception("Runtime Error: The predicate program {} returns None".format(error_code_str))
        return self._children[0].return_pred()

class PredicateDeclaration(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'PredicateDeclaration':
        assert isinstance(nodes, list), "Compiler Bug: Invalid predicate declaration"
        assert len(nodes) == 7, "Compiler Bug: Invalid predicate declaration"
        predicate_name = nodes[0]
        assert isinstance(predicate_name, LeafNode), "Compiler Bug: Invalid predicate declaration"
        assert isinstance(nodes[4], Z3Program), "Compiler Bug: Invalid predicate declaration"
        pred_decl = PredicateDeclaration(predicate_name.value, nodes[4])
        pred_decl._program_code = context.input_str[context.start_position:context.end_position]
        return pred_decl
    
    def __init__(self, predicate_name: str, program: 'Z3Program'):
        super().__init__()
        assert len(self._children) == 0, "Predicate declaration should not have any children"
        assert isinstance(predicate_name, str), "Compiler Bug: Predicate name should be a string"
        assert isinstance(program, Z3Program), "Compiler Bug: Predicate declaration should have a Z3Program"
        assert predicate_name is not None, "Compiler Bug: Predicate name should not be None"
        assert predicate_name.startswith("p"), "Compiler Bug: Predicate name should start with p and followed by digits"
        self.predicate_name = predicate_name
        self.sub_program = program

class FunctionDeclaration(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'FunctionDeclaration':
        assert isinstance(nodes, list), "Compiler Bug: Invalid function declaration"
        assert len(nodes) == 7, "Compiler Bug: Invalid function declaration"
        function_name = nodes[0]
        assert isinstance(function_name, LeafNode), "Compiler Bug: Invalid function declaration"
        assert isinstance(nodes[4], Z3Program), "Compiler Bug: Invalid function declaration"
        func_decl = FunctionDeclaration(function_name.value, nodes[4])
        func_decl._program_code = context.input_str[context.start_position:context.end_position]
        return func_decl
    
    def __init__(self, function_name: str, program: 'Z3Program'):
        super().__init__()
        assert len(self._children) == 0, "Function declaration should not have any children"
        assert isinstance(function_name, str), "Compiler Bug: Function name should be a string"
        assert isinstance(program, Z3Program), "Compiler Bug: Function declaration should have a Z3Program"
        assert function_name is not None, "Compiler Bug: Function name should not be None"
        assert function_name.startswith("f"), "Compiler Bug: Function name should start with f and followed by digits"
        self.function_name = function_name
        self.sub_program = program

class ArithExprSimple(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithExprSimple':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith expr simple"
        assert len(nodes) == 1, "Compiler Bug: Invalid arith expr simple"
        if isinstance(nodes[0], str):
            if nodes[0] == "None":
                return None
            else:
                raise Exception(f"Compiler Bug: Unexpected variable name at {get_error_code_str(context)}")
        assert isinstance(nodes[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr simple"
        arith_expr_simple = ArithExprSimple()
        arith_expr_simple._program_code = context.input_str[context.start_position:context.end_position]
        arith_expr_simple.add_child(nodes[0])
        return arith_expr_simple

    def return_expr(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid arith expr simple"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr simple"
        return self._children[0].return_expr()

    def return_pred(self):
        return self.return_expr()

class ArithExpr(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithExpr':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith expr"
        assert len(nodes) == 1, "Compiler Bug: Invalid arith expr"
        if isinstance(nodes[0], str):
            if nodes[0] == "None":
                return None
            else:
                raise Exception(f"Compiler Bug: Unexpected variable name at {get_error_code_str(context)}")
        assert isinstance(nodes[0], Z3ComposableDslParseNode), f"Compiler Bug: Invalid arith expr {get_error_code_str(context)}"
        arith_expr = ArithExpr()
        arith_expr._program_code = context.input_str[context.start_position:context.end_position]
        arith_expr.add_child(nodes[0])
        return arith_expr
    
    def return_expr(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid arith expr"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr"
        return self._children[0].return_expr()
    
    def return_pred(self):
        return self.return_expr()

class ArithExprFunc(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithExprFunc':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith expr func"
        assert len(nodes) >= 3 and len(nodes) <= 4, "Compiler Bug: Invalid arith expr func"
        assert isinstance(nodes[0], LeafNode), "Compiler Bug: Invalid arith expr func"
        variable_name = nodes[0].value
        assert isinstance(variable_name, str), "Compiler Bug: Invalid arith expr func"
        assert variable_name.startswith("f"), "Compiler Bug: Invalid arith expr func"
        assert variable_name[2:].isdigit(), "Compiler Bug: Invalid arith expr func"
        if len(nodes) == 4:
            assert isinstance(nodes[2], ArithBindSeq), "Compiler Bug: Invalid arith expr func"
            arith_expr_func = ArithExprFunc(variable_name, nodes[2])
            arith_expr_func._program_code = context.input_str[context.start_position:context.end_position]
            arith_expr_func.add_child(nodes[2])
            return arith_expr_func
        else:
            arith_bind_seq = ArithBindSeq()
            arith_expr_func = ArithExprFunc(variable_name, arith_bind_seq)
            arith_expr_func._program_code = context.input_str[context.start_position:context.end_position]
            return arith_expr_func
    
    def __init__(self, variable_name: str, bind_seq: 'ArithBindSeq'):
        super().__init__()
        assert len(self._children) == 0, "ArithExprFunc should not have any children"
        self.variable_name = variable_name
        self.bind_seq = bind_seq

    def return_expr(self):
        assert isinstance(self.parent_program, Z3Program), "Compiler Bug: Parent executable should be a Z3Program"
        sub_program = self.parent_program.get_arith_sub_program(self.variable_name)
        assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
        assert isinstance(self.bind_seq, ArithBindSeq), "Compiler Bug: Bind seq should be an ArithBindSeq"
        param_index_map = {}
        for child in self.bind_seq.children:
            assert isinstance(child, ArithBind), "Compiler Bug: Bind seq should have ArithBind children"
            param_index = child.param_index
            assert isinstance(param_index, int), "Compiler Bug: Param index should be an integer"
            assert param_index >= 0, "Compiler Bug: Param index should be greater than or equal to 0"
            param_value = child.arith_expr.return_expr()
            param_index_map[param_index] = param_value
        all_param_indexes = list(range(sub_program.params))
        all_param_indexes.sort()
        num_params = sub_program.params
        for param_index in all_param_indexes:
            if param_index >= num_params:
                raise Exception(f"Runtime Error: The arithmetic program {self.variable_name} has no parameter at argument index {param_index}")
        call_params = []
        for index in range(num_params):
            if index not in param_index_map:
                call_params.append(f"x_{index}")
            else:
                call_params.append(f"{param_index_map[index]}")
        call_params_str = " ".join(call_params)
        assert isinstance(sub_program.name, str), "Compiler Bug: Sub program name should be a string"
        assert len(sub_program.name) > 0, "Compiler Bug: Sub program name should not be empty"
        if num_params == 0:
            return f"{sub_program.name}"
        else:
            return f"({sub_program.name} {call_params_str})"
    
    def return_pred(self):
        return self.return_expr()

class BinOp(Z3ComposableDslParseNode):
    def __init__(self, op: str):
        super().__init__()
        assert len(self._children) == 0, "BinOp should not have any children"
        assert isinstance(op, str), "Operator should be a string"
        assert op is not None, "Operator should not be None"
        self.op = op
    
    def return_expr(self):
        assert len(self._children) == 3, "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[1], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[2], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr op"
        op = self._children[0].return_expr()
        left_expr = self._children[1].return_expr()
        right_expr = self._children[2].return_expr()
        return f"({op} {left_expr} {right_expr})"
    
    def return_pred(self):
        assert len(self._children) == 3, "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[1], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr op"
        assert isinstance(self._children[2], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr op"
        op = self._children[0].return_pred()
        left_expr = self._children[1].return_pred()
        right_expr = self._children[2].return_pred()
        return f"({op} {left_expr} {right_expr})"

class ArithExprOp(BinOp):
    def parse(context: LRStackNode, nodes) -> 'ArithExprOp':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith expr op"
        assert len(nodes) == 3, "Compiler Bug: Invalid arith expr op"
        # assert isinstance(nodes[0], ArithExpr), "Compiler Bug: Invalid arith expr op"
        # assert isinstance(nodes[2], ArithExpr), "Compiler Bug: Invalid arith expr op"
        assert isinstance(nodes[1], str), "Compiler Bug: Invalid arith expr op"
        ops_available = ["+", "-", "*", "/", "%", "^"]
        assert nodes[1] in ops_available, f"Compiler Bug: Invalid arith operator {nodes[1].value}"
        arith_expr_op = ArithExprOp(nodes[1])
        arith_expr_op._program_code = context.input_str[context.start_position:context.end_position]
        if nodes[1] == "/":
            nodes[1] = "div"
        elif nodes[1] == "%":
            nodes[1] = "mod"
        nodes[1] = LeafNode(nodes[1], True)
        arith_expr_op.add_child(nodes[1])
        arith_expr_op.add_child(nodes[0])
        arith_expr_op.add_child(nodes[2])
        return arith_expr_op
    
    def __init__(self, op: str):
        super().__init__(op)
        assert op in ["+", "-", "*", "/", "%", "^"], f"Compiler Bug: Invalid arith operator {op}"
        self.op = op

class ArithExprCompund(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithExprCompund':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith expr compound"
        assert len(nodes) == 3, "Compiler Bug: Invalid arith expr compound"
        assert isinstance(nodes[0], LeafNode), "Compiler Bug: Invalid arith expr compound"
        assert isinstance(nodes[1], ArithExpr), "Compiler Bug: Invalid arith expr compound"
        assert isinstance(nodes[2], LeafNode), "Compiler Bug: Invalid arith expr compound"
        assert nodes[0].value == '(', "Compiler Bug: Invalid arith expr compound"
        assert nodes[2].value == ')', "Compiler Bug: Invalid arith expr compound"
        expr = ArithExprCompund()
        expr._program_code = context.input_str[context.start_position:context.end_position]
        expr.add_child(nodes[1])
        return expr
    
    def return_expr(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid arith expr compound"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith expr compound"
        return f"({self._children[0].return_expr()})"
    
    def return_pred(self):
        return self.return_expr()

class LogicalExpr(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExpr':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr"
        assert len(nodes) == 1, "Compiler Bug: Invalid logical expr"
        if isinstance(nodes[0], str):
            if nodes[0] == "None":
                return None
            else:
                raise Exception(f"Compiler Bug: Unexpected variable name at {get_error_code_str(context)}")
        assert isinstance(nodes[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid logical expr"
        logical_expr = LogicalExpr()
        logical_expr._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr.add_child(nodes[0])
        return logical_expr

    def return_pred(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid logical expr"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid logical expr"
        if isinstance(self._children[0], LeafNode):
            if self._children[0].value == "None":
                error_code_str = self._children[0]._program_code
                raise Exception("Runtime Error: The logical program {} returns None".format(error_code_str))
        return self._children[0].return_pred()

class LogicalExprOp(BinOp):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprOp':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr op"
        assert len(nodes) == 3, "Compiler Bug: Invalid logical expr op"
        # assert isinstance(nodes[0], LogicalExpr), "Compiler Bug: Invalid logical expr op"
        # assert isinstance(nodes[2], LogicalExpr), "Compiler Bug: Invalid logical expr op"
        assert isinstance(nodes[1], str), "Compiler Bug: Invalid logical expr op"
        ops_available = ["==", ">", "<", ">=", "<=", "==", "!="]
        assert nodes[1] in ops_available, f"Compiler Bug: Invalid logical operator {nodes[1].value}"
        logical_expr_op = LogicalExprOp(nodes[1])
        logical_expr_op._program_code = context.input_str[context.start_position:context.end_position]
        if nodes[1] == "==":
            nodes[1] = "="
        nodes[1] = LeafNode(nodes[1], True)
        logical_expr_op.add_child(nodes[1])
        logical_expr_op.add_child(nodes[0])
        logical_expr_op.add_child(nodes[2])
        return logical_expr_op

    def __init__(self, op: str):
        super().__init__(op)
        assert op in ["==", ">", "<", ">=", "<=", "==", "!="], f"Compiler Bug: Invalid logical operator {op}"
        self.op = op

class LogicalExprQuantifier(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprQuantifier':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr quantifier"
        assert len(nodes) == 8, "Compiler Bug: Invalid logical expr quantifier"
        assert isinstance(nodes[0], str), "Compiler Bug: Invalid logical expr quantifier"
        nodes[0] = LeafNode(nodes[0], True)
        assert nodes[0].value in ["ForAll", "Exists"], f"Compiler Bug: Invalid logical expr quantifier {nodes[0].value}"
        assert isinstance(nodes[6], Z3ComposableDslParseNode), "Compiler Bug: Invalid logical expr quantifier"
        assert isinstance(nodes[3], BoundedVarSeq), "Compiler Bug: Invalid logical expr quantifier"
        logical_expr_quantifier = LogicalExprQuantifier()
        logical_expr_quantifier._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr_quantifier.add_child(nodes[0])
        logical_expr_quantifier.add_child(nodes[3])
        logical_expr_quantifier.add_child(nodes[6])
        return logical_expr_quantifier
    
    def return_pred(self):
        assert len(self._children) == 3, "Compiler Bug: Invalid logical expr quantifier"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid logical expr quantifier"
        assert isinstance(self._children[1], BoundedVarSeq), "Compiler Bug: Invalid logical expr quantifier"
        assert isinstance(self._children[2], Z3ComposableDslParseNode), "Compiler Bug: Invalid logical expr quantifier"
        quantifier = self._children[0].return_pred()
        if quantifier == "ForAll":
            quantifier = "forall"
        elif quantifier == "Exists":
            quantifier = "exists"
        else:
            raise Exception(f"Compiler Bug: Invalid logical expr quantifier {quantifier}")
        bounded_vars = []
        bounded_var_seq = self._children[1]
        assert isinstance(bounded_var_seq, BoundedVarSeq), "Compiler Bug: Invalid logical expr quantifier"
        for child in bounded_var_seq.children:
            assert isinstance(child, LeafNode), "Compiler Bug: Invalid logical expr quantifier"
            assert child.is_bounded, "Compiler Bug: Invalid logical expr quantifier"
            assert isinstance(child.value, str), "Compiler Bug: Invalid logical expr quantifier"
            assert child.value.startswith("b") and child.value[2:].isdigit(), "Compiler Bug: Invalid logical expr quantifier"
            bounded_vars.append(child.value)
        parent_program = self.parent_program
        assert isinstance(parent_program, Z3Program), f"Compiler Bug: Parent program should be a Z3Program at {self._program_code}"
        # assert len(bounded_vars) == parent_program.bounded_params.params, f"User Error: The number of bounded variables {len(bounded_vars)} does not match the specified number of bounded parameters {parent_program.bounded_params.params} at {self._program_code}"
        bound_vars = " ".join([f"({var} Int)" for var in bounded_vars])
        logical_expr = self._children[2].return_pred()
        if quantifier == "forall":
            all_vars_gtr_than_zero = ' '.join([f'(<= 0 {var})' for var in bounded_vars])
            logical_expr = f"(=> {all_vars_gtr_than_zero} {logical_expr})"
        elif quantifier == "exists":
            all_vars_gtr_than_zero = ' '.join([f'(<= 0 {var})' for var in bounded_vars])
            logical_expr = f"(and {all_vars_gtr_than_zero} {logical_expr})"
        else:
            raise Exception(f"Compiler Bug: Invalid logical expr quantifier {quantifier}")
        return f"({quantifier} ({bound_vars}) {logical_expr})"

class LogicalExprBoolean(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprBoolean':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr boolean"
        assert len(nodes) == 4, "Compiler Bug: Invalid logical expr boolean"
        logical_seq_node = nodes[2]
        assert isinstance(logical_seq_node, LogicalSeq), "Compiler Bug: Invalid logical expr boolean"
        assert isinstance(nodes[0], str), "Compiler Bug: Invalid logical expr boolean"
        nodes[0] = LeafNode(nodes[0], True)
        op_type = nodes[0]
        assert isinstance(op_type, LeafNode), "Compiler Bug: Invalid logical expr boolean"
        assert op_type.value in ["And", "Or", "Implies"], f"Compiler Bug: Invalid logical expr boolean {op_type.value}"
        logical_expr_boolean = LogicalExprBoolean()
        logical_expr_boolean._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr_boolean.add_child(op_type)
        for child in logical_seq_node.children:
            logical_expr_boolean.add_child(child)
        return logical_expr_boolean
    
    def return_pred(self):
        assert len(self._children) >= 3, "Compiler Bug: Invalid logical expr boolean"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid logical expr boolean"
        assert all(isinstance(child, LogicalExpr) for child in self._children[1:]), "Compiler Bug: Invalid logical expr boolean"
        op_type = self._children[0].return_pred()
        if op_type == "And":
            op_type = "and"
        elif op_type == "Or":
            op_type = "or"
        elif op_type == "Implies":
            op_type = "=>"
        else:
            raise Exception(f"Compiler Bug: Invalid logical expr boolean {op_type}")
        logical_exprs = []
        for child in self._children[1:]:
            assert isinstance(child, LogicalExpr), "Compiler Bug: Invalid logical expr boolean"
            logical_expr = child.return_pred()
            logical_exprs.append(logical_expr)
        logical_exprs_str = " ".join(logical_exprs)
        return f"({op_type} {logical_exprs_str})"

class LogicalExprNot(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprNot':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr not"
        assert len(nodes) == 4, f"Compiler Bug: Invalid logical expr not of length {len(nodes)} ({context.input_str[context.start_position:context.end_position]})"
        assert isinstance(nodes[0], str), "Compiler Bug: Invalid logical expr not"
        nodes[0] = LeafNode(nodes[0], True)
        assert nodes[0].value == "Not", "Compiler Bug: Invalid logical expr not"
        logical_expr_not = LogicalExprNot()
        logical_expr_not._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr_not.add_child(nodes[0])
        logical_expr_not.add_child(nodes[2])
        return logical_expr_not
    
    def return_pred(self):
        assert len(self._children) == 2, "Compiler Bug: Invalid logical expr not"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid logical expr not"
        assert isinstance(self._children[1], LogicalExpr), "Compiler Bug: Invalid logical expr not"
        op_type = self._children[0].return_pred()
        if op_type == "Not":
            op_type = "not"
        else:
            raise Exception(f"Compiler Bug: Invalid logical expr not {op_type}")
        logical_expr = self._children[1].return_pred()
        return f"({op_type} {logical_expr})"

class LogicalExprIsMember(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprIsMember':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr is member"
        assert len(nodes) == 6, "Compiler Bug: Invalid logical expr is member"
        assert isinstance(nodes[0], str), "Compiler Bug: Invalid logical expr is member"
        nodes[0] = LeafNode(nodes[0], True)
        assert nodes[0].value == "IsMember", "Compiler Bug: Invalid logical expr is member"
        assert isinstance(nodes[2], ArithExpr), "Compiler Bug: Invalid logical expr is member"
        assert isinstance(nodes[4], LeafNode), "Compiler Bug: Invalid logical expr is member"
        assert nodes[4].value == "NatSet", "Compiler Bug: Invalid logical expr is member"
        logical_expr_is_member = LogicalExprIsMember()
        logical_expr_is_member._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr_is_member.add_child(nodes[0])
        logical_expr_is_member.add_child(nodes[2])
        logical_expr_is_member.add_child(nodes[4])
        return logical_expr_is_member
    
    def return_pred(self):
        assert len(self._children) == 3, "Compiler Bug: Invalid logical expr is member"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid logical expr is member"
        assert isinstance(self._children[1], ArithExpr), "Compiler Bug: Invalid logical expr is member"
        assert isinstance(self._children[2], LeafNode), "Compiler Bug: Invalid logical expr is member"
        op_type = self._children[0].return_pred()
        return f"({op_type} {self._children[1].return_expr()} {self._children[2].return_pred()})"

class LogicalExprTrueFalse(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprTrueFalse':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr true false"
        assert len(nodes) == 2, "Compiler Bug: Invalid logical expr true false"
        assert isinstance(nodes[0], str), "Compiler Bug: Invalid logical expr true false"
        nodes[0] = LeafNode(nodes[0], True)
        assert nodes[0].value in ["True", "False"], "Compiler Bug: Invalid logical expr true false"
        logical_expr_true_false = LogicalExprTrueFalse()
        logical_expr_true_false._program_code = context.input_str[context.start_position:context.end_position]
        logical_expr_true_false.add_child(nodes[0])
        return logical_expr_true_false

    def return_pred(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid logical expr true false"
        assert isinstance(self._children[0], LeafNode), "Compiler Bug: Invalid logical expr true false"
        assert self._children[0].value in ["True", "False"], "Compiler Bug: Invalid logical expr true false"
        if self._children[0].value == "True":
            return "true"
        elif self._children[0].value == "False":
            return "false"
        else:
            raise Exception(f"Compiler Bug: Invalid logical expr true false {self._children[0].value}") 

class LogicalExprPred(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalExprPred':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical expr pred"
        assert len(nodes) >= 3 and len(nodes) <= 4, f"Compiler Bug: Invalid logical expr pred of length {len(nodes)} ({context.input_str[context.start_position:context.end_position]})"
        assert isinstance(nodes[0], LeafNode), "Compiler Bug: Invalid logical expr pred"
        variable_name = nodes[0].value
        assert isinstance(variable_name, str), "Compiler Bug: Invalid logical expr pred"
        assert variable_name.startswith("p"), "Compiler Bug: Invalid logical expr pred"
        assert variable_name[2:].isdigit(), "Compiler Bug: Invalid logical expr pred"
        if len(nodes) == 4:
            assert isinstance(nodes[2], ArithBindSeq), f"Compiler Bug: Invalid logical expr pred"
            logical_expr_pred = LogicalExprPred(variable_name, nodes[2])
            logical_expr_pred._program_code = context.input_str[context.start_position:context.end_position]
            logical_expr_pred.add_child(nodes[2])
            return logical_expr_pred
        else:
            binding_seq = ArithBindSeq()
            logical_expr_pred = LogicalExprPred(variable_name, binding_seq)
            logical_expr_pred._program_code = context.input_str[context.start_position:context.end_position]
            return logical_expr_pred
    
    def __init__(self, variable_name: str, bind_seq: 'ArithBindSeq'):
        super().__init__()
        assert len(self._children) == 0, "LogicalExprPred should not have any children"
        self.variable_name = variable_name
        self.bind_seq = bind_seq
    
    def return_pred(self):
        assert isinstance(self.parent_program, Z3Program), "Compiler Bug: Parent executable should be a Z3Program"
        sub_program = self.parent_program.get_predicate_sub_program(self.variable_name)
        assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
        assert isinstance(self.bind_seq, ArithBindSeq), "Compiler Bug: Bind seq should be an ArithBindSeq"
        param_index_map = {}
        for child in self.bind_seq.children:
            assert isinstance(child, ArithBind), "Compiler Bug: Bind seq should have ArithBind children"
            param_index = child.param_index
            assert isinstance(param_index, int), "Compiler Bug: Param index should be an integer"
            assert param_index >= 0, "Compiler Bug: Param index should be greater than or equal to 0"
            param_value = child.arith_expr.return_expr()
            param_index_map[param_index] = param_value
        all_param_indexes = list(range(sub_program.params))
        all_param_indexes.sort()
        num_params = sub_program.params
        for param_index in all_param_indexes:
            if param_index >= num_params:
                raise Exception(f"Runtime Error: The predicate program {self.variable_name} has no parameter at argument index {param_index}")
        call_params = []
        for index in range(num_params):
            if index not in param_index_map:
                call_params.append(f"x_{index}")
            else:
                call_params.append(f"{param_index_map[index]}")
        call_params_str = " ".join(call_params)
        assert isinstance(sub_program.name, str), "Compiler Bug: Sub program name should be a string"
        assert len(sub_program.name) > 0, "Compiler Bug: Sub program name should not be empty"
        if num_params == 0:
            return f"{sub_program.name}"
        else:
            return f"({sub_program.name} {call_params_str})"

class BoundedVarSeq(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'BoundedVarSeq':
        assert isinstance(nodes, list), "Compiler Bug: Invalid bounded var seq"
        assert len(nodes) == 1 or len(nodes) == 3, "Compiler Bug: Invalid bounded var seq"
        assert isinstance(nodes[0], LeafNode), "Compiler Bug: Invalid bounded var seq"
        assert nodes[0].is_bounded, "Compiler Bug: Invalid bounded var seq"
        assert isinstance(nodes[0].value, str), "Compiler Bug: Invalid bounded var seq"
        assert nodes[0].value.startswith("b") and nodes[0].value[2:].isdigit(), "Compiler Bug: Invalid bounded var seq"
        if len(nodes) == 1:
            bounded_var_seq = BoundedVarSeq()
            bounded_var_seq._program_code = context.input_str[context.start_position:context.end_position]
            bounded_var_seq.add_child(nodes[0])
            return bounded_var_seq
        elif len(nodes) == 3:
            bounded_var_seq = BoundedVarSeq()
            bounded_var_seq._program_code = context.input_str[context.start_position:context.end_position]
            bounded_var_seq.add_child(nodes[0])
            for child in nodes[2].children:
                bounded_var_seq.add_child(child)
            return bounded_var_seq
        else:
            raise Exception("Compiler Bug: Invalid bounded var seq")
    
    def __init__(self):
        super().__init__()
        assert len(self._children) == 0, "BoundedVarSeq should not have any children"

class LogicalSeq(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'LogicalSeq':
        assert isinstance(nodes, list), "Compiler Bug: Invalid logical seq"
        assert len(nodes) == 1 or len(nodes) == 3, "Compiler Bug: Invalid logical seq"
        if len(nodes) == 1:
            assert isinstance(nodes[0], LogicalExpr), "Compiler Bug: Invalid logical seq"
            logical_seq = LogicalSeq()
            logical_seq._program_code = context.input_str[context.start_position:context.end_position]
            logical_seq.add_child(nodes[0])
            return logical_seq
        elif len(nodes) == 3:
            assert isinstance(nodes[0], LogicalExpr), "Compiler Bug: Invalid logical seq"
            assert isinstance(nodes[2], LogicalSeq), "Compiler Bug: Invalid logical seq"
            logical_seq = LogicalSeq()
            logical_seq._program_code = context.input_str[context.start_position:context.end_position]
            logical_seq.add_child(nodes[0])
            for child in nodes[2].children:
                logical_seq.add_child(child)
            return logical_seq
        else:
            raise Exception("Compiler Bug: Invalid logical seq")
    
    def __init__(self):
        super().__init__()
        assert len(self._children) == 0, "LogicalSeq should not have any children"

class ArithVar(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithVar':
        assert isinstance(nodes, list), "Compiler Bug: Invalid arith var"
        assert len(nodes) == 1, "Compiler Bug: Invalid arith var"
        assert isinstance(nodes[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith var" 
        arith_var = ArithVar()
        arith_var._program_code = context.input_str[context.start_position:context.end_position]
        arith_var.add_child(nodes[0])
        return arith_var
    
    def return_expr(self):
        assert len(self._children) == 1, "Compiler Bug: Invalid arith var"
        assert isinstance(self._children[0], Z3ComposableDslParseNode), "Compiler Bug: Invalid arith var"
        return self._children[0].return_expr()
    
    def return_pred(self):
        return self.return_expr()

class ArithBind(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithBind':
        assert isinstance(nodes, list), "Compiler Bug: Invalid bind"
        assert len(nodes) == 3, "Compiler Bug: Invalid bind"
        assert isinstance(nodes[0], LeafNode), "Compiler Bug: Invalid bind"
        inp_var = nodes[0].value
        if isinstance(nodes[2], ArithExprSimple) or \
            isinstance(nodes[2], ArithExprCompund) or \
            isinstance(nodes[2], ArithExprFunc) or \
            isinstance(nodes[2], ArithExprOp) or \
            isinstance(nodes[2], ArithExpr):
            bind = ArithBind(inp_var, nodes[2])
            bind._program_code = context.input_str[context.start_position:context.end_position]
            bind.add_child(nodes[2].children[0])
            return bind
        elif isinstance(nodes[2], LogicalExprPred):
            bind = ArithBind(inp_var, nodes[2])
            bind._program_code = context.input_str[context.start_position:context.end_position]
            bind.add_child(nodes[2])
            return bind
        else:
            raise Exception(f"Compiler Bug: Invalid bind {type(nodes[2])}")
    
    def __init__(self, inp_var: str, expr: ArithExpr):
        super().__init__()
        assert len(self._children) == 0, "Bind should not have any children"
        assert isinstance(inp_var, str), "Compiler Bug: Input variable should be a string"
        assert inp_var is not None, "Compiler Bug: Input variable should not be None"
        assert inp_var.startswith("x") and inp_var[2:].isdigit(), "Compiler Bug: Input variable should start with x and followed by digits"
        self.param_name = inp_var
        self.param_index = int(inp_var[2:])
        self.arith_expr = expr

class ArithBindSeq(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'ArithBindSeq':
        assert isinstance(nodes, list), "Compiler Bug: Invalid bind seq"
        if len(nodes) == 1:
            if isinstance(nodes[0], ArithBind):
                bind_seq = ArithBindSeq()
                bind_seq._program_code = context.input_str[context.start_position:context.end_position]
                bind_seq.add_child(nodes[0])
                return bind_seq
            assert isinstance(nodes[0], ArithBindSeq), f"Compiler Bug: Invalid bind seq {type(nodes[0])}"
            bind_seq = ArithBindSeq()
            bind_seq._program_code = context.input_str[context.start_position:context.end_position]
            bind_seq.add_child(nodes[0])
            return bind_seq
        elif len(nodes) == 3:
            assert isinstance(nodes[0], ArithBind), "Compiler Bug: Invalid bind seq"
            assert isinstance(nodes[2], ArithBindSeq), "Compiler Bug: Invalid bind seq"
            bind_seq = ArithBindSeq()
            bind_seq._program_code = context.input_str[context.start_position:context.end_position]
            bind_seq.add_child(nodes[0])
            for child in nodes[2].children:
                bind_seq.add_child(child)
            return bind_seq
        else:
            raise Exception("Compiler Bug: Invalid bind seq")
    
    def __init__(self):
        super().__init__()
        assert len(self._children) == 0, "BindSeq should not have any children"

class Z3Program(Z3ComposableDslParseNode):
    def parse(context: LRStackNode, nodes) -> 'Z3Program':
        assert isinstance(nodes, list), "Compiler Bug: Invalid Z3 program"
        assert len(nodes) >= 10, "Compiler Bug: Invalid Z3 program"
        unbounded_params = nodes[0]
        assert isinstance(unbounded_params, UnboundedParams), f"Compiler Bug: Invalid Z3 program {type(unbounded_params)}"
        bounded_params = nodes[2]
        assert isinstance(bounded_params, BoundedParams), f"Compiler Bug: Invalid Z3 program {type(bounded_params)}"
        decl_idx = 4
        fun_decls: typing.Optional[FunctionDeclarations] = None
        if isinstance(nodes[decl_idx], FunctionDeclarations):
            fun_decls = nodes[decl_idx]
        decl_idx += 1
        pred_decls: typing.Optional[PredicateDeclarations] = None
        if isinstance(nodes[decl_idx], PredicateDeclarations):
            pred_decls = nodes[decl_idx]
        decl_idx += 1
        return_idx = decl_idx + 1
        expr = nodes[return_idx]
        if expr is not None:
            assert isinstance(expr, Expr), f"Compiler Bug: Invalid Z3 program {type(expr)}"
        return_idx += 3
        pred_expr = nodes[return_idx]
        if pred_expr is not None:
            assert isinstance(pred_expr, PredExpr), f"Compiler Bug: Invalid Z3 program {type(pred_expr)}"
        program = Z3Program()
        program._program_code = context.input_str[context.start_position:context.end_position]
        program.add_child(unbounded_params)
        program.add_child(bounded_params)
        program.add_child(fun_decls)
        program.add_child(pred_decls)
        program.add_child(expr)
        program.add_child(pred_expr)
        return program
    
    def __init__(self):
        super().__init__()
        self._name = None
        self.symbol_table : typing.Dict[str, 'Z3Program'] = {}
        self._populated = False
    
    @property
    def name(self):
        assert isinstance(self._name, str), "Compiler Bug: Name should be a string"
        assert len(self._name) > 0, "Compiler Bug: Name should not be empty"
        return self._name
    
    @property
    def bounded_params(self):
        assert len(self._children) >= 2, "Compiler Bug: Invalid Z3 program"
        assert isinstance(self._children[1], BoundedParams), "Compiler Bug: Invalid Z3 program"
        return self._children[1]
    
    @property
    def unbounded_params(self):
        assert len(self._children) >= 2, "Compiler Bug: Invalid Z3 program"
        assert isinstance(self._children[0], UnboundedParams), "Compiler Bug: Invalid Z3 program"
        return self._children[0]
    
    @property
    def function_decls(self):
        assert len(self._children) >= 4, "Compiler Bug: Invalid Z3 program"
        if isinstance(self._children[2], FunctionDeclarations):
            return self._children[2]
        else:
            return None
    
    @property
    def predicate_decls(self):
        assert len(self._children) >= 4, "Compiler Bug: Invalid Z3 program"
        if isinstance(self._children[3], PredicateDeclarations):
            return self._children[3]
        else:
            return None
        
    @property
    def expr(self):
        assert len(self._children) >= 5, "Compiler Bug: Invalid Z3 program"
        return self._children[4]
    
    @property
    def pred_expr(self):
        assert len(self._children) >= 6, "Compiler Bug: Invalid Z3 program"
        return self._children[5]
    
    @property
    def parent_program(self):
        assert isinstance(self._parent_executable, Z3Program), "Compiler Bug: Parent executable should be a Z3Program"
        return self._parent_executable
    
    def _add_all_sub_programs_to_symbol_table_recursive(self, idx: int = 0) -> int:
        idx = self._add_all_sub_programs_to_symbol_table(idx)
        if self.function_decls is not None:
            for func_decl in self.function_decls.children:
                assert isinstance(func_decl, FunctionDeclaration), "Compiler Bug: Function declaration should be a FunctionDeclaration"
                function_prog = func_decl.sub_program
                idx = function_prog._add_all_sub_programs_to_symbol_table_recursive(idx)
                for _, sub_program in function_prog.symbol_table.items():
                    assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
                    self.symbol_table[sub_program.name] = sub_program
                idx += 1
        if self.predicate_decls is not None:
            for pred_decl in self.predicate_decls.children:
                assert isinstance(pred_decl, PredicateDeclaration), "Compiler Bug: Predicate declaration should be a PredicateDeclaration"
                predicate_prog = pred_decl.sub_program
                idx = predicate_prog._add_all_sub_programs_to_symbol_table_recursive(idx)
                for _, sub_program in predicate_prog.symbol_table.items():
                    assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
                    self.symbol_table[sub_program.name] = sub_program
                idx += 1
        self._populated = True
        return idx
    
    def _assign_parent_executable(self):
        assert self._populated, "Compiler Bug: Symbol table should be populated"
        # for _, sub_program in self.symbol_table.items():
        #     assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
        #     sub_program._parent_executable = self
        self._set_parent_program_recursive()

    def _set_parent_program_recursive(self):
        all_sub_programs = [sub_program for _, sub_program in self.symbol_table.items()]
        stack = [self] + all_sub_programs
        while len(stack) > 0:
            node: Z3ComposableDslParseNode = stack.pop()
            if node is None:
                continue
            node._parent_executable = self
            for child in node.children:
                stack.append(child)

    def _add_all_sub_programs_to_symbol_table(self, idx: int = 0):
        if self.function_decls is not None:
            for func_decl in self.function_decls.children:
                assert isinstance(func_decl, FunctionDeclaration), "Compiler Bug: Function declaration should be a FunctionDeclaration"
                function_name = func_decl.function_name
                function_prog = func_decl.sub_program
                old_name = function_name
                new_name = f"{function_name}_r{idx}"
                idx += 1
                function_prog._rename(new_name)
                self._rename_sub_program(old_name, new_name)
                func_decl.function_name = new_name
                self.symbol_table[new_name] = function_prog
        if self.predicate_decls is not None:
            for pred_decl in self.predicate_decls.children:
                assert isinstance(pred_decl, PredicateDeclaration), "Compiler Bug: Predicate declaration should be a PredicateDeclaration"
                predicate_name = pred_decl.predicate_name
                predicate_prog = pred_decl.sub_program
                old_name = predicate_name
                new_name = f"{predicate_name}_r{idx}"
                idx += 1
                predicate_prog._rename(new_name)
                self._rename_sub_program(old_name, new_name)
                pred_decl.predicate_name = new_name
                self.symbol_table[new_name] = predicate_prog
        return idx
    
    def _rename(self, name: str):
        assert isinstance(name, str), "Compiler Bug: Name should be a string"
        assert len(name) > 0, "Compiler Bug: Name should not be empty"
        self._name = name
        pass
    
    def _rename_sub_program(self, name: str, new_name: str):
        assert isinstance(name, str), "Compiler Bug: Name should be a string"
        assert len(name) > 0, "Compiler Bug: Name should not be empty"
        assert isinstance(new_name, str), "Compiler Bug: New name should be a string"
        assert len(new_name) > 0, "Compiler Bug: New name should not be empty"
        self._rename_in_expr(name, new_name)
        self._rename_in_pred_expr(name, new_name)
        self._rename_in_fun_decl(name, new_name)
        self._rename_in_pred_decl(name, new_name)

    def _rename_in_expr(self, name: str, new_name: str):
        stack = []
        stack.append(self.expr)
        while len(stack) > 0:
            node : Z3ComposableDslParseNode = stack.pop()
            if node is None:
                continue
            for child in node.children:
                stack.append(child)
            if isinstance(node, LeafNode):
                if not node.is_bounded and node.value == name:
                    node.value = new_name
            elif isinstance(node, ArithExprFunc):
                if node.variable_name == name:
                    node.variable_name = new_name

    def _rename_in_pred_expr(self, name: str, new_name: str):
        stack = []
        stack.append(self.pred_expr)
        while len(stack) > 0:
            node : Z3ComposableDslParseNode = stack.pop()
            if node is None:
                continue
            for child in node.children:
                stack.append(child)
            if isinstance(node, LeafNode):
                if not node.is_bounded and node.value == name:
                    node.value = new_name
            elif isinstance(node, ArithExprFunc):
                if node.variable_name == name:
                    node.variable_name = new_name
            elif isinstance(node, LogicalExprPred):
                if node.variable_name == name:
                    node.variable_name = new_name

    def _rename_in_fun_decl(self, name: str, new_name: str):
        stack = []
        stack.append(self.function_decls)
        while len(stack) > 0:
            node : Z3ComposableDslParseNode = stack.pop()
            if node is None:
                continue
            for child in node.children:
                stack.append(child)
            if isinstance(node, FunctionDeclaration):
                if node.function_name == name:
                    node.function_name = new_name
    
    def _rename_in_pred_decl(self, name: str, new_name: str):
        stack = []
        stack.append(self.predicate_decls)
        while len(stack) > 0:
            node : Z3ComposableDslParseNode = stack.pop()
            if node is None:
                continue
            for child in node.children:
                stack.append(child)
            if isinstance(node, PredicateDeclaration):
                if node.predicate_name == name:
                    node.predicate_name = new_name

    def get_arith_sub_program(self, variable_name: str) -> 'Z3Program':
        if not self._populated:
            self._add_all_sub_programs_to_symbol_table_recursive()
        assert isinstance(self.symbol_table, dict), "Compiler Bug: Symbol table should be a dictionary"
        assert variable_name in self.symbol_table, f"Compiler Bug: Variable {variable_name} not found in symbol table"
        sub_program = self.symbol_table[variable_name]
        assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
        assert isinstance(sub_program.name, str), "Compiler Bug: Sub program name should be a string"
        assert len(sub_program.name) > 0, "Compiler Bug: Sub program name should not be empty"
        assert sub_program.name == variable_name, f"Compiler Bug: Sub program name {sub_program.name} does not match variable name {variable_name}"
        return sub_program
    
    def get_predicate_sub_program(self, variable_name: str) -> 'Z3Program':
        if not self._populated:
            self._add_all_sub_programs_to_symbol_table_recursive()
        assert isinstance(self.symbol_table, dict), "Compiler Bug: Symbol table should be a dictionary"
        assert variable_name in self.symbol_table, f"Compiler Bug: Variable {variable_name} not found in symbol table"
        sub_program = self.symbol_table[variable_name]
        assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
        assert isinstance(sub_program.name, str), "Compiler Bug: Sub program name should be a string"
        assert len(sub_program.name) > 0, "Compiler Bug: Sub program name should not be empty"
        assert sub_program.name == variable_name, f"Compiler Bug: Sub program name {sub_program.name} does not match variable name {variable_name}"
        return sub_program

    def return_pred(self):
        return self.pred_expr.return_pred()
    
    def return_expr(self):
        return self.expr.return_expr()
    
    def smt2(self):
        if not self._populated:
            self._add_all_sub_programs_to_symbol_table_recursive()
        self._assign_parent_executable()
        self._name = "main"
        fun_decls = FunctionDeclarations()
        pred_decls = PredicateDeclarations()
        ordered_symbols = list(self.symbol_table.values())
        ordered_symbols.sort(key=lambda x: -1*int(x.name.split("_r")[-1]))
        for sub_program in ordered_symbols:
            assert isinstance(sub_program, Z3Program), "Compiler Bug: Sub program should be a Z3Program"
            if sub_program.name.startswith("f_"):
                fun_decl = FunctionDeclaration(sub_program.name, sub_program)
                fun_decls.add_child(fun_decl)
            elif sub_program.name.startswith("p_"):
                pred_decl = PredicateDeclaration(sub_program.name, sub_program)
                pred_decls.add_child(pred_decl)
            else:
                raise Exception(f"Compiler Bug: Invalid sub program name {sub_program.name}")
        fun_decls_str = fun_decls.smt2()
        pred_decls_str = pred_decls.smt2()
        if self.expr is not None:
            expr_str = f"(define-fun return_expr () Int {self.expr.return_expr()})\n"
        else:
            expr_str = ""
        if self.pred_expr is not None:
            pred_expr_str = f"(define-fun return_pred () Bool {self.pred_expr.return_pred()})\n"
        else:
            pred_expr_str = ""
        smt_program_lines = [
            fun_decls_str,
            pred_decls_str,
            expr_str,
            pred_expr_str
        ]
        smt_program_lines = [line for line in smt_program_lines if line.strip()]
        smt_program = "\n".join(smt_program_lines)
        assert isinstance(smt_program, str), "Compiler Bug: SMT2 program should be a string"
        assert len(smt_program) > 0, "Compiler Bug: SMT2 program should not be empty"
        return smt_program
    
    def run(self, timeout: typing.Optional[int] = 10) -> 'Z3DslRunResult':
        assert self.params == 0, "Runtime Error: To run prover, the number of parameters should be 0"
        smt2 = self.smt2()
        assert isinstance(smt2, str), "Compiler Bug: SMT2 program should be a string"
        assert len(smt2) > 0, "Compiler Bug: SMT2 program should not be empty"
        z3_runtime = Z3DslRuntime()
        proof_found, counter_example = z3_runtime.prove(smtlib_string=smt2, timeout=timeout)
        z3_result = Z3DslRunResult(
            proved=proof_found,
            counter_example=counter_example,
            timed_out=(counter_example is None and proof_found is False),
            smt2=smt2
        )
        return z3_result
    
    @property
    def params(self):
        return self.unbounded_params.params

    @staticmethod
    def from_code(code: str) -> 'Z3Program':
        """
        Parse the code and return the Z3Program.
        """
        parser = Z3ComposableDsl()
        result = parser.parse(code)
        assert isinstance(result, Z3Program), "Compiler Bug: Result should be a Z3Program"
        return result

class Z3ComposableDsl(Grammar):
    Z3Result = namedtuple(
        "Z3Result",
        [
            "proved",
            "counter_example",
            "timed_out",
            "constraints",
            "expr",
        ],
    )

    grammar_path = os.path.join(os.path.dirname(__file__), "z3_dsl.grammar")
    keywords = [
        "Exec",
        "Return",
        ";",
        "'" "And",
        "Or",
        "Implies",
        "ForAll",
        "Exists",
        "NatSet",
        "NatVar",
        "Nat",
        "IsMember",
        "Not",
        "Param",
        "z3_dsl_nat_var_",
        "z3_pythonic_dsl_solver",
        "z3_pythonic_dsl_nat_var_counter",
        "z3_pythonic_dsl_nat_var_prefix",
        "z3_pythonic_dsl_variable_map",
        "z3_pythonic_dsl_nat_set",
        "z3_pythonic_dsl_result",
        "local_",
        "promoted_",
        "Call",
        "call",
        "True",
        "False",
        "true",
        "false",
        "Int",
        "int",
        "Bool",
        "bool",
        "Bounded",
        "None",
        "This",
        "this",
        "(",
        ")",
        ","
    ]            

    def __init__(self):
        with open(Z3ComposableDsl.grammar_path, "r") as f:
            grammar = f.read()
        super(Z3ComposableDsl, self).__init__(
            grammar, Z3ComposableDsl.keywords
        )

    def get_action(self, inp=None):
        actions = {
            "Prog": Z3Program.parse,
            "UnboundedParams": UnboundedParams.parse,
            "BoundedParams": BoundedParams.parse,
            "FunctionDeclarations": FunctionDeclarations.parse,
            "PredicateDeclarations": PredicateDeclarations.parse,
            "Expr": Expr.parse,
            "PredExpr": PredExpr.parse,
            "PredicateDeclaration": PredicateDeclaration.parse,
            "FunctionDeclaration": FunctionDeclaration.parse,
            "ArithExpr": ArithExpr.parse,
            "ArithExprSimple": ArithExprSimple.parse,
            "ArithExprFunc": ArithExprFunc.parse,
            "ArithExprOp": ArithExprOp.parse,
            "ArithExprCompund": ArithExprCompund.parse,
            "LogicalExpr": LogicalExpr.parse,
            "LogicalExprOp": LogicalExprOp.parse,
            "LogicalExprQuantifier": LogicalExprQuantifier.parse,
            "LogicalExprBoolean": LogicalExprBoolean.parse,
            "LogicalExprNot": LogicalExprNot.parse,
            "LogicalExprIsMember": LogicalExprIsMember.parse,
            "LogicalExprTrueFalse": LogicalExprTrueFalse.parse,
            "LogicalExprPred": LogicalExprPred.parse,
            "BoundedVarSeq": BoundedVarSeq.parse,
            "LogicalSeq": LogicalSeq.parse,
            "ArithVar": ArithVar.parse,
            "ArithBind": ArithBind.parse,
            "ArithBindSeq": ArithBindSeq.parse,
            "pred_var": LeafNode.parse,
            "fun_var": LeafNode.parse,
            "inp_var": LeafNode.parse,
            "bounded_var": LeafNode.parse,
            "nat_num": lambda context, nodes: LeafNode.parse(context, int(nodes)),
            "num": lambda context, nodes: LeafNode.parse(context, int(nodes)),
        }
        return actions
    
    def parse(self, code: str) -> Z3ComposableDslParseNode:
        actions = self.get_action()
        parser = self._get_parser(actions)
        result = parser.parse(code)
        return result

    def run(self, code: str) -> Z3DslRunResult:
        result = self.parse(code)
        assert isinstance(result, Z3Program), "Compiler Bug: Result should be a Z3Program"
        return result.run()