namespace Basics

/-- A simple type for propositional formulas using only atoms, falsum, and implication. -/
inductive Formula : Type
| var    : String → Formula
| falsum : Formula
| imp    : Formula → Formula → Formula

open Formula

-- Notations for our basic language.
notation "⊥" => falsum
infixr:20 " → " => imp

/-- Derived definitions for other connectives. -/
def not (A : Formula) : Formula := A → ⊥
def and (A B : Formula) : Formula := not (A → not B)
def or (A B : Formula) : Formula := not A → B
def iff (A B : Formula) : Formula := and (A → B) (B → A)

-- For convenience, you can introduce notation for the derived connectives too.
prefix:max "¬" => not
infixr:35 " ∧ " => and
infixr:30 " ∨ " => or
infixr:25 " ↔ " => iff

/--
  A Hilbert-style proof system for propositional logic.
  We use three axiom schemas (as given below) and modus ponens.

  The axioms are presented as schemas—each axiom holds for any formulas A, B (and C).
-/
inductive Provable : Formula → Prop
| ax1 : ∀ (A B : Formula), Provable (A → (B → A))
| ax2 : ∀ (A B C : Formula), Provable ((A → (B → C)) → ((A → B) → (A → C)))
| ax3 : ∀ (A B : Formula), Provable ((¬ B → ¬ A) → (A → B))
| mp  : ∀ {A B : Formula}, Provable A → Provable (A → B) → Provable B

end Basics
