#!/bin/bash

# Configuration
NUM_PARALLEL_RUNS=32  # N: Number of parallel FunSearch instances per iteration
FUNSEARCH_SCRIPT="scripts/run_funsearch_smaller_budget.sh" # Relative path from project root
NUM_ITERATIONS=1     # M: Number of times to run the block of N parallel instances

# This script assumes it is run from the project root directory (e.g., ~/Projects/FRAME)
# The FUNSEARCH_SCRIPT path is relative to this root.

# Check if the FunSearch script exists at the expected relative path
if [ ! -f "$FUNSEARCH_SCRIPT" ]; then
    echo "Error: FunSearch script not found at $(pwd)/$FUNSEARCH_SCRIPT"
    echo "Please ensure you are running this script from the project root directory (e.g., ~/Projects/FRAME)."
    exit 1
fi

# Ensure the target script is executable
chmod +x "$FUNSEARCH_SCRIPT"

echo "Starting $NUM_ITERATIONS iteration(s) of $NUM_PARALLEL_RUNS parallel FunSearch runs each."
echo "Running from: $(pwd)" # Should be the project root

for iter_num in $(seq 1 $NUM_ITERATIONS)
do
    echo "---------------------------------------------------"
    echo "Starting Iteration $iter_num of $NUM_ITERATIONS..."
    echo "---------------------------------------------------"

    for i in $(seq 1 $NUM_PARALLEL_RUNS)
    do
        echo "Iteration $iter_num, Launching instance $i of $NUM_PARALLEL_RUNS: $FUNSEARCH_SCRIPT"
        # The FUNSEARCH_SCRIPT itself (e.g., scripts/run_funsearch_test.sh) handles cd-ing to the project root.
        # We call it using its path relative to the current directory (which should be project root).
        (bash "$FUNSEARCH_SCRIPT" &)
        
        # Wait for 2 seconds before launching the next instance, if it's not the last one in this block.
        if [ "$i" -lt "$NUM_PARALLEL_RUNS" ]; then
            echo "Iteration $iter_num, Instance $i: Waiting 2 seconds before launching next instance..."
            sleep 2
        fi
    done

    echo "Iteration $iter_num: All $NUM_PARALLEL_RUNS instances launched. Waiting for this block to complete..."
    # Wait for all background jobs started in this iteration's block to complete
    wait
    echo "Iteration $iter_num: All $NUM_PARALLEL_RUNS instances in this block have completed."

done

echo "---------------------------------------------------"
echo "All $NUM_ITERATIONS iteration(s) completed." 