"""
Test interestingness function for testing DSL primitives.
"""

from frame.knowledge_base.knowledge_graph import KnowledgeGraph, NodeType
import math
import numpy as np

# Import necessary DSL primitives
from frame.interestingness.learning.dsl_primitives import (
    get_ancestors, get_descendants, get_construction_depth,
    get_in_degree, get_out_degree, get_construction_history_rule_names,
    get_entity_node_type, get_concept_category, get_input_arity, get_num_component_types,
    get_examples, get_nonexamples,
    get_num_construction_inputs, is_proven
)

def calculate_interestingness(entity_id: str, graph: KnowledgeGraph) -> float:
    """
    A complex interestingness function for testing DSL primitives.
    
    Args:
        entity_id: Entity ID to score
        graph: Knowledge graph containing the entity
        
    Returns:
        Interestingness score between 0 and 1
    """
    try:
        # Extract entity type information
        entity_type = get_entity_node_type(entity_id, graph)
        
        # Structural metrics
        ancestors = get_ancestors(entity_id, graph)
        descendants = get_ancestors(entity_id, graph)
        depth = get_construction_depth(entity_id, graph)
        
        # Examples information
        examples = get_examples(entity_id, graph)
        nonexamples = get_nonexamples(entity_id, graph)
        
        # Calculate a complexity score
        complexity_score = min(0.1 * depth, 0.5)
        
        # Calculate a examples-based score
        examples_score = 0.0
        if examples + non_examples > 0:
            examples_ratio = examples / (examples + non_examples + 0.01)
            examples_score = min(0.2 * examples_ratio, 0.3)
        
        # Final score depends on entity type
        base_score = 0.0
        if entity_type == "concept":
            base_score = 0.2
        elif entity_type == "conjecture":
            base_score = 0.1
            
        # Get proven status for conjectures
        proven_bonus = 0.0
        if entity_type == "conjecture":
            proven_bonus = 0.2 if is_proven(entity_id, graph) else 0.0
        
        # Combine all scores
        final_score = (
            base_score + 
            complexity_score + 
            examples_score + 
            proven_bonus
        )
        
        # Ensure score is in [0, 1] range
        return max(0.0, min(final_score, 1.0))
        
    except Exception as e:
        print(f"Error in test_program interestingness calculator: {e}")
        # Default fallback score
        return 0.0 