"""Tests for the REPL system."""

import unittest
from unittest.mock import Mock, patch
from typing import List, Dict, Any, Optional
import io
import sys

from frame.repl.core.engine import MathREPLCore
from frame.repl.core.state import StateSnapshot, CommandHistory
from frame.repl.interface.cli import MathREPLShell
from frame.repl.interface.protocols import CommandResult
from frame.repl.commands.base import BaseCommand, BaseCommandRegistry
from frame.repl.commands.math import ApplyRuleCommand
from frame.repl.core.events import EventType, Event
from frame.knowledge_base.knowledge_graph import KnowledgeGraph
from frame.knowledge_base.initial_concepts import create_successor_concept, zero_concept
from frame.productions.concepts import (
    MapIterateRule, ComposeRule, NegateRule, ExistsRule,
    MatchRule, SpecializeRule, SizeRule, ConstantRule, ForallRule
)
from frame.productions.conjectures import (
    EquivalenceRule, ImplicationRule, NonexistenceRule, ExclusivityRule
)

class TestREPLCommands(unittest.TestCase):
    """Test cases for REPL command functionality."""
    
    def setUp(self):
        """Set up test environment."""
        # Create production rules
        self.rules = [
            MapIterateRule(),
            ComposeRule(),
            NegateRule(),
            ExistsRule(),
            MatchRule(),
            SpecializeRule(),
            SizeRule(),
            ConstantRule(),
            ForallRule(),
            EquivalenceRule(),
            ImplicationRule(),
            NonexistenceRule(),
            ExclusivityRule(),
        ]
        
        # Initialize REPL with rules
        self.repl = MathREPLCore(production_rules=self.rules)
        
        # Register math commands
        from frame.repl.commands.basic import RenameCommand
        self.repl.registry.register(RenameCommand())
        self.repl.registry.register(ApplyRuleCommand())
        
        # Track emitted events
        self.events = []
        def event_handler(event: Event):
            self.events.append(event)
        
        # Subscribe to all event types
        for event_type in EventType:
            self.repl.subscribe(event_type, event_handler)
            
    def test_initial_state(self):
        """Test initial REPL state."""
        state = self.repl.get_state()
        
        # Check command registry
        self.assertIsInstance(state.get_command_registry(), BaseCommandRegistry)
        commands = state.get_available_commands()
        self.assertIn("rename", commands)
        self.assertIn("apply", commands)
        
        # Check initial graph has Zero and Successor
        concepts = state.get_concepts()
        self.assertEqual(len(concepts), 3)
        concept_names = [c.name for c in concepts if hasattr(c, 'name')]
        self.assertIn("zero", concept_names)
        self.assertIn("successor", concept_names)
        
        # Check empty history
        self.assertEqual(len(state.command_history), 0)
            
    def test_apply_rule_command(self):
        """Test applying production rules via command."""
        # Test creating addition from successor using map_iteration
        result = self.repl.execute_command("apply map_iteration successor")
        self.assertTrue(result.success, f"Command failed: {result.message}")
        self.assertIn("add", result.message) # renamed to add automatically, so check for that.
        
        # Test error cases
        result = self.repl.execute_command("apply nonexistent_rule successor")
        self.assertFalse(result.success)
        self.assertIn("Unknown rule", result.message)
        
        result = self.repl.execute_command("apply map_iteration nonexistent_concept")
        self.assertFalse(result.success)
        self.assertIn("Unknown entity", result.message)
        
    def test_command_execution_and_history(self):
        """Test command execution and history tracking."""
        # Execute a command
        result = self.repl.execute_command("apply map_iteration successor")
        self.assertTrue(result.success)
        self.assertEqual(len(self.repl.history), 1)
        self.assertEqual(self.repl.history[0].command, "apply")
        
        # Execute unknown command
        result = self.repl.execute_command("unknown")
        self.assertFalse(result.success)
        self.assertEqual(len(self.repl.history), 1)  # History unchanged
        
        # Execute command with args
        result = self.repl.execute_command("rename add add*")
        self.assertTrue(result.success)
        self.assertEqual(len(self.repl.history), 2)
        self.assertEqual(self.repl.history[1].command, "rename")
        
    def test_apply_rule_with_params(self):
        """Test applying rules with parameters."""
        # First create multiplication concept
        result = self.repl.execute_command("apply map_iteration successor")
        self.assertTrue(result.success, f"Failed to create addition: {result.message}")
        
        # Now test match rule with parameters
        result = self.repl.execute_command("apply match add indices_to_match=[0,1]")
        self.assertTrue(result.success, f"Failed to apply match rule: {result.message}")
        
        # Test exists rule with parameters
        result = self.repl.execute_command("apply exists add indices_to_quantify=[0]")
        self.assertTrue(result.success, f"Failed to apply exists rule: {result.message}")
        
    def test_apply_rule_events(self):
        """Test event emission for rule application."""
        # Clear events from initialization
        self.events.clear()
        
        # Apply a rule
        result = self.repl.execute_command("apply map_iteration successor")
        self.assertTrue(result.success, f"Command failed: {result.message}")
        
        # Check command executed event
        command_events = [e for e in self.events 
                         if e.type == EventType.COMMAND_EXECUTED]
        self.assertEqual(len(command_events), 2) # auto_rename adds an extra command event.
        self.assertEqual(command_events[0].data['command'], 'auto_rename')
        self.assertEqual(command_events[1].data['command'], 'apply')
        
        # Check new entity event
        new_entity_events = [e for e in self.events 
                            if e.type == EventType.NEW_ENTITY]
        self.assertEqual(len(new_entity_events), 1)
        
        # Verify the event data
        event = new_entity_events[0]
        self.assertIn('entity', event.data)
        self.assertIn('name', event.data)
        self.assertEqual(event.data['name'], 'add')
        
        # Check error event for invalid rule
        self.events.clear()
        result = self.repl.execute_command("apply nonexistent_rule successor")
        error_events = [e for e in self.events if e.type == EventType.ERROR]
        self.assertEqual(len(error_events), 1)
        
    def test_apply_rule_chain(self):
        """Test chaining multiple rule applications."""
        # Create addition from successor
        result = self.repl.execute_command("apply map_iteration successor")
        self.assertTrue(result.success, "Failed to create addition")
        
        result = self.repl.execute_command("rename add add*")
        self.assertTrue(result.success, "Failed to rename addition")

        # Create multiplication from addition
        result = self.repl.execute_command("apply map_iteration add* zero")
        self.assertTrue(result.success, "Failed to create multiplication")
        
        result = self.repl.execute_command("rename iterate_(add*_with_zero) multiply*")
        self.assertTrue(result.success, "Failed to rename multiplication")

        
        # Create square from multiplication using match
        result = self.repl.execute_command("apply match multiply* indices_to_match=[0,1]")
        self.assertTrue(result.success, "Failed to create square")
        
        # Create divides from multiplication using exists
        result = self.repl.execute_command("apply exists multiply* indices_to_quantify=[0]")
        self.assertTrue(result.success, "Failed to create divides")
        
        # Verify the concepts were created
        state = self.repl.get_state()
        concepts = state.get_concepts()
        concept_names = [c.name for c in concepts if hasattr(c, 'name')]
        
        # Verify the construction history through command history
        history = self.repl.history
        self.assertEqual(len(history), 6)  # One entry for each successful command
        
        # Verify each command in history has the correct result
        for entry in history:
            self.assertTrue(entry.result.success)

if __name__ == '__main__':
    unittest.main() 