#!/usr/bin/env python
"""
Script to automatically generate RST documentation files for all environment modules
and update the main environments.rst file accordingly.
"""

import os
import glob

# Paths
DOCS_DIR = "source/frame"
ENVIRONMENTS_DIR = "../frame/environments"
RST_OUTPUT_DIR = f"{DOCS_DIR}/environments"


def ensure_dir(directory):
    """Create directory if it doesn't exist."""
    if not os.path.exists(directory):
        os.makedirs(directory)


def get_module_names(directory, exclude_init=True):
    """Get module names from a directory."""
    modules = glob.glob(f"{directory}/*.py")
    if exclude_init:
        modules = [m for m in modules if not os.path.basename(m).startswith("__")]
    return sorted([os.path.splitext(os.path.basename(m))[0] for m in modules])


def create_environment_rst(module_name, output_dir):
    """Create an RST file for an environment module."""
    # Convert module_name to a title case for display
    display_name = " ".join(word.capitalize() for word in module_name.split("_"))

    rst_content = f"""{display_name} Environment
{'=' * (len(display_name) + 12)}

.. automodule:: frame.environments.{module_name}
   :members:
   :undoc-members:
   :show-inheritance:
"""

    # Write the RST file
    output_file = f"{output_dir}/{module_name}.rst"
    with open(output_file, "w") as f:
        f.write(rst_content)

    print(f"Created: {output_file}")


def update_environments_rst():
    """Update the main environments.rst file."""
    # Get module names
    env_modules = get_module_names(ENVIRONMENTS_DIR)

    # Build the environments.rst content
    content = """Environments
===========

FRAME environments define the context in which concepts and conjectures are generated.

.. toctree::
   :maxdepth: 1
   :caption: Contents:

"""

    # Add module paths
    for module in env_modules:
        content += f"   environments/{module}\n"

    # Write the updated environments.rst file
    with open(f"{DOCS_DIR}/environments.rst", "w") as f:
        f.write(content)

    print(f"Updated: {DOCS_DIR}/environments.rst")


def main():
    """Main entry point."""
    # Ensure directory exists
    ensure_dir(RST_OUTPUT_DIR)

    # Generate RST files for environment modules
    for module in get_module_names(ENVIRONMENTS_DIR):
        create_environment_rst(module, RST_OUTPUT_DIR)

    # Update the main environments.rst file
    update_environments_rst()

    print("\nDone! Environment documentation has been updated.")


if __name__ == "__main__":
    main()
