#!/usr/bin/env python
"""
Script to automatically generate RST documentation files for all policy modules
and update the main policies.rst file accordingly.
"""

import os
import glob

# Paths
DOCS_DIR = "source/frame"
POLICIES_DIR = "../frame/policies"
RST_OUTPUT_DIR = f"{DOCS_DIR}/policies"


def ensure_dir(directory):
    """Create directory if it doesn't exist."""
    if not os.path.exists(directory):
        os.makedirs(directory)


def get_module_names(directory, exclude_init=True):
    """Get module names from a directory."""
    modules = glob.glob(f"{directory}/*.py")
    if exclude_init:
        modules = [m for m in modules if not os.path.basename(m).startswith("__")]
    return sorted([os.path.splitext(os.path.basename(m))[0] for m in modules])


def create_policy_rst(module_name, output_dir):
    """Create an RST file for a policy module."""
    # Convert module_name to a title case for display
    display_name = " ".join(word.capitalize() for word in module_name.split("_"))

    rst_content = f"""{display_name}
{'=' * (len(display_name) + 7)}

.. automodule:: frame.policies.{module_name}
   :members:
   :undoc-members:
   :show-inheritance:
"""

    # Write the RST file
    output_file = f"{output_dir}/{module_name}.rst"
    with open(output_file, "w") as f:
        f.write(rst_content)

    print(f"Created: {output_file}")


def update_policies_rst():
    """Update the main policies.rst file."""
    # Get module names
    policy_modules = get_module_names(POLICIES_DIR)

    # Build the policies.rst content
    content = """Policies
========

FRAME policies determine how conjectures and concepts are selected during theory building.

.. toctree::
   :maxdepth: 1
   :caption: Contents:

"""

    # Add module paths
    for module in policy_modules:
        content += f"   policies/{module}\n"

    # Write the updated policies.rst file
    with open(f"{DOCS_DIR}/policies.rst", "w") as f:
        f.write(content)

    print(f"Updated: {DOCS_DIR}/policies.rst")


def main():
    """Main entry point."""
    # Ensure directory exists
    ensure_dir(RST_OUTPUT_DIR)

    # Generate RST files for policy modules
    for module in get_module_names(POLICIES_DIR):
        create_policy_rst(module, RST_OUTPUT_DIR)

    # Update the main policies.rst file
    update_policies_rst()

    print("\nDone! Policies documentation has been updated.")


if __name__ == "__main__":
    main()
