#!/usr/bin/env python
"""
Script to automatically generate RST documentation files for all production rules
and update the main productions.rst file accordingly.

This script:
1. Scans for all production rule modules in frame/productions/concepts and frame/productions/conjectures
2. Creates individual RST files for each module
3. Updates the main productions.rst file with all the discovered modules
"""

import os
import re
import glob

# Paths
DOCS_DIR = "source/frame"
PRODUCTIONS_DIR = "../frame/productions"
RST_OUTPUT_DIR = f"{DOCS_DIR}/productions"
CONCEPTS_DIR = f"{PRODUCTIONS_DIR}/concepts"
CONJECTURES_DIR = f"{PRODUCTIONS_DIR}/conjectures"


def ensure_dir(directory):
    """Create directory if it doesn't exist."""
    if not os.path.exists(directory):
        os.makedirs(directory)


def get_module_names(directory, exclude_init=True):
    """Get module names from a directory."""
    modules = glob.glob(f"{directory}/*.py")
    if exclude_init:
        modules = [m for m in modules if not os.path.basename(m).startswith("__")]
    return sorted([os.path.splitext(os.path.basename(m))[0] for m in modules])


def create_rule_rst(module_path, module_name, output_dir):
    """Create an RST file for a production rule."""
    # Convert module_name to a class name (e.g., map_iterate -> MapIterateRule)
    class_name = "".join(word.capitalize() for word in module_name.split("_")) + "Rule"

    rst_content = f"""{class_name}
{"=" * len(class_name)}

.. automodule:: {module_path}.{module_name}
   :members:
   :undoc-members:
   :show-inheritance:
"""

    # Write the RST file
    output_file = f"{output_dir}/{module_name}.rst"
    with open(output_file, "w") as f:
        f.write(rst_content)

    print(f"Created: {output_file}")


def update_productions_rst():
    """Update the main productions.rst file with all discovered modules."""
    # Get module names
    concept_modules = get_module_names(CONCEPTS_DIR)
    conjecture_modules = get_module_names(CONJECTURES_DIR)

    # Build the productions.rst content - only with TOCs
    content = """Productions
===========

Productions are the rules that allow for creating new concepts from existing ones.

.. toctree::
   :maxdepth: 1
   :caption: Contents:

   productions/base
   productions/concept_rules
   productions/conjecture_rules
"""

    # Create concept_rules.rst file
    concept_content = """Concept Production Rules
======================

.. toctree::
   :maxdepth: 1
   :caption: Concept Production Rules:

"""
    for module in concept_modules:
        concept_content += f"   concepts/{module}\n"

    os.makedirs(f"{RST_OUTPUT_DIR}", exist_ok=True)
    with open(f"{RST_OUTPUT_DIR}/concept_rules.rst", "w") as f:
        f.write(concept_content)
    print(f"Created: {RST_OUTPUT_DIR}/concept_rules.rst")

    # Create conjecture_rules.rst file
    conjecture_content = """Conjecture Production Rules
========================

.. toctree::
   :maxdepth: 1
   :caption: Conjecture Production Rules:

"""
    for module in conjecture_modules:
        conjecture_content += f"   conjectures/{module}\n"

    with open(f"{RST_OUTPUT_DIR}/conjecture_rules.rst", "w") as f:
        f.write(conjecture_content)
    print(f"Created: {RST_OUTPUT_DIR}/conjecture_rules.rst")

    # Write the updated productions.rst file
    with open(f"{DOCS_DIR}/productions.rst", "w") as f:
        f.write(content)

    print(f"Updated: {DOCS_DIR}/productions.rst")


def create_base_rst():
    """Create the base.rst file for production rules."""
    content = """Production Rules Base
===================

Base classes for all production rules.

.. automodule:: frame.productions.base
   :members:
   :undoc-members:
   :show-inheritance:
"""

    # Write the base.rst file
    output_file = f"{RST_OUTPUT_DIR}/base.rst"
    with open(output_file, "w") as f:
        f.write(content)

    print(f"Created: {output_file}")


def main():
    """Main entry point."""
    # Ensure directories exist
    ensure_dir(f"{RST_OUTPUT_DIR}/concepts")
    ensure_dir(f"{RST_OUTPUT_DIR}/conjectures")

    # Create the base.rst file
    create_base_rst()

    # Generate RST files for concept production rules
    for module in get_module_names(CONCEPTS_DIR):
        create_rule_rst(
            "frame.productions.concepts", module, f"{RST_OUTPUT_DIR}/concepts"
        )

    # Generate RST files for conjecture production rules
    for module in get_module_names(CONJECTURES_DIR):
        create_rule_rst(
            "frame.productions.conjectures", module, f"{RST_OUTPUT_DIR}/conjectures"
        )

    # Update the main productions.rst file
    update_productions_rst()

    print("\nDone! You can now build the documentation with 'make html'")


if __name__ == "__main__":
    main()
