"""Utilities for persisting the FunSearch ProgramsDatabase."""
import os
import pickle
import logging
from typing import Optional, Any

logger = logging.getLogger(__name__)

def save_database(database: Any, filepath: str) -> bool:
    """
    Save the ProgramsDatabase to a pickle file.
    
    Args:
        database: The ProgramsDatabase object to save
        filepath: Path where to save the database
        
    Returns:
        True if successful, False otherwise
    """
    try:
        # Create directory if it doesn't exist
        os.makedirs(os.path.dirname(filepath), exist_ok=True)
        
        # Write to a temporary file first, then rename to avoid corruption on failure
        temp_filepath = f"{filepath}.tmp"
        with open(temp_filepath, 'wb') as f:
            pickle.dump(database, f)
        
        # If we get here, the write was successful, so rename
        if os.path.exists(filepath):
            os.remove(filepath)
        os.rename(temp_filepath, filepath)
        
        logger.info(f"Database saved to {filepath}")
        return True
    
    except Exception as e:
        logger.error(f"Failed to save database to {filepath}: {e}")
        return False

def load_database(filepath: str) -> Optional[Any]:
    """
    Load a ProgramsDatabase from a pickle file.
    
    Args:
        filepath: Path to the pickle file containing the database
        
    Returns:
        The loaded ProgramsDatabase object, or None if loading failed
    """
    if not os.path.exists(filepath):
        logger.info(f"No database found at {filepath}")
        return None
    
    try:
        with open(filepath, 'rb') as f:
            database = pickle.load(f)
        
        logger.info(f"Database loaded from {filepath}")
        return database
    
    except Exception as e:
        logger.error(f"Failed to load database from {filepath}: {e}")
        return None 