"""TheoryBuilder sandbox for evaluating generated interestingness functions."""

import os
import sys
import logging
import tempfile
import importlib.util
from pathlib import Path
import textwrap
from typing import Tuple, Optional, Dict, Any, List, Union
import omegaconf
from omegaconf import OmegaConf, DictConfig
import traceback
import time
import numpy as np
import copy
import yaml
import shutil # Import shutil for directory removal

# Import from FRAME
from frame.theory_builder import TheoryBuilder
from frame.knowledge_base.knowledge_graph import KnowledgeGraph

logger = logging.getLogger(__name__)

def _simple_normalize_indentation(code: str) -> str:
    """
    Normalize indentation in code using Python's built-in textwrap.
    
    This simpler approach:
    1. Identifies the function signature
    2. Extracts the function body
    3. Dedents the body (removes common leading whitespace)
    4. Re-indents with a consistent 4-space indentation
    
    Args:
        code: The Python code string to normalize
        
    Returns:
        Code with normalized indentation
    """
    try:
        # Split the code into lines
        lines = code.splitlines()
        
        # Find the function definition line
        func_def_idx = -1
        for i, line in enumerate(lines):
            if line.strip().startswith("def ") and ":" in line:
                func_def_idx = i
                break
        
        if func_def_idx == -1:
            logger.warning("No function definition found, returning original code")
            return code
            
        # Extract function signature and body separately
        func_signature = lines[func_def_idx]
        func_body = "\n".join(lines[func_def_idx + 1:])
        
        # Dedent the body to remove inconsistent indentation
        dedented_body = textwrap.dedent(func_body)
        
        # Re-indent with a consistent 4-space indentation
        indented_body = textwrap.indent(dedented_body, "    ")
        
        # Combine signature and normalized body
        normalized_code = func_signature + "\n" + indented_body
        
        return normalized_code
    except Exception as e:
        logger.warning(f"Error normalizing indentation: {e}")
        return code

class TheoryBuilderSandbox:
    """Sandbox for evaluating interestingness functions with TheoryBuilder."""
    
    def __init__(
        self,
        base_cfg_fragment: Union[DictConfig, Dict],
        episodes: int = 10,
        evals_base_dir: Optional[Union[str, Path]] = None,
        main_run_output_dir: Optional[Union[str, Path]] = None,
        prompt_yaml_path: Optional[str] = None,
        abstractions_dir_path: Optional[Path] = None,
        save_visualizations: bool = False,
        shared_db_path: Optional[str] = None
    ):
        """
        Initialize the TheoryBuilder sandbox.
        
        Args:
            base_cfg_fragment: A DictConfig or dict containing the relevant base configuration for TheoryBuilder.
            episodes: Number of episodes to run for each evaluation
            evals_base_dir: Base directory to store evaluation results
            main_run_output_dir: The output directory of the main FunSearch run.
            prompt_yaml_path: Path to the interestingness prompt YAML file.
            abstractions_dir_path: Absolute path to the directory containing island abstraction files.
            save_visualizations: Flag to enable/disable visualizations during evaluation runs.
            shared_db_path: Path to the shared episodic cache database file.
        """
        # Store the received config fragment directly
        # Ensure we store a DictConfig
        if isinstance(base_cfg_fragment, dict) and not isinstance(base_cfg_fragment, DictConfig):
            self.base_cfg = OmegaConf.create(base_cfg_fragment)
        else:
            self.base_cfg = base_cfg_fragment
        self.episodes = episodes
        self.main_run_output_dir = Path(main_run_output_dir) if main_run_output_dir else None
        
        # Store the visualization flag
        self.save_visualizations = save_visualizations

        # Store the shared DB path
        self.shared_db_path = shared_db_path
        if self.shared_db_path:
            logger.info(f"Sandbox initialized to use shared DB path: {self.shared_db_path}")
        else:
            logger.warning("Sandbox initialized WITHOUT a shared DB path. TheoryBuilder will create its own.")

        # Output directory for TB logs/results
        if evals_base_dir is None:
            self.evals_dir = Path("./evaluations") # Default relative to CWD
        else:
            self.evals_dir = Path(evals_base_dir)
        os.makedirs(self.evals_dir, exist_ok=True)
        logger.info(f"Evaluation RESULTS will be stored in {self.evals_dir}")

        # Store path to stable abstractions directory (passed from FunSearchManager)
        self.abstractions_dir_path = abstractions_dir_path
        if self.abstractions_dir_path:
             logger.info(f"Abstractions will be imported from: {self.abstractions_dir_path}")
        else:
             logger.info("No abstractions directory provided (abstraction likely disabled).")
        
        # Store the prompt YAML path
        if not prompt_yaml_path:
            # Attempt to construct a default relative path if not provided
            # This might be fragile, relying on the execution directory
            self.prompt_yaml_path = os.path.join(os.getcwd(), 'frame', 'configs', 'prompts', 'interestingness_prompt.yaml')
            logger.warning(f"prompt_yaml_path not provided to sandbox, defaulting to: {self.prompt_yaml_path}")
        else:
            self.prompt_yaml_path = prompt_yaml_path
    
    def load_function_from_code(self, code: str) -> Optional[callable]:
        """
        Load a function from code string. DEPRECATED: Use file path instead.
        
        Args:
            code: Python code containing a calculate_interestingness function
            
        Returns:
            The loaded function or None if loading failed
        """
        logger.warning("load_function_from_code is deprecated. Use file path mechanism.")
        try:
            # Create a temporary module
            module_name = f"interestingness_temp_{id(code)}"
            spec = importlib.util.spec_from_loader(module_name, loader=None)
            module = importlib.util.module_from_spec(spec)
            
            # Execute the code in the module context
            exec(code, module.__dict__)
            
            # Check if the calculate_interestingness function exists
            if not hasattr(module, "calculate_interestingness"):
                logger.error("No calculate_interestingness function found in the code")
                return None
            
            # Return the function
            return module.calculate_interestingness
        
        except Exception as e:
            logger.error(f"Error loading function from code: {e}")
            return None
    
    def create_theory_builder(self, function_path: str, eval_output_dir: Optional[Union[str, Path]] = None) -> TheoryBuilder:
        """
        Create a TheoryBuilder instance configured to use the interestingness function
        at the specified path.
        
        Args:
            function_path: Path to the Python file containing the interestingness function
            eval_output_dir: The output directory for this specific TheoryBuilder evaluation run
            
        Returns:
            A configured TheoryBuilder instance
            
        Raises:
            ValueError: If the function_path does not exist, eval_output_dir is None, or configuration fails.
        """
        if not os.path.exists(function_path):
            raise ValueError(f"Interestingness function path does not exist: {function_path}")
        if eval_output_dir is None:
            raise ValueError("Evaluation output directory must be provided.")
            
        try:
            # Start with a deep copy of the base configuration fragment
            cfg = copy.deepcopy(self.base_cfg)

            # Ensure it's a DictConfig and allow modifications
            if not isinstance(cfg, DictConfig):
                cfg = OmegaConf.create(cfg) # Convert if it was a dict
            OmegaConf.set_struct(cfg, False)

            # Ensure necessary top-level keys exist
            if 'experiment' not in cfg: cfg.experiment = {}
            if 'policy' not in cfg: cfg.policy = {}
            if 'hydra' not in cfg: cfg.hydra = {'run': {}}
            elif 'run' not in cfg.hydra: cfg.hydra.run = {}
            # Ensure initial_state and production_rules are present if needed by TheoryBuilder,
            # even if empty, to avoid errors in _process_config. Let TB handle defaults.
            if 'initial_state' not in cfg: 
                if 'theory_building.initial_state' in cfg:
                    cfg.initial_state = cfg.theory_building.initial_state
                else:
                    cfg.initial_state = {}
            if 'production_rules' not in cfg: cfg.production_rules = {}

            # Ensure nested dicts are modifiable DictConfigs if they exist
            if isinstance(cfg.experiment, dict): cfg.experiment = OmegaConf.create(cfg.experiment)
            if isinstance(cfg.policy, dict): cfg.policy = OmegaConf.create(cfg.policy)
            if isinstance(cfg.hydra, dict): cfg.hydra = OmegaConf.create(cfg.hydra)
            if 'run' in cfg.hydra and isinstance(cfg.hydra.run, dict): cfg.hydra.run = OmegaConf.create(cfg.hydra.run)
            if isinstance(cfg.initial_state, dict): cfg.initial_state = OmegaConf.create(cfg.initial_state)
            if isinstance(cfg.production_rules, dict): cfg.production_rules = OmegaConf.create(cfg.production_rules)
            
            # Set struct to False for modifications
            OmegaConf.set_struct(cfg.experiment, False)
            OmegaConf.set_struct(cfg.policy, False)
            if 'run' in cfg.hydra: OmegaConf.set_struct(cfg.hydra.run, False)
            OmegaConf.set_struct(cfg.initial_state, False)
            OmegaConf.set_struct(cfg.production_rules, False)

            # 1. Update experiment settings for this evaluation run
            cfg.experiment.num_episodes = self.episodes
            # Use a unique seed for each evaluation run based on time
            cfg.experiment.seed = int(time.time() * 1000) % 1000000 
            # Explicitly set flags needed by TheoryBuilder init/processing
            cfg.experiment.evaluate_multiple_interestingness = False
            cfg.experiment.num_interestingness_to_generate = 1
            # Override max_steps if desired, or keep from base_cfg
            # cfg.experiment.max_steps = 100 # Example override


            # 2. Update policy config to use the specific function path
            # Ensure the policy has a _target_ if it's expected by instantiate_policy
            if '_target_' not in cfg.policy:
                 # Attempt to infer from base config or set a default if absolutely necessary
                 # This might indicate an issue with how base_cfg_fragment is created/passed
                 logger.warning("Policy config in base_cfg_fragment is missing '_target_'. Attempting to use a default.")
                 # You might need a more robust way to determine the target here
                 cfg.policy._target_ = "frame.policies.interestingness_guided_policy.InterestingnessGuidedPolicy" 
                 
            # Update the function path. Check if 'params' structure is used.
            if OmegaConf.select(cfg.policy, 'params', default=None) is not None:
                 if isinstance(cfg.policy.params, dict): cfg.policy.params = OmegaConf.create(cfg.policy.params)
                 OmegaConf.set_struct(cfg.policy.params, False)
                 cfg.policy.params.interestingness_function_path = function_path
                 cfg.policy.params.generate_interestingness = False
                 cfg.policy.params.concept_selection = "INTERESTINGNESS" # Force use of the provided function
                 OmegaConf.set_struct(cfg.policy.params, True)
            else:
                 cfg.policy.interestingness_function_path = function_path
                 cfg.policy.generate_interestingness = False
                 cfg.policy.concept_selection = "INTERESTINGNESS"

            # 3. Update Hydra config for output directory
            # Ensure TheoryBuilder uses this specific directory for its internal logging and output
            cfg.hydra.run.dir = str(eval_output_dir)
            # Also try setting experiment.log_dir if TheoryBuilder uses that as a fallback or primary
            # Adjust this based on how TheoryBuilder actually determines episode log paths
            cfg.experiment.log_dir = str(eval_output_dir) # Explicitly set experiment log dir

            # 4. Set FunSearch-specific evaluation settings
            if 'output' not in cfg: cfg.output = {}
            if isinstance(cfg.output, dict): cfg.output = OmegaConf.create(cfg.output)
            OmegaConf.set_struct(cfg.output, False)
            cfg.output.mark_best_episode = False # Disable marking for FunSearch runs
            OmegaConf.set_struct(cfg.output, True)

            # 5. Set visualization flag based on sandbox setting
            if 'output' not in cfg: cfg.output = {}
            if isinstance(cfg.output, dict): cfg.output = OmegaConf.create(cfg.output)
            OmegaConf.set_struct(cfg.output, False)
            cfg.output.save_visualizations = self.save_visualizations # Use the flag passed to sandbox
            OmegaConf.set_struct(cfg.output, True)

            # *** Add the DB path override ***
            if self.shared_db_path:
                # Ensure cfg is mutable at the top level
                OmegaConf.set_struct(cfg, False)
                cfg.episodic_cache_db_path = self.shared_db_path
                logger.info(f"Injecting shared DB path into TheoryBuilder config: {self.shared_db_path}")
                # Re-enable struct mode at the top level
                OmegaConf.set_struct(cfg, True)
            else:
                logger.error("No shared DB path available for sandbox; TheoryBuilder will use its default.")

            # Re-enable struct mode
            OmegaConf.set_struct(cfg.experiment, True)
            OmegaConf.set_struct(cfg.policy, True)
            if 'run' in cfg.hydra: OmegaConf.set_struct(cfg.hydra.run, True)
            OmegaConf.set_struct(cfg.initial_state, True)
            OmegaConf.set_struct(cfg.production_rules, True)
            OmegaConf.set_struct(cfg, True)

            # Log the final config being passed
            logger.info(f"Creating TheoryBuilder with final configuration:")
            logger.info(f"Function path: {function_path}")
            logger.info(f"Episodes: {self.episodes}")
            # Log the intended output directory for THIS evaluation
            logger.info(f"TheoryBuilder internal output directory will be forced to: {eval_output_dir}") 

            # Create the TheoryBuilder with the modified config
            theory_builder = TheoryBuilder(cfg=cfg)
            logger.info("TheoryBuilder instance created successfully")
            
            # --- CRITICAL OVERRIDE --- 
            # Force the TheoryBuilder instance to use the specific output directory for this evaluation run.
            # This overrides the output_dir set during TheoryBuilder.__init__ based on global Hydra config.
            theory_builder.output_dir = str(eval_output_dir)
            logger.info(f"Successfully overrode theory_builder.output_dir to: {theory_builder.output_dir}")
            # --- END OVERRIDE ---
            
            return theory_builder

        except Exception as e:
            logger.error(f"Failed to configure TheoryBuilder: {e}")
            logger.debug(traceback.format_exc())
            raise ValueError(f"Configuration error for TheoryBuilder: {e}")

    def run(self, 
            main_function_code: str, 
            iteration: Optional[int] = None,
            island_id: Optional[int] = None
           ) -> Tuple[float, bool]:
        """
        Run an evaluation of a function code, adding necessary imports and paths.
        
        Args:
            main_function_code: The Python code string containing the function(s) to evaluate.
            iteration: Optional FunSearch iteration number for directory organization.
            island_id: Optional ID of the island, used to determine abstraction import.
            
        Returns:
            Tuple of (reward, success_flag)
        """
        eval_id = f"eval_{abs(hash(main_function_code))}_{os.getpid()}_{int(time.time()*1000)}"
        
        # Determine Output Directory for Logs (Candidate file written here too)
        # Structure: evals_dir / island_X / iteration_Y / eval_id
        
        # Determine island part of the path
        if island_id is not None:
            island_id_str = f"island_{island_id}"
        else:
            # Use a specific name for the initial evaluation before islands are assigned
            island_id_str = "island_initial" 
        
        # Determine iteration part of the path
        if iteration is not None:
            iteration_str = f"iteration_{iteration}"
        else:
            # Should generally have an iteration number, but handle None defensively
            iteration_str = "iteration_unknown" 
            
        # Construct the final evaluation directory path
        eval_dir = self.evals_dir / island_id_str / iteration_str / eval_id

        # Create the directory structure if it doesn't exist
        os.makedirs(eval_dir, exist_ok=True)
        logger.info(f"Evaluation files and output for this run will be in: {eval_dir}")
        eval_dir_abs_path = str(eval_dir.resolve()) # Absolute path for sys.path
        
        # Path for the candidate function file
        candidate_function_full_path = eval_dir / "candidate_function.py" 

        # --- Write Main Candidate File (with imports) --- 
        try:
            # No abstraction file writing here anymore
            
            # Prepare Main Function Code
            # 1. Load Base Imports
            base_imports_str = "" 
            try: 
                yaml_path = self.prompt_yaml_path
                if not yaml_path or not os.path.exists(yaml_path):
                     raise FileNotFoundError(f"Missing prompt config: {yaml_path}")
                with open(yaml_path, 'r') as f:
                    prompt_config = yaml.safe_load(f)
                base_imports_str = prompt_config['required_imports'].strip() + '\n\n' 
            except Exception as e: 
                logger.error(f"Failed to load base imports: {e}")
                base_imports_str = "# ERROR: Failed to load base imports!\n"
            
            # 2. Prepare Abstraction Import (only if abstractions are enabled globally)
            abstraction_import_str = ""
            if self.abstractions_dir_path and island_id is not None:
                # Check if the specific island file exists (it should have been created by Manager)
                abstraction_filename = f"island_{island_id}_abstractions.py"
                abstraction_filepath = self.abstractions_dir_path / abstraction_filename
                if abstraction_filepath.exists(): # Check existence instead of content
                     abstraction_module_name = abstraction_filename.replace('.py', '')
                     abstraction_import_str = f"from {abstraction_module_name} import *\n\n" 
                     logger.info(f"Prepared import statement for {abstraction_module_name}")
                else:
                    logger.warning(f"Abstractions enabled, but file not found: {abstraction_filepath}. Skipping import.")
            
            # 3. Combine: Base Imports + Abstraction Import (if any) + Function Code
            code_to_write = base_imports_str + abstraction_import_str + main_function_code
            
            # 4. Write Main Candidate Function File
            with open(candidate_function_full_path, "w") as f:
                f.write(code_to_write)
            logger.info(f"Saved main candidate function to {candidate_function_full_path}")

        except Exception as e:
            logger.error(f"Failed to prepare/write candidate function file {candidate_function_full_path}: {e}")
            return float('-inf'), False
        # --- End Write Main Candidate File --- 

        # --- Run TheoryBuilder with Modified sys.path --- 
        paths_to_remove_from_sys = []
        try:
            # Add evaluation directory (for candidate_function.py)
            logger.debug(f"Adding {eval_dir_abs_path} to sys.path")
            sys.path.insert(0, eval_dir_abs_path) 
            paths_to_remove_from_sys.append(eval_dir_abs_path)
            
            # Add abstractions directory (if it exists)
            if self.abstractions_dir_path:
                abs_dir_str = str(self.abstractions_dir_path.resolve()) # Ensure absolute path
                if abs_dir_str not in sys.path: # Avoid duplicates if somehow already there
                     logger.debug(f"Adding {abs_dir_str} to sys.path for abstractions")
                     sys.path.insert(0, abs_dir_str) 
                     paths_to_remove_from_sys.append(abs_dir_str)

            # Create and run TheoryBuilder
            theory_builder = self.create_theory_builder(
                 function_path=str(candidate_function_full_path), 
                 eval_output_dir=eval_dir 
            )
            average_reward = theory_builder.run() 
            logger.info(f"TheoryBuilder run completed for {eval_id}. Average reward: {average_reward:.4f}")

            success = np.isfinite(average_reward)
            if not success:
                 logger.warning(f"Evaluation {eval_id} resulted in non-finite reward: {average_reward}")
                 average_reward = float('-inf')

            return average_reward, success
            
        except Exception as e:
            logger.error(f"Error during TheoryBuilder evaluation run for {eval_id}: {e}", exc_info=True)
            return float('-inf'), False
        finally:
            # Remove paths from sys.path in reverse order of insertion
            for path_to_remove in reversed(paths_to_remove_from_sys):
                 if path_to_remove in sys.path:
                    try:
                        sys.path.remove(path_to_remove)
                        logger.debug(f"Removed {path_to_remove} from sys.path")
                    except ValueError:
                         # Should not happen if we track additions, but handle defensively
                        logger.warning(f"Tried to remove {path_to_remove} from sys.path, but it wasn't found.")
                 else:
                      logger.debug(f"Path {path_to_remove} was already removed from sys.path.")
        # --- End Run TheoryBuilder --- 
    
    # ... (rest of file) ...
