"""Base classes for interestingness function generation.

This module defines the base classes and interfaces for generating
interestingness functions using various algorithms.
"""

import os
import importlib.util
import inspect
import logging
import yaml
from abc import ABC, abstractmethod
from typing import Dict, List, Any, Optional, Callable, Union, Tuple
from datetime import datetime

from frame.knowledge_base.knowledge_graph import KnowledgeGraph


class BaseInterestingnessGenerator(ABC):
    """Base class for all interestingness function generators.
    
    This class defines the common interface and utilities for generating
    interestingness functions with different algorithms.
    """
    
    def __init__(self, 
                 config_path: Optional[str] = None,
                 output_dir: str = "frame/interestingness/learning/generated_programs",
                 logger: Optional[logging.Logger] = None):
        """
        Initialize the interestingness function generator.
        
        Args:
            config_path: Path to the configuration file for the generator
            output_dir: Directory to save generated functions
            logger: Optional logger instance
        """
        # Configure logging
        self.logger = logger or logging.getLogger(__name__)
        
        # Store parameters
        self.config_path = config_path
        self.output_dir = output_dir
        
        # Ensure output directory exists
        os.makedirs(self.output_dir, exist_ok=True)
        
        # Store generated artifacts
        self.generated_artifacts = {}
    
    @abstractmethod
    def generate(self, **kwargs) -> Tuple[Callable, Optional[str]]:
        """
        Generate an interestingness function.
        
        Args:
            **kwargs: Additional arguments specific to the generator implementation
            
        Returns:
            Tuple of (generated_function, artifact_path_if_saved)
        """
        pass
    
    @abstractmethod
    def save(self, artifact: Any, output_path: Optional[str] = None) -> str:
        """
        Save the generated artifact to a file.
        
        Args:
            artifact: The generated artifact (e.g., function code, model weights)
            output_path: Optional explicit path to save to
            
        Returns:
            Path to the saved artifact
        """
        pass
    
    @classmethod
    @abstractmethod
    def load(cls, artifact_path: str) -> Callable:
        """
        Load a previously generated interestingness function.
        
        Args:
            artifact_path: Path to the saved artifact
            
        Returns:
            Loaded interestingness function
        """
        pass
    
    def _resolve_path(self, path: str) -> str:
        """
        Resolve a path relative to the project root.
        
        Args:
            path: The path to resolve
            
        Returns:
            Absolute path
        """
        if os.path.isabs(path):
            return path
            
        # Get the project root (FRAME directory)
        project_root = os.path.dirname(os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))
        
        # If path starts with 'frame/', remove it to avoid double 'frame/'
        if path.startswith('frame/'):
            # Remove the 'frame/' prefix
            relative_path = path[6:]  # Skip 'frame/'
            return os.path.join(project_root, relative_path)
        else:
            return os.path.join(project_root, path)
    
    def _load_yaml_config(self, config_path: str) -> Dict[str, Any]:
        """
        Load a YAML configuration file.
        
        Args:
            config_path: Path to the YAML config file
            
        Returns:
            Loaded configuration as dictionary
            
        Raises:
            FileNotFoundError: If the config file doesn't exist
        """
        resolved_path = self._resolve_path(config_path)
        
        # Check if the file exists
        if not os.path.exists(resolved_path):
            self.logger.error(f"Config file not found at {resolved_path}")
            raise FileNotFoundError(f"Config file not found at {resolved_path}")
            
        # Load the YAML file
        with open(resolved_path, 'r') as f:
            config = yaml.safe_load(f)
            
        return config
    
    def _get_dsl_primitives(self) -> List[Callable]:
        """
        Get all available DSL primitives for interestingness functions.
        
        Returns:
            List of primitive functions
        """
        from frame.interestingness.learning.dsl_primitives import ALL_PRIMITIVES
        return ALL_PRIMITIVES
    
    def _format_dsl_primitives_for_docstring(self, primitives: List[Callable]) -> str:
        """
        Format DSL primitives as an import statement for docstrings.
        
        Args:
            primitives: List of primitive functions
            
        Returns:
            Formatted string for use in import statements
        """
        # Get all primitive names
        primitive_names = [func.__name__ for func in primitives]
        
        # Format as comma-separated string with wrapping
        formatted = ""
        line = "    "
        for i, name in enumerate(primitive_names):
            if i > 0 and i < len(primitive_names) - 1:
                if len(line + name + ", ") > 80:
                    formatted += line + "\n"
                    line = "    "
                
            if i < len(primitive_names) - 1:
                line += name + ", "
            else:
                line += name
                
        formatted += line
        return formatted
    
    @staticmethod
    def load_function_from_file(filepath: str) -> Callable:
        """
        Load an interestingness function from a Python file.
        
        Args:
            filepath: Path to the file containing the function
            
        Returns:
            Loaded interestingness function
            
        Raises:
            ValueError: If no valid interestingness function is found
        """
        try:
            # Get the module name from the filepath
            module_name = os.path.basename(filepath).split('.')[0]
            
            # Import the module
            spec = importlib.util.spec_from_file_location(module_name, filepath)
            module = importlib.util.module_from_spec(spec)
            spec.loader.exec_module(module)
            
            # Find the calculate_interestingness function
            if hasattr(module, "calculate_interestingness"):
                return module.calculate_interestingness
            
            # If not found, look for any function that accepts entity_id and graph
            for name, obj in inspect.getmembers(module, inspect.isfunction):
                params = inspect.signature(obj).parameters
                if len(params) >= 2 and "entity_id" in params and "graph" in params:
                    return obj
            
            raise ValueError(f"No valid interestingness function found in {filepath}")
            
        except Exception as e:
            logging.error(f"Error loading interestingness function from {filepath}: {e}")
            raise 