"""
This module defines the base ProductionRule class for generating new mathematical concepts.

Production rules are transformations that take existing mathematical entities
(concepts, conjectures) and produce new ones. They can operate on:
- Symbolic definitions
- Computational implementations
- Example sets
- Target language translations (Lean4, Prolog, Z3)
"""

from abc import ABC, abstractmethod
from typing import List, Optional, Set, Tuple, Any, Literal, Dict, Type, Union
from frame.knowledge_base.entities import Entity, Concept, Conjecture, Example, ExampleType, ConceptType

class ProductionRule(ABC):
    """
    Abstract base class for production rules that generate new mathematical entities.
    
    A production rule takes one or more input entities and produces a new entity
    through some transformation of their components (symbolic definitions,
    computational implementations, examples, etc.)
    """
    
    def __init__(self, name: str, description: str, type: Literal["Concept", "Conjecture"]):
        self.name = name
        self.description = description
        self.type = type
        self.max_examples = 50  # Maximum number of examples to generate per transformation
    
    @abstractmethod
    def can_apply(self, *inputs: Entity, verbose: bool = True) -> bool:
        """
        Check if this production rule can be applied to the given input entities.
        
        Args:
            *inputs: Variable number of Entity objects to check
            verbose: Whether to print debug information
            
        Returns:
            bool: True if the rule can be applied to these inputs
        """
        raise NotImplementedError
    
    @abstractmethod
    def apply(self, *inputs: Entity, verbose: bool = True) -> Entity:
        """
        Apply this production rule to the input entities to produce a new entity.
        
        Args:
            *inputs: Variable number of Entity objects to transform
            verbose: Whether to print debug information
            
        Returns:
            Entity: The newly generated mathematical entity
        """
        raise NotImplementedError
    
    @abstractmethod
    def get_valid_parameterizations(self, *inputs: Entity) -> List[Dict[str, Any]]:
        """
        Get a list of valid parameter dictionaries for applying this rule to the given inputs.
        
        This method allows rules to specify what parameterizations are valid for given inputs,
        which is useful for policies that need to generate actions without enumerating all
        possible combinations.
        
        Args:
            *inputs: Variable number of Entity objects to check
            
        Returns:
            List[Dict[str, Any]]: List of valid parameter dictionaries, or empty list if
                                 the rule cannot be applied to these inputs.
                                 If the rule doesn't require parameters, returns [{}].
        """
        raise NotImplementedError
    
    @abstractmethod
    def get_input_types(self) -> List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]:
        """
        Get the types of inputs this rule expects.
        
        Returns:
            List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]: 
                A list of tuples where each tuple contains:
                - The expected type of the input (e.g., Concept, Conjecture)
                - The required ConceptType(s) if the input is a Concept, or None if any ConceptType is acceptable
                  or if the input is not a Concept
                
            OR (for rules with alternative input specifications):
            
            List[List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]]:
                A list of alternative input specifications, where each alternative is a list of tuples.
                This format is used when a rule can accept different combinations of inputs.
                
        Example:
            [(Concept, ConceptType.PREDICATE)]  # A predicate concept
            [(Concept, [ConceptType.FUNCTION, ConceptType.PREDICATE])]  # A function or predicate concept
            [(Concept, None)]  # Any concept
            [(Conjecture, None)]  # Any conjecture
            
            # Alternative input specifications:
            [
                [(Concept, ConceptType.FUNCTION)],  # Option 1: Single function concept
                [(Concept, ConceptType.PREDICATE), (Concept, None)]  # Option 2: Predicate and any concept
            ]
        """
        raise NotImplementedError
    
    def determine_verification_capabilities(self, *inputs: Entity) -> Tuple[bool, bool]:
        """
        Determine whether the resulting concept can reliably verify examples and nonexamples.
        
        This method allows production rules to specify whether computational implementations
        created by the rule can reliably verify positive examples and negative examples.
        For example, existential quantification can reliably verify positive examples
        (by finding a witness) but not negative examples (would require checking all values).
        
        Production rules should override this method to account for:
        1. The inherent verification limitations of the rule itself
        2. The verification capabilities of the input concepts
        
        For instance:
        - If a rule negates a concept, the verification capabilities should be flipped
        - If a rule combines concepts (AND, OR), the capabilities combine in specific ways
        - If a rule transforms a concept in a way that makes verification harder, capabilities should be restricted
        
        Args:
            *inputs: Variable number of Entity objects that will be used to create the new concept
                
        Returns:
            Tuple[bool, bool]: (can_add_examples, can_add_nonexamples)
        """
        # Default implementation assumes full verification capability
        return True, True
    
    def check_input_types(self, *inputs: Entity) -> bool:
        """
        Check if the given inputs match the expected input types.
        
        Args:
            *inputs: Variable number of Entity objects to check
            
        Returns:
            bool: True if the inputs match any of the expected type alternatives, False otherwise
        """
        expected_types = self.get_input_types()
        
        # Handle both the original format (list of tuples) and the new format (list of lists of tuples)
        if expected_types and isinstance(expected_types[0], list):
            # New format: list of alternative input specifications
            for alternative in expected_types:
                if len(inputs) != len(alternative):
                    continue  # Try next alternative
                
                # Check if all inputs match this alternative
                all_match = True
                for i, (input_entity, (expected_type, expected_concept_type)) in enumerate(zip(inputs, alternative)):
                    # Check if the input is of the expected type
                    if not isinstance(input_entity, expected_type):
                        all_match = False
                        break
                        
                    # If it's a concept, check if it has the expected concept type
                    if expected_type == Concept and expected_concept_type is not None:
                        concept = input_entity
                        actual_concept_type = concept.examples.example_structure.concept_type
                        
                        # Handle both single ConceptType and list of ConceptTypes
                        if isinstance(expected_concept_type, list):
                            if actual_concept_type not in expected_concept_type:
                                all_match = False
                                break
                        elif actual_concept_type != expected_concept_type:
                            all_match = False
                            break
                
                if all_match:
                    return True  # Found a matching alternative
            
            return False  # No matching alternative found
        else:
            # Original format: single list of tuples
            if len(inputs) != len(expected_types):
                return False
                
            for i, (input_entity, (expected_type, expected_concept_type)) in enumerate(zip(inputs, expected_types)):
                # Check if the input is of the expected type
                if not isinstance(input_entity, expected_type):
                    return False
                    
                # If it's a concept, check if it has the expected concept type
                if expected_type == Concept and expected_concept_type is not None:
                    concept = input_entity
                    actual_concept_type = concept.examples.example_structure.concept_type
                    
                    # Handle both single ConceptType and list of ConceptTypes
                    if isinstance(expected_concept_type, list):
                        if actual_concept_type not in expected_concept_type:
                            return False
                    elif actual_concept_type != expected_concept_type:
                        return False
            
            return True

    def transform_examples(self, new_concept: Entity, *inputs: Entity):
        """
        Transform examples from input concepts to examples for the new concept.
        Each production rule should implement its own _transform_examples method.
        """
        raise NotImplementedError
    
    def transform_nonexamples(self, *inputs: Entity) -> Set[Example]:
        """
        Transform the non-examples from input entities into non-examples for the output entity.
        This is an optional operation - some rules may not operate on examples.
        
        Args:
            *inputs: Variable number of Entity objects whose non-examples to transform
            
        Returns:
            Set[Example]: The transformed set of non-examples, may be empty
        """
        return set()  # Default implementation returns empty set
    
    def verify_example(self, example: Example, *inputs: Entity) -> bool:
        """
        Verify if a transformed example is valid for the output entity.
        This is an optional operation - some rules may not operate on examples.
        
        Args:
            example: The example to verify
            *inputs: The input entities used to generate this example
            
        Returns:
            bool: True if the example is valid, False otherwise
        """
        return True  # Default implementation accepts all examples
    
    def __str__(self) -> str:
        return f"{self.name}: {self.description}" 
