"""
This module implements the Constant production rule for creating constant concepts.

A constant concept represents a fixed mathematical object. This rule can create constants by
taking an example from a concept and creating a constant concept from it
"""

from typing import Optional, Any, Tuple, List, Dict, Type, Union
from frame.productions.base import ProductionRule
from frame.knowledge_base.entities import (
    Entity,
    Concept,
    Example,
    ExampleType,
    ExampleStructure,
    ConceptType,
    ConceptApplication,
)
from frame.tools.z3_template import Z3Template


class ConstantRule(ProductionRule):
    """
    Production rule that creates constant concepts by converting an example into a constant concept
    """

    def __init__(self, verbose: bool = False):
        super().__init__(
            name="constant",
            description="Creates a new constant concept by converting a concept example",
            type="Concept",
        )
        self.verbose = verbose

    def get_input_types(
        self,
    ) -> List[
        List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]
    ]:
        """Return the expected input types for this rule.

        The constant rule expects any concept with an example parameter
        """
        return [(Concept, None)]  # Example parameter handled separately

    def get_valid_parameterizations(self, *inputs: Entity) -> List[Dict[str, Any]]:
        """Return valid parameterizations for creating constants from the given inputs.

        For example-based creation (1 input), generate parameterizations for each valid example.
        """
        if len(inputs) == 1 and isinstance(inputs[0], Concept):
            # Example-based case - generate parameterizations for each example
            concept = inputs[0]
            parameterizations = []

            # Add parameterizations for examples
            for example in concept.examples.get_examples():
                parameterizations.append({"example": example})

            return parameterizations

        return []

    def can_apply(
        self, *inputs: Entity, example: Optional[Example] = None, verbose: bool = True
    ) -> bool:
        """
        Check if this rule can be applied to the inputs.

        One input: example from any concept
           - Example must be valid for its concept
           - Example must have size of 1
           (input arity 1 for predicate concept, output length 1 for function)
        """
        if len(inputs) != 1 or not isinstance(inputs[0], Concept):
            if verbose:
                print("❌ Failed: Must have exactly one input of type Concept")
            return False

        concept = inputs[0]
        if verbose:
            print(f"✓ Input is a concept: {concept.name}")

        if example is not None:
            example_structure = concept.examples.example_structure
            if not example_structure.validate_example(example.value):
                if verbose:
                    print(
                        f"❌ Failed: Example {example.value} is not valid for concept {concept.name}"
                    )
                return False
            if verbose:
                print("✓ Example is valid for the concept")

            if example_structure.concept_type == ConceptType.PREDICATE:
                if example_structure.input_arity != 1:
                    if verbose:
                        print(
                            f"❌ Failed: Predicate has input arity > 1, not supported by Constant"
                        )
                    return False
            elif example_structure.concept_type == ConceptType.FUNCTION:
                example_len = len(example_structure.component_types)
                output_len = example_len - example_structure.input_arity
                if output_len != 1:
                    if verbose:
                        print(
                            f"❌ Failed: Function has output length > 1, not supported by Constant"
                        )
                    return False
            elif example_structure.concept_type == ConceptType.CONSTANT:
                if verbose:
                    print(
                        f"❌ Failed: Constant does not support applying constant rule to Constant"
                    )
                return False

        else:
            if verbose:
                print("❌ Failed: Must provide example value")
            return False

        return True

    def determine_verification_capabilities(
        self, *inputs: Entity, example: Optional[Example] = None
    ) -> Tuple[bool, bool]:
        """
        Determine verification capabilities for constant concepts.

        Constants generally have high verification reliability:
        - For constants created from examples, they are inherently verifiable.
        - For constants created by applying a function to a constant, we need both
          the function and the constant to be reliable.

        Returns:
            Tuple[bool, bool]: (can_add_examples, can_add_nonexamples)
        """
        # Constants don't have nonexamples, so we'll always return True for the second value
        # They either are or aren't the specific constant value

        # Creating constant from example
        if example is not None:
            return True, True

        # # Applying function to constant
        # if len(inputs) == 2:
        #     constant, func = inputs
        #     constant_can_add_examples = constant.can_add_examples
        #     func_can_add_examples = func.can_add_examples

        #     # Need both the constant and function to be reliable
        #     return constant_can_add_examples and func_can_add_examples, True

        return True, True

    def _z3_translate_constant(self, constant_value: Any) -> Z3Template:
        """Translate a constant value to a Z3 program template."""
        try:
            # Basic translation: assumes the value can be directly represented in Z3/DSL.
            # More complex objects might need specific handling.
            # For now, we assume numeric or boolean constants primarily.
            if isinstance(constant_value, bool):
                value_str = str(constant_value).lower() # Z3 uses 'true'/'false'
            else:
                # Attempt to convert other types directly to string. Might need refinement.
                value_str = str(constant_value)

            code = f"""
            params 0;
            bounded params 0;
            ReturnExpr {value_str};
            ReturnPred None;
            """
            # No arguments needed for a constant template itself, but the value is embedded.
            template = Z3Template(code=code)
            return template
        except Exception as e:
            print(f"Error translating constant value {constant_value} to Z3: {e}")
            # Depending on desired behavior, could return None or raise
            raise # Re-raise for now

    def apply(self, *inputs: Entity, example: Optional[Example] = None) -> Entity:
        """
        Apply the constant rule to create a new concept.

        Convert example to constant: creates constant matching example's type
        """
        if not self.can_apply(*inputs, example=example, verbose=self.verbose):
            raise ValueError("Cannot apply Constant rule to these inputs")

        concept = inputs[0]
        return self._create_from_example(concept, example)

    def _create_from_example(self, concept: Concept, example: Example) -> Concept:
        """Create a constant concept from an example"""
        if self.verbose:
            print(
                f"\nCreating constant from example {example.value} of concept {concept.name}"
            )

        value = example.value

        # If the example is a single value (not a tuple), wrap it
        if not isinstance(value, tuple):
            value = (value,)

        try:
            # Get the type of the example
            if concept.examples.example_structure.concept_type == ConceptType.FUNCTION:
                # For functions, use the output type
                output_type = concept.get_component_types()[-1]
                component_types = (output_type,)
                if self.verbose:
                    print(f"Using function output type: {output_type}")
            else:
                # For other concepts, use all types
                component_types = concept.get_component_types()[-1:]
                if self.verbose:
                    print(f"Using concept types: {component_types}")

            def constant_compute():
                # Return the actual value used for the constant
                return value[-1] 

            # Constants created from examples are inherently verifiable
            can_add_examples, can_add_nonexamples = True, True
            
            # Extract the actual constant value
            constant_value = value[-1]

            new_concept = Concept(
                name=f"constant_({constant_value}_from_{concept.name})",
                description=f"Constant value {constant_value} derived from {concept.name}",
                # Symbolic definition should ideally use a proper Constant entity if available
                # For now, using the value directly might suffice for simple cases
                symbolic_definition=lambda val=constant_value: val, 
                computational_implementation=constant_compute,
                example_structure=ExampleStructure(
                    concept_type=ConceptType.CONSTANT,
                    component_types=component_types,
                    input_arity=0,
                ),
                can_add_examples=can_add_examples,
                can_add_nonexamples=can_add_nonexamples,
                 # Add Z3 translation
                z3_translation=(lambda val=constant_value: self._z3_translate_constant(val))
                if concept.has_z3_translation()
                else None,
            )

            # Add the value as an example (ensure it's a tuple)
            new_concept.add_example((constant_value,))
            new_concept.map_iterate_depth = concept.map_iterate_depth

            if self.verbose:
                print(f"✓ Successfully created constant concept: {new_concept.name}")

            return new_concept

        except Exception as e:
            if self.verbose:
                print(f"❌ Failed to create constant concept: {str(e)}")
            raise
