"""
This module implements the Negate production rule for creating logical negations.
"""

from typing import List, Optional, Tuple, Any, Dict, Type, Union
from frame.productions.base import ProductionRule
from frame.knowledge_base.entities import (
    Not,
    ConceptApplication,
    Entity,
    Concept,
    ConceptType,
)
from frame.tools.z3_template import Z3Template, _format_args

class NegateRule(ProductionRule):
    """
    Production rule that takes a predicate concept and produces its logical negation.

    For example:
    - Given is_even(n) = (n mod 2 = 0)
    - Produces is_odd(n) = ¬(n mod 2 = 0)
    """

    def __init__(self):
        super().__init__(
            name="negation",
            description="Creates a new concept by negating an existing concept",
            type="Concept",
        )

    def determine_verification_capabilities(self, *inputs: Entity) -> Tuple[bool, bool]:
        """
        Negation flips the verification capabilities of the input concept.

        If a concept C can reliably verify examples but not non-examples,
        then NOT(C) can reliably verify non-examples but not examples.

        Returns:
            Tuple[bool, bool]: (can_add_examples, can_add_nonexamples)
        """
        if not inputs:
            return True, True

        # Get capabilities of the input concept
        input_concept = inputs[0]
        input_can_add_examples = input_concept.can_add_examples
        input_can_add_nonexamples = input_concept.can_add_nonexamples

        # Negation flips the capabilities
        return input_can_add_nonexamples, input_can_add_examples

    def get_input_types(
        self,
    ) -> List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]:
        """
        Get the types of inputs this rule expects.

        Returns:
            List[Tuple[Type[Entity], Optional[Union[ConceptType, List[ConceptType]]]]]:
                A list containing a single tuple for the predicate concept
        """
        return [(Concept, ConceptType.PREDICATE)]

    def get_valid_parameterizations(self, *inputs: Entity) -> List[Dict[str, Any]]:
        """
        Get valid parameterizations for negating the given concept.

        Args:
            *inputs: A single predicate concept to negate

        Returns:
            List of valid parameter dictionaries (empty for this rule as it requires no parameters)
        """
        # Check if the inputs match the expected input types
        if not self.check_input_types(*inputs):
            return []

        # The negate rule doesn't require any additional parameters
        return [{}]  # Return a list with a single empty dictionary

    def can_apply(self, *inputs: Entity, verbose: bool = True) -> bool:
        """
        Check if this rule can be applied to the inputs.

        Requirements:
        1. Single input concept
        2. Input concept must be a predicate (returns boolean)

        Args:
            *inputs: Input entities to check
            verbose: Whether to print debug information
        """
        if verbose:
            print("Checking requirements:")

        if len(inputs) != 1 or not isinstance(inputs[0], Concept):
            if verbose:
                print("❌ Failed: Must have exactly one input of type Concept")
            return False

        concept = inputs[0]
        if verbose:
            print(f"✓ Input is a concept: {concept.name}")

        # Check that the concept is a predicate
        if concept.examples.example_structure.concept_type != ConceptType.PREDICATE:
            if verbose:
                print("❌ Failed: Input concept must be a predicate")
            return False

        if verbose:
            print("✓ Input concept is a predicate")
        return True

    def apply(self, *inputs: Entity) -> Entity:
        """
        Apply the negation rule to create a new concept.

        Given a concept C, creates ¬C with:
        - Symbolic definition: Not(C(...))
        - Computational implementation: not C(...) (if C has a computational implementation)
        - Examples and nonexamples swapped
        """
        concept = inputs[0]

        def negate_compute(*args):
            """Helper function to negate the computation"""
            try:
                result = concept.compute(*args)
                if result is not None:
                    return not result
                return None
            except Exception:
                raise ValueError(f"Error negating concept {concept.name}")

        # Get verification capabilities (flipped from input)
        can_add_examples, can_add_nonexamples = (
            self.determine_verification_capabilities(*inputs)
        )

        # Create the new concept that represents the negation
        new_concept = Concept(
            name=f"not_({concept.name})",
            description=f"Negation of {concept.name}",
            symbolic_definition=lambda *args: Not(ConceptApplication(concept, *args)),
            computational_implementation=(
                negate_compute if concept.has_computational_implementation() else None
            ),
            example_structure=concept.examples.example_structure,  # Negation preserves structure
            z3_translation=(lambda *args: self._z3_translate_negation(concept, *args))
            if concept.has_z3_translation()
            else None,
            can_add_examples=can_add_examples,
            can_add_nonexamples=can_add_nonexamples,
        )
        new_concept.map_iterate_depth = concept.map_iterate_depth
        # Transform examples from the base concept
        self._transform_examples(concept, new_concept)

        return new_concept

    def _z3_translate_negation(self, concept: Concept, *args) -> Z3Template:
        """Translate the negation of a concept to a Z3 program."""
        concept_template = concept.to_z3(*([None] * concept.get_input_arity()))
        program = concept_template.program
        template = Z3Template(
            code = f"""
            params {program.params};
            bounded params 0;
            p_0 := Pred(
            {program.dsl()}
            );
            ReturnExpr None;
            ReturnPred Not(p_0({_format_args([f"x_{i}" for i in range(len(args))])}));
        """)
        template.set_args(*args)
        return template
    
    def _transform_examples(self, concept: Concept, new_concept: Concept) -> None:
        """Transform examples and nonexamples from the original concept to the negated concept.

        Args:
            concept: The original concept.
            new_concept: The negated concept.
        """
        # Add the nonexamples of the original concept as examples of the negated concept
        for nonexample in concept.examples.get_nonexamples():
            new_concept.add_example(nonexample.value)

        # Add the examples of the original concept as nonexamples of the negated concept
        for example in concept.examples.get_examples():
            new_concept.add_nonexample(example.value)
