"""
This module implements the Equivalence production rule for generating conjectures about concept equivalence.
"""

from typing import List, Optional, Set, Tuple, Any, Dict, Type
from frame.productions.base import ProductionRule
from frame.knowledge_base.entities import (
    Expression, Var, ConceptApplication, Forall, NatDomain,
    Entity, Concept, Conjecture, Example, ExampleType, ExampleStructure, ConceptType,
    Equals, Implies, And, Exists, Nat, Fold, Lambda
)
from frame.tools.z3_template import Z3Template, _format_args
from frame.knowledge_base.demonstrations import addition, is_even, divides, multiplication
from frame.knowledge_base.entities import Succ, Zero

class EquivalenceRule(ProductionRule):
    """
    Production rule that takes two concepts with matching arities and produces a conjecture stating their equivalence.
    
    For functions f, g: A₁ × ... × Aₙ → B, produces:
    ∀x₁...xₙ. f(x₁,...,xₙ) = g(x₁,...,xₙ)
    
    For predicates P, Q: A₁ × ... × Aₙ → Bool, produces:
    ∀x₁...xₙ. P(x₁,...,xₙ) ↔ Q(x₁,...,xₙ)
    
    Examples:
    - Given is_even_perfect(n) and is_mersenne_form(n)
    - Produces conjecture that n is an even perfect number iff 
      it is of the form 2^k * (2^(k+1) - 1) where 2^(k+1) - 1 is prime
    
    - Given add1(n) and successor(n)
    - Produces conjecture that add1(n) = successor(n) for all n
    """
    
    def __init__(self):
        super().__init__(
            name="equivalence",
            description="Creates a conjecture stating that two concepts are equivalent",
            type="Conjecture"
        )
    
    def determine_verification_capabilities(self, *inputs: Entity) -> Tuple[bool, bool]:
        """
        Determine verification capabilities for equivalence conjectures.
        
        For equivalence conjectures, verification capabilities depend on the constituent concepts:
        - For examples: We can verify an example of equivalence if both concepts can verify examples.
        - For nonexamples: We can verify a nonexample of equivalence if either concept can verify nonexamples.
        
        Returns:
            Tuple[bool, bool]: (can_add_examples, can_add_nonexamples)
        """
        if len(inputs) != 2:
            return False, False
            
        concept1, concept2 = inputs
        
        # For equivalence (A ⟷ B), we can verify an example if both A and B can be verified
        concept1_can_add_examples = concept1.can_add_examples
        concept2_can_add_examples = concept2.can_add_examples
        
        # For nonexamples (cases where equivalence fails), we need at least one concept 
        # to be able to verify nonexamples
        concept1_can_add_nonexamples = concept1.can_add_nonexamples
        concept2_can_add_nonexamples = concept2.can_add_nonexamples
        
        return (concept1_can_add_examples and concept2_can_add_examples,
                concept1_can_add_nonexamples or concept2_can_add_nonexamples)
    
    def get_input_types(self) -> List[List[Tuple[Type, Any]]]:
        """
        Returns the valid input types for this production rule.
        
        Returns:
            A list of lists, where each inner list represents a valid combination of input types.
            Each element in the inner list is a tuple of (type, subtype).
        """
        return [
            [(Concept, ConceptType.PREDICATE), (Concept, ConceptType.PREDICATE)],
            [(Concept, ConceptType.FUNCTION), (Concept, ConceptType.FUNCTION)]
        ]
    
    def get_valid_parameterizations(self, *inputs: Entity) -> List[Dict[str, Any]]:
        """
        Get valid parameterizations for creating an equivalence conjecture.
        
        Args:
            *inputs: Two concepts (predicates or functions) to create an equivalence between
            
        Returns:
            List of valid parameter dictionaries (empty dict for this rule as it requires no additional parameters)
        """
        # Check if the rule can be applied to these inputs
        if not self.can_apply(*inputs, verbose=False):
            return []
            
        # Equivalence rule doesn't require any additional parameters
        return [{}]

    def can_apply(self, *inputs: Entity, verbose: bool = True) -> bool:
        """
        Check if this rule can be applied to the inputs.
        
        Requirements:
        1. Two input concepts
        2. Both must be either functions or predicates
        3. Must have same arity and matching types
        4. Examples and nonexamples must be consistent between concepts
        5. The two concepts must be different (not the same concept)
        
        Args:
            *inputs: Input entities to check
            verbose: Whether to print debug information
        """
        if verbose:
            print("Checking requirements:")
        
        if len(inputs) != 2 or not all(isinstance(x, Concept) for x in inputs):
            if verbose:
                print("❌ Failed: Must have exactly two inputs of type Concept")
            return False
            
        concept1, concept2 = inputs
        
        # Note(_; 3/27): Removed a check for the same concept name as this is not strong enough to tell if two concepts are different with ground truth entity renaming.

        if verbose:
            print(f"✓ Inputs are concepts: {concept1.name}, {concept2.name}")
        
        # Check that both are the same type (function or predicate)
        if (concept1.examples.example_structure.concept_type != 
            concept2.examples.example_structure.concept_type):
            if verbose:
                print("❌ Failed: Both concepts must be the same type (function or predicate)")
            return False
            
        # Check arities match
        if (concept1.examples.example_structure.input_arity != 
            concept2.examples.example_structure.input_arity):
            if verbose:
                print("❌ Failed: Concepts must have the same arity")
            return False
            
        # Check types match
        if concept1.examples.example_structure.component_types != concept2.examples.example_structure.component_types:
            if verbose:
                print("❌ Failed: Concepts must have matching types")
            return False

        # Note(_; 2/15): This is written assuming the tables are not populated with precisely the same inputs
        # For example, whenever generating a new counterexample, all entities with the same example structure will add this
        # example to their table, as a global consistency check and to ensure that there is a distinction between
        # an implication conjecture and an equivalence conjecture. 
        
        # Check example consistency
        examples1 = set(ex.value for ex in concept1.examples.get_examples())
        examples2 = set(ex.value for ex in concept2.examples.get_examples())
        common_examples = examples1.intersection(examples2)
        
        nonexamples1 = set(ex.value for ex in concept1.examples.get_nonexamples())
        nonexamples2 = set(ex.value for ex in concept2.examples.get_nonexamples())
        common_nonexamples = nonexamples1.intersection(nonexamples2)
        # Check for contradictions
        if examples1.intersection(nonexamples2) or examples2.intersection(nonexamples1):
            if verbose:
                print("❌ Failed: Concepts have contradicting examples/nonexamples")
            return False
            
        # Ensure there are some shared examples or nonexamples
        # TODO(_; 4/22): We can't keep this rn, filling out (non)examples is not robust over many PR applications
        # if not (common_examples or common_nonexamples):
        #     if verbose:
        #         print("❌ Failed: Concepts have no common examples or nonexamples")
        #     return False
                
        if verbose:
            print("✓ Valid concepts for equivalence")
        return True

    def _z3_translate_equivalence(self, concept1: Concept, concept2: Concept, *args) -> Z3Template:
        """Translate an equivalence (P <-> Q or f == g) to a Z3 program."""
        try:
            arity = concept1.get_input_arity() # Arity must be the same
            is_predicate = concept1.examples.example_structure.concept_type == ConceptType.PREDICATE

            template1 = concept1.to_z3(*([None] * arity))
            program1 = template1.program
            template2 = concept2.to_z3(*([None] * arity))
            program2 = template2.program

            quantified_args = "[" + ", ".join([f"b_{i}" for i in range(arity)]) + "]"
            params = [f"b_{i}" for i in range(arity)]
            args_string = _format_args(params)

            if is_predicate:
                # Predicate equivalence: P <-> Q
                code = f"""
                params 0;
                bounded params {arity};
                p_0 := Pred({program1.dsl()});
                p_1 := Pred({program2.dsl()});
                ReturnExpr None;
                ReturnPred ForAll({quantified_args}, And(Implies(p_0({args_string}), p_1({args_string})), Implies(p_1({args_string}), p_0({args_string}))));
                """
            else:
                # Function equivalence: f(x) == g(x)
                code = f"""
                params 0;
                bounded params {arity};
                f_0 := Func({program1.dsl()});
                f_1 := Func({program2.dsl()});
                ReturnExpr None;
                ReturnPred ForAll({quantified_args}, f_0({args_string}) == f_1({args_string}));
                """

            template = Z3Template(code=code)
            template.set_args(*args) # Pass any top-level args if needed
            return template
        except Exception as e:
            print(f"Error translating equivalence {concept1.name} <-> {concept2.name}: {e}")
            return None

    def apply(self, *inputs: Entity) -> Entity:
        """
        Apply the equivalence rule to create a new conjecture.
        
        For functions f, g: A₁ × ... × Aₙ → B:
        ∀x₁...xₙ₋₁. f(x₁,...,xₙ₋₁) = g(x₁,...,xₙ₋₁)
        
        For predicates P, Q: A₁ × ... × Aₙ → Bool:
        ∀x₁...xₙ. P(x₁,...,xₙ) ↔ Q(x₁,...,xₙ)
        """
        if not self.can_apply(*inputs, verbose=False):
            raise ValueError("Cannot apply Equivalence to these inputs")
            
        concept1, concept2 = inputs
        arity = concept1.examples.example_structure.input_arity
        is_predicate = concept1.examples.example_structure.concept_type == ConceptType.PREDICATE
        
        # Determine verification capabilities
        can_add_examples, can_add_nonexamples = self.determine_verification_capabilities(*inputs)
        
        # Create variables for quantification
        vars = [Var(f"x{i}") for i in range(arity)]
        
        # Build the equivalence expression
        if is_predicate:
            # For predicates: P ↔ Q is equivalent to (P → Q) ∧ (Q → P)
            equiv_expr = And(
                Implies(
                    ConceptApplication(concept1, *vars),
                    ConceptApplication(concept2, *vars)
                ),
                Implies(
                    ConceptApplication(concept2, *vars),
                    ConceptApplication(concept1, *vars)
                )
            )
        else:
            # For functions: direct equality
            input_arity = concept1.examples.example_structure.input_arity
            equiv_expr = Equals(
                ConceptApplication(concept1, *vars[:input_arity]),  # Only use input variables
                ConceptApplication(concept2, *vars[:input_arity])   # Only use input variables
            )
        
        # Wrap with universal quantifiers
        expr = equiv_expr
        for i in reversed(range(arity)):
            expr = Forall(f"x{i}", NatDomain(), expr)
        
        # Create the conjecture with the same example structure as the input concepts
        conjecture = Conjecture(
            name=f"equiv_({concept1.name}_{concept2.name})",
            description=f"Conjecture that {concept1.name} and {concept2.name} are equivalent",
            symbolic_definition=lambda: expr,
            example_structure=concept1.examples.example_structure,
            can_add_examples=can_add_examples,
            can_add_nonexamples=can_add_nonexamples,
            z3_translation=(lambda *args: self._z3_translate_equivalence(concept1, concept2, *args))
            if concept1.has_z3_translation() and concept2.has_z3_translation()
            else None
        )
        conjecture.map_iterate_depth = max(concept1.map_iterate_depth, concept2.map_iterate_depth)
        return conjecture

def create_subtraction_concept():
    """Helper to create a subtraction concept"""
    from frame.knowledge_base.demonstrations import addition
    
    concept = Concept(
        name="subtraction",
        description="Subtraction of natural numbers (returns 0 if result would be negative)",
        symbolic_definition=lambda a, b: 
            Exists("k", NatDomain(),
                And(
                    ConceptApplication(addition, b, Var("k")),
                    Equals(Var("k"), a)
                )
            ),
        computational_implementation=lambda a, b: max(0, a - b),
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC, ExampleType.NUMERIC),
            input_arity=2
        )
    )
    
    # Add examples
    concept.add_example((5, 3, 2))  # 5 - 3 = 2
    concept.add_example((7, 4, 3))  # 7 - 4 = 3
    concept.add_example((3, 5, 0))  # 3 - 5 = 0 (no negative numbers)
    concept.add_example((10, 7, 3))  # 10 - 7 = 3
    
    return concept

def create_power_concept():
    """Helper to create a power concept"""
    from frame.knowledge_base.demonstrations import multiplication
    from frame.knowledge_base.initial_concepts import one_concept
    
    concept = Concept(
        name="power",
        description="Power function a^n defined by repeated multiplication",
        symbolic_definition=lambda a, n:
            Fold(n, one_concept.symbolic(), Lambda("x", multiplication(Var("x"), a))),
        computational_implementation=lambda a, n: a ** n,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC, ExampleType.NUMERIC),
            input_arity=2
        )
    )
    
    # Add examples
    concept.add_example((2, 3, 8))   # 2^3 = 8
    concept.add_example((3, 2, 9))   # 3^2 = 9
    concept.add_example((2, 4, 16))  # 2^4 = 16
    concept.add_example((5, 2, 25))  # 5^2 = 25
    
    return concept

def create_mersenne_form_concept():
    """Helper to create a concept for numbers of the form 2^k * (2^(k+1) - 1) where 2^(k+1) - 1 is prime"""
    from frame.knowledge_base.demonstrations import is_prime, multiplication, addition
    
    power = create_power_concept()
    subtraction = create_subtraction_concept()
    
    concept = Concept(
        name="is_mersenne_form",
        description="Tests if n is of the form 2^k * (2^(k+1) - 1) where 2^(k+1) - 1 is prime",
        symbolic_definition=lambda n:
            Exists("k", NatDomain(),
                And(
                    ConceptApplication(is_prime,
                        ConceptApplication(subtraction,
                            ConceptApplication(power, Nat(2), 
                                ConceptApplication(addition, Var("k"), Nat(1))
                            ),
                            Nat(1)
                        )
                    ),
                    Equals(n,
                        ConceptApplication(multiplication,
                            ConceptApplication(power, Nat(2), Var("k")),
                            ConceptApplication(subtraction,
                                ConceptApplication(power, Nat(2), 
                                    ConceptApplication(addition, Var("k"), Nat(1))
                                ),
                                Nat(1)
                            )
                        )
                    )
                )
            ),
        computational_implementation=lambda n:
            any(
                is_prime.compute(2**(k+1) - 1) and
                n == 2**k * (2**(k+1) - 1)
                for k in range(1, int(n ** 0.5) + 1)
            ),
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,),
            input_arity=1
        ),
        lean4_translation=lambda n: f"(∃ k : ℕ, is_prime(2^(k+1) - 1) ∧ {n} = 2^k * (2^(k+1) - 1))",
        prolog_translation=lambda n: f"exists(k, nat, (is_prime(minus(power(2, plus(k, 1)), 1)), equals({n}, times(power(2, k), minus(power(2, plus(k, 1)), 1)))))",
        z3_translation=lambda n: f"(exists ((k Int)) (and (is_prime (- (^ 2 (+ k 1)) 1)) (= {n} (* (^ 2 k) (- (^ 2 (+ k 1)) 1)))))"
    )
    
    # Add examples (first few even perfect numbers)
    concept.add_example((6,))    # 6 = 2^1 * (2^2 - 1), and 3 is prime
    concept.add_example((28,))   # 28 = 2^2 * (2^3 - 1), and 7 is prime
    concept.add_example((496,))  # 496 = 2^4 * (2^5 - 1), and 31 is prime
    
    # Add nonexamples
    concept.add_nonexample((4,))   # Not of the required form
    concept.add_nonexample((10,))  # Not of the required form
    concept.add_nonexample((15,))  # Not of the required form
    
    return concept

def create_add1_concept():
    """Helper to create an add1 concept"""
    from frame.knowledge_base.demonstrations import addition
    from frame.knowledge_base.entities import Succ, Zero

    concept = Concept(
        name="add1",
        description="Add 1 to a natural number",
        symbolic_definition=lambda n: Lambda("x", addition(Var("x"), Succ(Zero()))),
        computational_implementation=lambda n: n + 1,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC),
            input_arity=1
        )
    )
    
    # Add examples
    concept.add_example((0, 1))  # 0 + 1 = 1
    concept.add_example((1, 2))  # 1 + 1 = 2
    concept.add_example((2, 3))  # 2 + 1 = 3
    concept.add_example((3, 4))  # 3 + 1 = 4
    concept.add_example((4, 5))  # 4 + 1 = 5
    
    return concept

def test_equivalence():
    """Test the Equivalence production rule"""
    from frame.knowledge_base.initial_concepts import create_successor_concept
    print("\n=== Testing Equivalence Production Rule ===")
    
    # Test predicate equivalence
    from frame.knowledge_base.demonstrations import is_perfect, is_even
    even_perfect = Concept(
        name="is_even_perfect",
        description="Tests if n is an even perfect number",
        symbolic_definition=lambda n:
            And(
                ConceptApplication(is_even, n),
                ConceptApplication(is_perfect, n)
            ),
        computational_implementation=lambda n:
            is_even.compute(n) and is_perfect.compute(n),
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,),
            input_arity=1
        )
    )
    
    # Add examples (first few even perfect numbers)
    even_perfect.add_example((6,))
    even_perfect.add_example((28,))
    even_perfect.add_example((496,))
    
    # Add nonexamples
    even_perfect.add_nonexample((4,))   # Even but not perfect
    even_perfect.add_nonexample((10,))  # Even but not perfect
    even_perfect.add_nonexample((15,))  # Not even
    
    # Create concept for Mersenne form numbers
    mersenne_form = create_mersenne_form_concept()
    
    # Test function equivalence
    add1 = create_add1_concept()
    successor = create_successor_concept()

    # Create and apply Equivalence
    equivalence = EquivalenceRule()
    
    # Test get_input_types
    print("\nTesting get_input_types:")
    input_types = equivalence.get_input_types()
    print(f"Valid input types: {input_types}")
    
    # Test predicate equivalence
    print("\nTesting predicate equivalence (even_perfect ↔ mersenne_form)...")
    can_apply = equivalence.can_apply(even_perfect, mersenne_form)
    print(f"Can apply: {can_apply}")
    
    if can_apply:
        print("\nApplying rule to create predicate conjecture...")
        conjecture = equivalence.apply(even_perfect, mersenne_form)
        
        # Show symbolic form
        print("\nSymbolic form of conjecture:")
        symbolic_expr = conjecture.symbolic()
        
    
    # Test function equivalence
    print("\nTesting function equivalence (add1 ↔ successor)...")
    can_apply = equivalence.can_apply(add1, successor)
    print(f"Can apply: {can_apply}")
    
    if can_apply:
        print("\nApplying rule to create function conjecture...")
        conjecture = equivalence.apply(add1, successor)
        print("\nSymbolic form of function conjecture:")
        symbolic_expr = conjecture.symbolic()

        
        return conjecture
    return None

# ==============================
# Z3 TEST HELPERS AND TEST
# ==============================

# --- Helpers with Z3 ---
def create_is_even_concept_z3():
    """Helper: is_even concept with Z3 translation."""
    concept = Concept(
        name="is_even_z3",
        description="Tests if a number is even (Z3)",
        symbolic_definition=lambda n: n % 2 == 0, # Simplified for testing
        computational_implementation=lambda n: n % 2 == 0,
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,), input_arity=1
        ),
        z3_translation=lambda n: Z3Template("""
            params 1; 
            bounded params 0; 
            ReturnExpr None; 
            ReturnPred x_0 % 2 == 0;
            """, 
            n)
    )
    concept.add_example((0,)); concept.add_example((2,)); concept.add_nonexample((1,)); concept.add_nonexample((3,))
    return concept

def create_is_square_concept_z3():
    """Helper: is_square concept with Z3 translation."""
    concept = Concept(
        name="is_square_z3",
        description="Tests if a number is a perfect square (Z3)",
        symbolic_definition=lambda n: Exists("k", NatDomain(), Equals(n, ConceptApplication(multiplication, Var("k"), Var("k")))),
        computational_implementation=lambda n: n >= 0 and int(n**0.5)**2 == n,
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,), input_arity=1
        ),
        # Note: Z3 needs explicit quantification for exists. Assuming integers >= 0.
        z3_translation=lambda n: Z3Template(
            """
            params 1;
            bounded params 0;
            ReturnExpr None;
            ReturnPred Exists([b_0], And(b_0 >= 0, b_0 * b_0 == x_0));
            """, n
            ) 
    )
    concept.add_example((0,)); concept.add_example((1,)); concept.add_example((4,)); concept.add_nonexample((2,)); concept.add_nonexample((3,))
    return concept


def create_is_even_square_concept_z3():
    """Helper: is_even_square concept using is_even and is_square with Z3."""
    is_even_c = create_is_even_concept_z3()
    is_square_c = create_is_square_concept_z3()
    concept = Concept(
        name="is_even_square_z3",
        description="Tests if a number's square is an even number (Z3)",
        symbolic_definition=lambda n: And(ConceptApplication(is_even_c, n), ConceptApplication(is_square_c, n)),
        computational_implementation=lambda n: (n % 2 == 0) and (n >= 0 and int(n**0.5)**2 == n),
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,), input_arity=1
        ),
        z3_translation=lambda n: Z3Template(
            f"""
            params 1;
            bounded params 0;
            ReturnExpr None;
            ReturnPred x_0 * x_0 % 2 == 0;
            """, n
            )
    )
    concept.add_example((0,)); concept.add_example((4,)); concept.add_example((16,));
    concept.add_nonexample((1,)); concept.add_nonexample((9,))
    return concept


def create_divisible_by_4_concept_z3():
    """Helper: is_divisible_by_4 concept with Z3 translation."""
    concept = Concept(
        name="is_divisible_by_4_z3",
        description="Tests if a number is divisible by 4 (Z3)",
        symbolic_definition=lambda n: ConceptApplication(divides, Nat(4), n),
        computational_implementation=lambda n: n % 4 == 0,
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,), input_arity=1
        ),
        z3_translation=lambda n: Z3Template("""
            params 1; 
            bounded params 0; 
            ReturnExpr None; 
            ReturnPred x_0 % 4 == 0;
            """, n)
    )
    concept.add_example((0,)); concept.add_example((4,)); concept.add_nonexample((1,))
    return concept

def create_double_concept_z3():
    """Helper: double(x) = 2 * x function with Z3."""
    concept = Concept(
        name="double_z3",
        description="Multiplies input by 2 (Z3)",
        symbolic_definition=lambda x: ConceptApplication(multiplication, Nat(2), x),
        computational_implementation=lambda x: 2 * x,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC), input_arity=1
        ),
        z3_translation=lambda x: Z3Template("""
            params 1; 
            bounded params 0; 
            ReturnExpr 2 * x_0; 
            ReturnPred None;
            """, x)
    )
    concept.add_example((0, 0)); concept.add_example((1, 2)); concept.add_example((5, 10))
    return concept

def create_add_self_concept_z3():
    """Helper: add_self(x) = x + x function with Z3."""
    concept = Concept(
        name="add_self_z3",
        description="Adds input to itself (Z3)",
        symbolic_definition=lambda x: ConceptApplication(addition, x, x),
        computational_implementation=lambda x: x + x,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC), input_arity=1
        ),
        z3_translation=lambda x: Z3Template("""
            params 1; 
            bounded params 0; 
            ReturnExpr x_0 + x_0; 
            ReturnPred None;
            """, x)
    )
    concept.add_example((0, 0)); concept.add_example((1, 2)); concept.add_example((5, 10))
    return concept

def create_add_two_concept_z3():
    """Helper: add_two(x) = x + 2 function with Z3."""
    concept = Concept(
        name="add_two_z3",
        description="Adds 2 to input (Z3)",
        symbolic_definition=lambda x: ConceptApplication(addition, x, Nat(2)),
        computational_implementation=lambda x: x + 2,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC), input_arity=1
        ),
        z3_translation=lambda x: Z3Template("""
            params 1; 
            bounded params 0; 
            ReturnExpr x_0 + 2; 
            ReturnPred None;
            """, x)
    )
    concept.add_example((0, 2)); concept.add_example((1, 3)); concept.add_example((5, 7))
    return concept


# --- Z3 Test Function ---
def test_equivalence_z3():
    """Test the Equivalence production rule with Z3 translation."""
    print("\n--- Testing Z3 Translation for Equivalence ---")
    equivalence_z3 = EquivalenceRule()

    # Concepts for testing
    is_even = create_is_even_concept_z3()
    is_even_square = create_is_even_square_concept_z3()
    is_div4 = create_divisible_by_4_concept_z3()
    double = create_double_concept_z3()
    add_self = create_add_self_concept_z3()
    add_two = create_add_two_concept_z3()


    # Test Case 1: Predicate Equivalence (False: is_even_square <-> is_even)
    # An even number is not necessarily an even square (e.g., 2 is even but not square).
    print(f"\nTesting Z3 Predicate: {is_even_square.name} <-> {is_even.name} (Should be Falsifiable)")
    if equivalence_z3.can_apply(is_even_square, is_even, verbose=False):
        equiv_conj_false1 = equivalence_z3.apply(is_even_square, is_even)
        try:
            template = equiv_conj_false1.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert result.proved, f"Z3 should prove {equiv_conj_false1.name}"
            print("✓ Z3 Predicate Test Passed")
        except Exception as e: print(f"❌ Z3 Predicate Test Failed: {type(e).__name__} - {e}")
    else: print(f"❌ Cannot apply rule for Z3 test: {is_even_square.name} <-> {is_even.name}")


    # Test Case 2: Predicate Equivalence (False: is_even <-> is_divisible_by_4)
    # An even number is not necessarily divisible by 4 (e.g., 2, 6).
    print(f"\nTesting Z3 Predicate: {is_even.name} <-> {is_div4.name} (Should be Falsifiable)")
    if equivalence_z3.can_apply(is_even, is_div4, verbose=False):
        equiv_conj_false2 = equivalence_z3.apply(is_even, is_div4)
        try:
            template = equiv_conj_false2.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert not result.proved, f"Z3 should falsify {equiv_conj_false2.name}"
            print("✓ Z3 Predicate Test Passed (Falsified as expected)")
        except Exception as e: print(f"❌ Z3 Predicate Test Failed: {type(e).__name__} - {e}")
    else: print(f"❌ Cannot apply rule for Z3 test: {is_even.name} <-> {is_div4.name}")


    # Test Case 3: Function Equivalence (True: double == add_self, i.e., 2*x == x+x)
    print(f"\nTesting Z3 Function: {double.name} == {add_self.name} (Should be Provable)")
    if equivalence_z3.can_apply(double, add_self, verbose=False):
        equiv_func_true = equivalence_z3.apply(double, add_self)
        try:
            template = equiv_func_true.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert result.proved, f"Z3 should prove {equiv_func_true.name}"
            print("✓ Z3 Function Test Passed (Proved True as expected)")
        except Exception as e: print(f"❌ Z3 Function Test Failed: {type(e).__name__} - {e}")
    else: print(f"❌ Cannot apply rule for Z3 test: {double.name} == {add_self.name}")


    # Test Case 4: Function Equivalence (False: double == add_two, i.e., 2*x == x+2)
    # This is only true for x=2. Should be falsified universally.
    print(f"\nTesting Z3 Function: {double.name} == {add_two.name} (Should be Falsifiable)")
    if equivalence_z3.can_apply(double, add_two, verbose=False):
        equiv_func_false = equivalence_z3.apply(double, add_two)
        try:
            template = equiv_func_false.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert not result.proved, f"Z3 should falsify {equiv_func_false.name}"
            print("✓ Z3 Function Test Passed (Falsified as expected)")
        except Exception as e: print(f"❌ Z3 Function Test Failed: {type(e).__name__} - {e}")
    else: print(f"❌ Cannot apply rule for Z3 test: {double.name} == {add_two.name}")


if __name__ == "__main__":
    standard_conjecture = test_equivalence() # Keep standard tests if desired
    test_equivalence_z3() # Run the Z3 tests 