"""
This module implements the Nonexistence production rule for generating conjectures about the
nonexistence of inputs satisfying certain properties.

For example:
- "There are no odd perfect numbers"
- "There are no prime numbers which are also square numbers"
"""

from typing import List, Optional, Set, Tuple, Any, Dict, Type
from frame.productions.base import ProductionRule
from frame.knowledge_base.entities import (
    Expression, Var, ConceptApplication, Forall, NatDomain,
    Entity, Concept, Conjecture, Example, ExampleType, ExampleStructure, ConceptType,
    Equals, Not, And, Exists, Nat
)
from frame.knowledge_base.demonstrations import (
    is_perfect, is_odd, is_prime, multiplication, addition
)
from frame.tools.z3_template import Z3Template, _format_args

class NonexistenceRule(ProductionRule):
    """
    Production rule that takes a concept and produces a conjecture stating that there are no inputs that satisfy the concept.
    
    For predicates P: A₁ × ... × Aₙ → Bool, produces:
    ∀x₁...xₙ. ¬P(x₁,...,xₙ)
    
    For functions f: A₁ × ... × Aₙ → B and value concept v: () → B, produces:
    ∀x₁...xₙ. ¬(f(x₁,...,xₙ) = v())
    
    Example:
    - Given is_odd_perfect(n) = is_odd(n) ∧ is_perfect(n)
    - Produces conjecture that ∀n. ¬(is_odd(n) ∧ is_perfect(n))
    """
    
    def __init__(self):
        super().__init__(
            name="nonexistence",
            description="Creates a conjecture stating that no inputs satisfy a concept",
            type="Conjecture"
        )
    
    def determine_verification_capabilities(self, *inputs: Entity) -> Tuple[bool, bool]:
        """
        Determine verification capabilities for nonexistence conjectures.
        
        For nonexistence conjectures (∀x. ¬P(x)), verification capabilities depend on:
        - For examples: Since nonexistence conjectures assert universal negation,
          we can verify examples (which are instances where the negation holds) only
          if we can verify nonexamples of the original concept.
        - For nonexamples: Nonexamples of a nonexistence conjecture would be
          counterexamples to the original concept, which we can verify if we can
          verify examples of the original concept.
        
        Returns:
            Tuple[bool, bool]: (can_add_examples, can_add_nonexamples)
        """
        if not inputs:
            return False, False
            
        concept = inputs[0]
        
        # For nonexistence conjectures:
        # - Examples of "No x.P(x)" are instances where P(x) is false,
        #   which are nonexamples of the original concept P.
        # - Nonexamples of "No x.P(x)" are instances where P(x) is true,
        #   which are examples of the original concept P.
        concept_can_add_examples = concept.can_add_examples
        concept_can_add_nonexamples = concept.can_add_nonexamples
        
        # We flip the verification capabilities:
        # - We can verify examples of "No x.P(x)" if we can verify nonexamples of P
        # - We can verify nonexamples of "No x.P(x)" if we can verify examples of P
        return concept_can_add_nonexamples, concept_can_add_examples
    
    def get_input_types(self) -> List[List[Tuple[Type, Any]]]:
        """
        Returns the valid input types for this production rule.
        
        Returns:
            A list of lists, where each inner list represents a valid combination of input types.
            Each element in the inner list is a tuple of (type, subtype).
        """
        return [
            [(Concept, ConceptType.PREDICATE)],  # Single predicate concept
            [(Concept, ConceptType.FUNCTION), (Concept, ConceptType.FUNCTION)]  # Function concept and a target value concept
        ]
    
    def get_valid_parameterizations(self, *inputs: Entity) -> List[Dict[str, Any]]:
        """
        Get valid parameterizations for applying the nonexistence rule to the given concept(s).
        
        Args:
            *inputs: Either a single predicate concept or a function concept and a target value concept
            
        Returns:
            List of valid parameter dictionaries (empty dict for this rule as it requires no additional parameters)
        """
        # Check if the rule can be applied to these inputs
        if not self.can_apply(*inputs, verbose=False):
            return [] # Invalid inputs
        return [{}]
    
    def can_apply(self, *inputs: Entity, verbose: bool = True) -> bool:
        """
        Check if this rule can be applied to the inputs.
        
        Requirements:
        1. Either a single predicate concept or a function concept and a target value concept
        2. For predicates: no positive examples
        3. For functions: no examples mapping to target_value
        4. Must have some nonexamples to support conjecture
        5. Target value concept must have no inputs and one output (Note(_; 2/27): We should probably add a constant type because its usage is required in
        
        Args:
            *inputs: Input entities to check
            verbose: Whether to print debug information
        """
        if verbose:
            print("Checking requirements:")
        
        # Check input structure
        if len(inputs) == 1:
            # Single predicate case
            if not isinstance(inputs[0], Concept):
                if verbose:
                    print("❌ Failed: Input must be a Concept")
                return False
                
            concept = inputs[0]
            
            # Check that it's a predicate
            if concept.examples.example_structure.concept_type != ConceptType.PREDICATE:
                if verbose:
                    print("❌ Failed: Single input must be a predicate concept")
                return False
                
            if verbose:
                print(f"✓ Input is a predicate concept: {concept.name}")
            
            # Check that there are no positive examples
            if concept.examples.get_examples():
                if verbose:
                    print("❌ Failed: Predicate has positive examples")
                return False
                
        elif len(inputs) == 2:
            # Function and target value case
            if not isinstance(inputs[0], Concept) or not isinstance(inputs[1], Concept):
                if verbose:
                    print("❌ Failed: Both inputs must be Concepts")
                return False
                
            concept = inputs[0]
            target_concept = inputs[1]
            
            # Check that first input is a function
            if concept.examples.example_structure.concept_type != ConceptType.FUNCTION:
                if verbose:
                    print("❌ Failed: First input must be a function concept")
                return False
                
            # Check that target concept is a function
            if target_concept.examples.example_structure.concept_type != ConceptType.FUNCTION:
                if verbose:
                    print("❌ Failed: Target value must be a function concept")
                return False
                
            # Check that target concept has no inputs and one output
            if target_concept.examples.example_structure.input_arity != 0:
                if verbose:
                    print("❌ Failed: Target value concept must have no inputs")
                return False
                
            # Check that output types match
            concept_output_type = concept.examples.example_structure.component_types[-1]
            target_output_type = target_concept.examples.example_structure.component_types[-1]
            
            if concept_output_type != target_output_type:
                if verbose:
                    print("❌ Failed: Output types of function and target value must match")
                return False
                
            if verbose:
                print(f"✓ Input is a function concept: {concept.name} with target value concept: {target_concept.name}")
            
            # Get target value from target concept
            target_examples = list(target_concept.examples.get_examples())
            if not target_examples:
                if verbose:
                    print("❌ Failed: Target value concept must have at least one example")
                return False
                
            target_value = target_examples[0].value[-1]
            
            # Check that no examples map to the target value
            for ex in concept.examples.get_examples():
                if ex.value[-1] == target_value:  # Check output value
                    if verbose:
                        print(f"❌ Failed: Function has examples mapping to target value {target_value}")
                    return False
        else:
            if verbose:
                print("❌ Failed: Must have either one predicate concept or a function concept and a target value concept")
            return False
            
        # Must have some nonexamples to support conjecture
        # Note(_; 5/4): we don't always have nonexamples
        if not concept.examples.get_nonexamples():
            if verbose:
                print("❌ Failed: Must have nonexamples to support nonexistence")
            return False
            
        if verbose:
            print("✓ Valid concept for nonexistence conjecture")
        return True

    def apply(self, *inputs: Entity) -> Entity:
        """
        Apply the nonexistence rule to create a new conjecture.
        
        For predicates P: A₁ × ... × Aₙ → Bool:
        ∀x₁...xₙ. ¬P(x₁,...,xₙ)
        
        For functions f: A₁ × ... × Aₙ → B and value concept v: () → B:
        ∀x₁...xₙ. ¬(f(x₁,...,xₙ) = v())
        """
        if not self.can_apply(*inputs, verbose=False):
            raise ValueError("Cannot apply Nonexistence to these inputs")
            
        if len(inputs) == 1:
            # Predicate case
            concept = inputs[0]
            target_concept = None
            target_value = None
        else:
            # Function case
            concept, target_concept = inputs
            # Get target value from target concept
            target_examples = list(target_concept.examples.get_examples())
            target_value = target_examples[0].value[-1]
            
        arity = concept.examples.example_structure.input_arity
        
        # Determine verification capabilities
        can_add_examples, can_add_nonexamples = self.determine_verification_capabilities(*inputs)
        
        # Create variables for quantification
        vars = [Var(f"x{i}") for i in range(arity)]
        
        # Build the nonexistence expression
        if concept.examples.example_structure.concept_type == ConceptType.PREDICATE:
            nonexist_expr = Not(ConceptApplication(concept, *vars))
            name_suffix = ""
        else:
            nonexist_expr = Not(
                Equals(ConceptApplication(concept, *vars), ConceptApplication(target_concept))
            )
            name_suffix = f"_equals_{target_concept.name}"
        
        # Wrap with universal quantifiers
        expr = nonexist_expr
        for i in reversed(range(arity)):
            expr = Forall(f"x{i}", NatDomain(), expr)
        
        # Create the conjecture with the same example structure as the input concept
        conjecture = Conjecture(
            name=f"no_({concept.name}{name_suffix})",
            description=f"Conjecture that no inputs to {concept.name} equal {target_concept.name if target_concept else ''}",
            symbolic_definition=lambda: expr,
            example_structure=concept.examples.example_structure,
            can_add_examples=can_add_examples,
            can_add_nonexamples=can_add_nonexamples,
            z3_translation=
                (lambda *args:
                    self._z3_translate_nonexistence_predicate(concept, *args))
                if concept.examples.example_structure.concept_type == ConceptType.PREDICATE and concept.has_z3_translation()
                else (lambda *args:
                    self._z3_translate_nonexistence_function(concept, target_concept, *args))
                if concept.has_z3_translation()
                else None
        )
        conjecture.map_iterate_depth = concept.map_iterate_depth
        return conjecture

    def _z3_translate_nonexistence_predicate(self, concept: Concept, *args) -> Z3Template:
        """Translate the negation of a predicate to a Z3 program."""
        try:
            concept_template = concept.to_z3(*([None] * concept.get_input_arity()))
            program = concept_template.program
            quantified_args = "[" + ", ".join([f"b_{i}" for i in range(concept.get_input_arity())]) + "]"
            params = [f"b_{i}" for i in range(concept.get_input_arity())]
            args_string = _format_args(params)
            template = Z3Template(
                code = f"""
                params 0;
                bounded params {concept.get_input_arity()};
                p_0 := Pred({program.dsl()});
                ReturnExpr None;
                ReturnPred ForAll({quantified_args}, Not(p_0({args_string})));
            """)
            template.set_args(*args)
            return template
        except Exception as e:
            print(f"Error translating nonexistence predicate: {e}")
            return None
    
    def _z3_translate_nonexistence_function(self, concept: Concept, target_concept: Concept, *args) -> Z3Template:
        """Translate the negation of a function to a Z3 program."""
        try:
            concept_template = concept.to_z3(*([None] * concept.get_input_arity()))
            program = concept_template.program
            target_template = target_concept.to_z3()
            target_program = target_template.program
            quantified_args = "[" + ", ".join([f"b_{i}" for i in range(concept.get_input_arity())]) + "]"
            params = [f"b_{i}" for i in range(concept.get_input_arity())]
            args_string = _format_args(params)
            template = Z3Template(
                code = f"""
                params 0;
                bounded params {concept.get_input_arity()};
                f_0 := Func({program.dsl()});
                f_1 := Func({target_program.dsl()});
                ReturnExpr None;
                ReturnPred ForAll({quantified_args}, Not(f_0({args_string}) == f_1()));
            """)
            template.set_args(*args)
            return template
        except Exception as e:
            print(f"Error translating nonexistence function: {e}")
            return None    

def create_odd_perfect_concept():
    """Helper to create a concept for odd perfect numbers"""

    concept = Concept(
        name="is_odd_perfect",
        description="Tests if a number is both odd and perfect",
        symbolic_definition=lambda n: And(
            ConceptApplication(is_odd, n),
            ConceptApplication(is_perfect, n)
        ),
        computational_implementation=lambda n:
            is_odd.compute(n) and is_perfect.compute(n),
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,),
            input_arity=1
        )
    )
    
    # Add nonexamples (no known examples exist!)
    concept.add_nonexample((1,))  # Not perfect
    concept.add_nonexample((3,))  # Not perfect
    concept.add_nonexample((5,))  # Not perfect
    concept.add_nonexample((6,))  # Not odd
    concept.add_nonexample((28,))  # Not odd
    
    return concept

def create_prime_square_concept():
    """Helper to create a concept for prime square numbers"""
    # For now, create it directly for testing
    concept = Concept(
        name="is_prime_square",
        description="Tests if a number is both prime and a perfect square",
        symbolic_definition=lambda n: And(
            ConceptApplication(is_prime, n),
            Exists("k", NatDomain(),
                Equals(n, ConceptApplication(multiplication, Var("k"), Var("k")))
            )
        ),
        computational_implementation=lambda n:
            is_prime.compute(n) and int(n ** 0.5) ** 2 == n,
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,),
            input_arity=1
        )
    )
    
    # Add nonexamples (no examples exist - a square number has at least two factors)
    concept.add_nonexample((1,))  # Not prime
    concept.add_nonexample((2,))  # Not square
    concept.add_nonexample((3,))  # Not square
    concept.add_nonexample((4,))  # Not prime
    concept.add_nonexample((9,))  # Not prime
    concept.add_nonexample((16,))  # Not prime
    concept.add_nonexample((25,))  # Not prime
    
    return concept

def create_function_concept():
    """Helper to create a function concept for testing nonexistence with target value"""
    concept = Concept(
        name="add_one",
        description="Adds 1 to a number",
        symbolic_definition=lambda n: ConceptApplication(addition, n, Nat(1)),
        computational_implementation=lambda n: n + 1,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC),
            input_arity=1
        )
    )
    
    # Add examples
    concept.add_example((1, 2))  # 1 + 1 = 2
    concept.add_example((2, 3))  # 2 + 1 = 3
    concept.add_example((3, 4))  # 3 + 1 = 4
    concept.add_example((4, 5))  # 4 + 1 = 5
    
    # Add nonexamples - inputs that don't produce specific outputs
    concept.add_nonexample((1, 1))  # 1 + 1 ≠ 1
    concept.add_nonexample((2, 2))  # 2 + 1 ≠ 2
    concept.add_nonexample((3, 3))  # 3 + 1 ≠ 3
    concept.add_nonexample((0, 0))  # 0 + 1 ≠ 0
    
    return concept

def create_constant_concept(value, name=None):
    """Helper to create a constant concept (function with no inputs) for a specific value"""
    name = name or f"constant_{value}"
    
    concept = Concept(
        name=name,
        description=f"Constant function that always returns {value}",
        symbolic_definition=lambda: Nat(value),
        computational_implementation=lambda: value,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC,),  # Just the output type
            input_arity=0  # No inputs
        ),
        z3_translation=lambda: Z3Template(
            f"""
            params 0;
            bounded params 0;
            ReturnExpr {value};
            ReturnPred None;
            """,
            value # Pass the argument to the template
        )
    )
    
    # Add an example
    concept.add_example((value,))  # Just the output value
    
    return concept

def test_nonexistence():
    """Test the Nonexistence production rule"""
    print("\n=== Testing Nonexistence Production Rule ===")
    
    # Test the get_input_types method
    nonexistence = NonexistenceRule()
    input_types = nonexistence.get_input_types()
    print("\nTesting get_input_types:")
    print(f"Valid input types: {input_types}")
    
    # Test 1: No odd perfect numbers (predicate case)
    print("\n--- Testing Odd Perfect Numbers (Predicate Case) ---")
    odd_perfect = create_odd_perfect_concept()
    print("\nCreated odd_perfect concept with examples:")
    print(f"Examples: {[ex.value for ex in odd_perfect.examples.get_examples()]}")
    print(f"Nonexamples: {[ex.value for ex in odd_perfect.examples.get_nonexamples()]}")
    
    # Test get_valid_parameterizations
    print("\nTesting get_valid_parameterizations for odd_perfect:")
    valid_params = nonexistence.get_valid_parameterizations(odd_perfect)
    print(f"Valid parameterizations: {valid_params}")
    
    # Check if rule can be applied
    print("\nChecking if rule can be applied to odd_perfect...")
    can_apply = nonexistence.can_apply(odd_perfect)
    print(f"Can apply: {can_apply}")
    
    if can_apply:
        print("\nApplying rule to create no_odd_perfect_numbers conjecture...")
        no_odd_perfect = nonexistence.apply(odd_perfect)
        
        # Show symbolic form
        print("\nSymbolic form of conjecture:")
        symbolic_expr = no_odd_perfect.symbolic()
        
    
    # Test 2: No prime square numbers (predicate case)
    print("\n--- Testing Prime Square Numbers (Predicate Case) ---")
    prime_square = create_prime_square_concept()
    print("\nCreated prime_square concept with examples:")
    print(f"Examples: {[ex.value for ex in prime_square.examples.get_examples()]}")
    print(f"Nonexamples: {[ex.value for ex in prime_square.examples.get_nonexamples()]}")
    
    # Test get_valid_parameterizations
    print("\nTesting get_valid_parameterizations for prime_square:")
    valid_params = nonexistence.get_valid_parameterizations(prime_square)
    print(f"Valid parameterizations: {valid_params}")
    
    print("\nChecking if rule can be applied to prime_square...")
    can_apply = nonexistence.can_apply(prime_square)
    print(f"Can apply: {can_apply}")
    
    if can_apply:
        print("\nApplying rule to create no_prime_squares conjecture...")
        no_prime_squares = nonexistence.apply(prime_square)
        
        # Show symbolic form
        print("\nSymbolic form of conjecture:")
        symbolic_expr = no_prime_squares.symbolic()
        
    
    
    # Test 3: Function case - no number maps to 0 when add_one is applied
    print("\n--- Testing Function Case (add_one never equals 0) ---")
    add_one = create_function_concept()
    print("\nCreated add_one function concept with examples:")
    print(f"Examples: {[ex.value for ex in add_one.examples.get_examples()]}")
    print(f"Nonexamples: {[ex.value for ex in add_one.examples.get_nonexamples()]}")
    
    # Create a constant concept for the target value 0
    target_value = 0
    zero_concept = create_constant_concept(target_value, "zero")
    print(f"\nCreated zero_concept with value {target_value}:")
    print(f"Examples: {[ex.value for ex in zero_concept.examples.get_examples()]}")
    
    # Test get_valid_parameterizations
    print("\nTesting get_valid_parameterizations for add_one and zero_concept:")
    valid_params = nonexistence.get_valid_parameterizations(add_one, zero_concept)
    print(f"Valid parameterizations: {valid_params}")
    
    print(f"\nChecking if rule can be applied to add_one with zero_concept...")
    can_apply = nonexistence.can_apply(add_one, zero_concept)
    print(f"Can apply: {can_apply}")
    
    if can_apply:
        print(f"\nApplying rule to create no_add_one_equals_zero conjecture...")
        no_add_one_equals_0 = nonexistence.apply(add_one, zero_concept)
        
        # Show symbolic form
        print("\nSymbolic form of conjecture:")
        symbolic_expr = no_add_one_equals_0.symbolic()
        
    
        return no_odd_perfect, no_prime_squares, no_add_one_equals_0
    
    return no_odd_perfect, no_prime_squares, None

if __name__ == "__main__":
    no_odd_perfect, no_prime_squares, _ = test_nonexistence() 

    # --- Z3 Translation Test (x^2 != 2) ---
    print("\n--- Testing Z3 Translation (x^2 != 2) ---")
    nonexistence_z3 = NonexistenceRule()
    
    # Create concepts: square(x) = x*x and two = 2
    # Define square concept locally for this test
    square_concept = Concept(
        name="square_local",
        description="Square a number",
        symbolic_definition=lambda x: ConceptApplication(multiplication, x, x),
        computational_implementation=lambda x: x * x,
        example_structure=ExampleStructure(
            concept_type=ConceptType.FUNCTION,
            component_types=(ExampleType.NUMERIC, ExampleType.NUMERIC),
            input_arity=1,
        ),
        z3_translation=lambda x: Z3Template(
            f"""
            params 1;
            bounded params 0;
            ReturnExpr x_0 * x_0;
            ReturnPred None;
            """,
            x # Pass the argument to the template
        )
    )
    square_concept.add_example((0, 0))
    square_concept.add_example((1, 1))
    square_concept.add_example((2, 4))
    square_concept.add_nonexample((1, 2)) # 1*1 != 2
    square_concept.add_nonexample((2, 2)) # 2*2 != 2

    # Use the updated create_constant_concept for 'two'
    two_concept = create_constant_concept(2, "two")
    twentyfive_concept = create_constant_concept(25, "twentyfive")
    print("\nCreated square_concept and two_concept:")
    print(f"Square examples: {[ex.value for ex in square_concept.examples.get_examples()]}")
    print(f"Square nonexamples: {[ex.value for ex in square_concept.examples.get_nonexamples()]}")
    print(f"Two examples: {[ex.value for ex in two_concept.examples.get_examples()]}")

    # Check if rule can be applied
    print(f"\nChecking if rule can be applied to square_concept and two_concept...")
    can_apply_z3 = nonexistence_z3.can_apply(square_concept, two_concept, verbose=True)
    print(f"Can apply: {can_apply_z3}")
    
    if not can_apply_z3:
        print("❌ Cannot apply nonexistence rule, skipping Z3 test.")
    else:
        # Apply the rule
        print("\nApplying rule to create no_square_equals_two conjecture...")
        no_square_equals_2 = nonexistence_z3.apply(square_concept, two_concept)
        no_square_equals_25 = nonexistence_z3.apply(square_concept, twentyfive_concept)
        # Test Z3 translation
        print("\nGenerating and running Z3 program for the conjecture...")
        try:
            # The create_constant_concept function now adds z3_translation
            template = no_square_equals_2.to_z3()
            result = template.run()
            
            print(f"Z3 Result Proved: {result.proved}")
            assert result.proved, "Z3 should prove that ∀x. ¬(x^2 = 2) for natural numbers x"
            print("✓ Z3 Test Passed")
        except AttributeError as ae:
             print(f"❌ Z3 Test Failed due to missing attribute/method: {ae}")
             print("   Ensure Concept/Expression classes and required concepts have 'to_z3' methods or 'z3_translation'.")
        except Exception as e:
            print(f"❌ Z3 Test Failed: {type(e).__name__} - {e}") 

        try:

            template = no_square_equals_25.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert not result.proved, "Z3 should falsify that ∀x. ¬(x^2 = 25) for natural numbers x"
            print("✓ Z3 Test Passed")
        except Exception as e:
            print(f"❌ Z3 Test Failed: {type(e).__name__} - {e}") 
    # let's add a test for the predicate n < 0
    print("\n--- Testing Z3 Translation (n < 0) ---")
    less_than_zero_concept = Concept(
        name="less_than_zero",
        description="Tests if a number is less than 0",
        symbolic_definition=lambda n: Forall(), # placeholder
        computational_implementation=lambda n: n < 0,
        example_structure=ExampleStructure(
            concept_type=ConceptType.PREDICATE,
            component_types=(ExampleType.NUMERIC,),
            input_arity=1
        ),
        z3_translation=lambda n: Z3Template(
            f"""
            params 1;
            bounded params 0;
            ReturnExpr None;
            ReturnPred x_0 < 0;
            """,
            n # Pass the argument to the template
        )
    )

    less_than_zero_concept.add_nonexample((0,))

    # Check if rule can be applied
    print(f"\nChecking if rule can be applied to less_than_zero_concept...")
    can_apply_z3 = nonexistence_z3.can_apply(less_than_zero_concept, verbose=True)
    print(f"Can apply: {can_apply_z3}")

    if can_apply_z3:
        print("\nApplying rule to create no_less_than_zero conjecture...")
        no_less_than_zero = nonexistence_z3.apply(less_than_zero_concept)
        # Test Z3 translation
        print("\nGenerating and running Z3 program for the conjecture...")
        try:
            template = no_less_than_zero.to_z3()
            result = template.run()
            print(f"Z3 Result Proved: {result.proved}")
            assert result.proved, "Z3 should prove that ∀x. ¬(x < 0) for natural numbers x"
            print("✓ Z3 Test Passed")
        except Exception as e:
            print(f"❌ Z3 Test Failed: {type(e).__name__} - {e}") 