from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Optional, Any, Union

@dataclass
class ProofResult:
    """Result of a proof attempt."""
    is_proven: bool
    proof_object: Optional[Any] = None  # Could be a Lean proof term, Z3 proof, etc.
    counterexample: Optional[Any] = None  # Could be a concrete counterexample
    confidence: float = 0.0  # Confidence in the result (especially for LLM-based provers)

class BaseProver(ABC):
    """Abstract base class for theorem provers."""
    
    @abstractmethod
    def prove(self, conjecture: Any) -> ProofResult:
        """
        Attempt to prove a given conjecture.
        
        Args:
            conjecture: The conjecture to prove in our internal format
            
        Returns:
            ProofResult containing the proof status and additional information
        """
        pass
    
    @abstractmethod
    def verify_proof(self, conjecture: Any, proof: Any) -> bool:
        """
        Verify if a given proof is valid for the conjecture.
        
        Args:
            conjecture: The conjecture being proved
            proof: The proof object to verify
            
        Returns:
            bool indicating if the proof is valid
        """
        pass 