"""
This module defines the proof-related classes used across different provers.
"""

from dataclasses import dataclass
from typing import Optional, Any, Union, Literal
from abc import ABC, abstractmethod

@dataclass
class Proof(ABC):
    """Abstract base class for proofs."""
    conjecture: Any  # The conjecture being proved
    timestamp: float  # When the proof was created
    
    @abstractmethod
    def verify(self) -> bool:
        """Verify that this proof is still valid."""
        pass

@dataclass
class LeanProof(Proof):
    """A proof in Lean theorem prover format."""
    proof_term: str  # The Lean proof term
    
    def verify(self) -> bool:
        """Verify the Lean proof term."""
        # TODO: Implement Lean proof verification
        raise NotImplementedError

@dataclass
class Z3Proof(Proof):
    """A proof in Z3 SMT solver format."""
    proof_object: Any  # The Z3 proof object
    
    def verify(self) -> bool:
        """Verify the Z3 proof."""
        return self.proof_object is not None  # Z3 proofs are self-verifying 