"""Core REPL system commands.

This module contains the essential commands needed for basic REPL operation,
such as help and listing available entities.
"""

import os
from typing import List, Dict, Any, Optional
from frame.repl.commands.base import BaseCommand
from frame.repl.interface.protocols import CommandResult, REPLContext
from frame.repl.interface.cli import RichCmd

class HelpCommand(BaseCommand):
    """Show help information about available commands."""
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the help command."""
        # Get commands from registry
        commands = state.get_command_registry().get_all_commands()
        
        # If no args, show all commands
        if not args:
            help_text = "Available commands:\n"
            for name, cmd in commands.items():
                help_text += f"  {name}: {cmd.help.split('.')[0]}.\n"
            return CommandResult(True, help_text)
            
        # Show help for specific command
        command_name = args[0]
        command = state.get_command_registry().get_command(command_name)
        if not command:
            return CommandResult(False, f"Unknown command: {command_name}")
            
        return CommandResult(True, f"{command_name}: {command.help}")
        
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for command names."""
        commands = state.get_command_registry().get_all_commands()
        if not partial:
            return list(commands.keys())
        return [name for name in commands.keys() 
                if name.startswith(partial)]

class ListCommand(BaseCommand):
    """List available concepts, rules, or conjectures."""
    
    VALID_TYPES = ["concepts", "rules", "conjectures"]
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the list command."""
        if not args:
            return CommandResult(False, "Usage: list <concepts|rules|conjectures>")
            
        list_type = args[0]
        if list_type not in self.VALID_TYPES:
            return CommandResult(False, f"Invalid type: {list_type}. Use one of: {', '.join(self.VALID_TYPES)}")
            
        if list_type == "concepts":
            concepts = state.get_concepts()
            if not concepts:
                return CommandResult(True, "No concepts available")
            result = "Available concepts:\n"
            for concept in concepts:
                if hasattr(concept, 'name'):
                    result += f"  {concept.name}\n"
            return CommandResult(True, result)
            
        elif list_type == "rules":
            rules = state.available_rules
            if not rules:
                return CommandResult(True, "No rules available")
            result = "Available rules:\n"
            for rule in rules:
                result += f"  {rule.name}: {rule.__doc__.split('.')[0]}\n"
            return CommandResult(True, result)
            
        elif list_type == "conjectures":
            conjectures = state.get_conjectures()
            if not conjectures:
                return CommandResult(True, "No conjectures available")
            result = "Available conjectures:\n"
            for conjecture in conjectures:
                if hasattr(conjecture, 'name'):
                    result += f"  {conjecture.name}\n"
            return CommandResult(True, result)
            
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for list types."""
        if not partial:
            return self.VALID_TYPES
        return [t for t in self.VALID_TYPES if t.startswith(partial)]

class RenameCommand(BaseCommand):
    """Rename an entity.
    
    Usage: rename <old_name_or_id> <new_name>
    
    You can use either the entity's name or its ID (e.g., 'concept_0', 'conjecture_1')
    """
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the rename command."""
        if len(args) != 2:
            return CommandResult(False, "Usage: rename <old_name_or_id> <new_name>")
            
        old_name_or_id, new_name = args
        entity = state.get_entity_by_name(old_name_or_id)
        
        if not entity:
            return CommandResult(False, f"Entity '{old_name_or_id}' not found")
            
        # Check if new name is already taken
        for e in state.get_concepts() + state.get_conjectures():
            if hasattr(e, 'name') and e.name == new_name:
                return CommandResult(False, f"Name '{new_name}' is already in use")
                
        # Get the old name for the success message
        old_display_name = entity.name if hasattr(entity, 'name') else old_name_or_id
        
        # Rename the entity
        entity.name = new_name
        return CommandResult(True, f"Renamed '{old_display_name}' to '{new_name}'")
        
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for entity names."""
        entities = []
        for concept in state.get_concepts():
            if hasattr(concept, 'name'):
                entities.append(concept.name)
        for conjecture in state.get_conjectures():
            if hasattr(conjecture, 'name'):
                entities.append(conjecture.name)
        return [e for e in entities if e.startswith(partial)]

class ClearCommand(BaseCommand):
    """Clear the screen and show welcome message."""
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the clear command."""
        # Clear the screen
        os.system('cls' if os.name == 'nt' else 'clear')
        
        # Get the console from the state
        if hasattr(state, 'console'):
            console = state.console
            console.print("\nWelcome to the Mathematical Discovery REPL.", style="bold blue")
            console.print("Type 'help' for a list of commands.\n")
        
        return CommandResult(True, "")
        
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """No completions needed for clear command."""
        return []

class VisualizeCommand(BaseCommand):
    """Create a visualization of the current knowledge graph.
    
    Usage: visualize [output_file]
    
    If no output file is specified, defaults to 'knowledge_graph.png'
    """
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the visualize command."""
        # Get output file path
        output_file = args[0] if args else "knowledge_graph.png"
        
        try:
            # Create data/visualizations directory if it doesn't exist
            os.makedirs("data/visualizations", exist_ok=True)
            
            # Generate the visualization
            state.graph.visualize_construction_tree(
                output_file=f"data/visualizations/{output_file}"
            )
            
            return CommandResult(True, f"Visualization saved to data/visualizations/{output_file}")
        except Exception as e:
            return CommandResult(False, f"Error creating visualization: {str(e)}")
            
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for output file names."""
        if not partial:
            return ["knowledge_graph.png"]
        return [f for f in ["knowledge_graph.png"] if f.startswith(partial)]

class ExitCommand(BaseCommand):
    """Exit the REPL."""
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the exit command."""
        # Set a flag in the state to indicate exit
        if hasattr(state, 'should_exit'):
            state.should_exit = True
        return CommandResult(True, "Goodbye!")
        
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """No completions needed for exit command."""
        return []

class SaveCommand(BaseCommand):
    """Save the current REPL state to a file.
    
    Usage: save [filename]
    
    If no filename is provided, defaults to 'repl_state.dill'
    """
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the save command."""
        # Get output file path
        output_file = args[0] if args else "repl_state.dill"
        
        try:
            # Create data/saves directory if it doesn't exist
            os.makedirs("data/graphs", exist_ok=True)
            
            # Save the graph
            state.graph.save(f"data/graphs/{output_file}")
            
            return CommandResult(True, f"State saved to data/graphs/{output_file}")
        except Exception as e:
            return CommandResult(False, f"Error saving state: {str(e)}")
            
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for output file names."""
        if not partial:
            return ["repl_state.dill"]
        return [f for f in ["repl_state.dill"] if f.startswith(partial)]

class RemoveCommand(BaseCommand):
    """Remove a concept or conjecture from the knowledge graph.
    
    Usage: remove <entity_name_or_id>
    
    You can use either the entity's name or its ID (e.g., 'concept_0', 'conjecture_1')
    """
    
    def execute(self, state: REPLContext, *args) -> CommandResult:
        """Execute the remove command."""
        if not args:
            return CommandResult(False, "Usage: remove <entity_name_or_id>")
            
        entity_name_or_id = args[0]
        entity = state.get_entity_by_name(entity_name_or_id)
        
        if not entity:
            return CommandResult(False, f"Entity '{entity_name_or_id}' not found")
        
        if entity_name_or_id.startswith("concept_") or entity_name_or_id.startswith("conjecture_"):
            # If the input already looks like an ID, use it directly
            entity_id = entity_name_or_id
        else:
            # Otherwise, get the entity ID
            entity_id = state.get_entity_id(entity)
            if not entity_id:
                return CommandResult(False, f"Could not find ID for entity '{entity.name}'")
        
        # Remove based on entity type
        try:
            if entity_id.startswith('concept_'):
                removed_nodes, removed_steps = state.graph.remove_concept(entity_id)
            elif entity_id.startswith('conjecture_'):
                removed_nodes, removed_steps = state.graph.remove_conjecture(entity_id)
            else:
                return CommandResult(False, f"Cannot remove entity of type '{entity_id.split('_')[0]}'")
                
            return CommandResult(True, f"Removed entity '{entity_name_or_id}' and {len(removed_nodes)-1} dependent entities")
        except Exception as e:
            return CommandResult(False, f"Error removing entity: {str(e)}")
            
    def get_completions(self, state: REPLContext, partial: str) -> List[str]:
        """Get completions for entity names."""
        entities = []
        for concept in state.get_concepts():
            if hasattr(concept, 'name'):
                entities.append(concept.name)
        for conjecture in state.get_conjectures():
            if hasattr(conjecture, 'name'):
                entities.append(conjecture.name)
        return [e for e in entities if e.startswith(partial)]