"""Core REPL system for mathematical discovery."""

import time
from typing import List, Dict, Any, Optional, Tuple

from frame.environments.math_env import MathEnv
from frame.knowledge_base.knowledge_graph import KnowledgeGraph, NodeType
from frame.productions.base import ProductionRule
from frame.repl.core.events import EventEmitter, Event, EventType, create_new_entity_event
from frame.repl.core.state import StateSnapshot, CommandHistory
from frame.repl.commands.base import BaseCommandRegistry
from frame.repl.interface.protocols import CommandResult
from frame.environments.ground_truth_entities import is_ground_truth_entity, update_entity_implementation, normalize_name

class MathREPLCore(EventEmitter):
    """Core logic for the mathematical discovery REPL."""
    
    def __init__(self, initial_graph: Optional[KnowledgeGraph] = None,
                 production_rules: Optional[List[ProductionRule]] = None):
        """Initialize the REPL core with optional initial state."""
        super().__init__()
        
        # Initialize command registry
        self.registry = BaseCommandRegistry()
        
        # Initialize history
        self.history: List[CommandHistory] = []
        
        # Initialize environment
        self.env = MathEnv(
            initial_graph=initial_graph or self._create_initial_graph(),
            production_rules=production_rules or self._get_default_rules(),
            max_steps=1000,
            enumerate_actions=True
        )
        
    def get_state(self) -> StateSnapshot:
        """Get current REPL state."""
        return StateSnapshot(
            command_registry=self.registry,
            knowledge_graph=self.env.graph,
            rules=self.env.rules,
            history=self.history,
            event_emitter=self
        )
        
    def execute_command(self, command_str: str) -> CommandResult:
        """Execute a command string."""
        # Parse command and arguments
        parts = command_str.strip().split()
        if not parts:
            return CommandResult(False, "Empty command")
            
        cmd_name, args = parts[0], parts[1:]
        
        # Get command handler
        command = self.registry.get_command(cmd_name)
        if not command:
            return CommandResult(False, f"Unknown command: {cmd_name}")
            
        try:
            # Get state before command execution
            old_state = self.get_state()
            
            # Execute command
            result = command.execute(old_state, *args)
            
            # Record in history if successful
            if result.success:
                self.history.append(CommandHistory(
                    command=cmd_name,
                    args=args,
                    result=result,
                    timestamp=time.time()
                ))
                
                # Get new state and check for new entities
                new_state = self.get_state()
                self._check_for_new_entities(old_state, new_state)
                
            # Emit command executed event
            self.emit(Event(
                type=EventType.COMMAND_EXECUTED,
                data={'command': cmd_name, 'args': args},
                message=result.message
            ))
            
            return result
            
        except Exception as e:
            # Emit error event
            self.emit(Event(
                type=EventType.ERROR,
                message=str(e)
            ))
            return CommandResult(False, f"Error executing command: {e}")
            
    def get_completions(self, partial: str) -> List[str]:
        """Get possible completions for partial input."""
        return self.registry.get_completions(self.get_state(), partial)
        
    def _check_for_new_entities(self, old_state: StateSnapshot, new_state: StateSnapshot):
        """Check for new entities and emit events for them."""
        # Get sets of entity IDs
        old_concepts = set(old_state.graph.get_all_concepts())
        new_concepts = set(new_state.graph.get_all_concepts())
        old_conjectures = set(old_state.graph.get_all_conjectures())
        new_conjectures = set(new_state.graph.get_all_conjectures())
        
        # Find new concepts
        for concept_id in new_concepts - old_concepts:
            concept, _, _ = new_state.graph.get_node(concept_id)
            self.emit(create_new_entity_event(concept, concept_id, "concept"))
            
        # Find new conjectures
        for conjecture_id in new_conjectures - old_conjectures:
            conjecture, _, _ = new_state.graph.get_node(conjecture_id)
            self.emit(create_new_entity_event(conjecture, conjecture_id, "conjecture"))
        
    def _create_initial_graph(self) -> KnowledgeGraph:
        """Create initial graph with Zero and Successor."""
        from frame.knowledge_base.initial_concepts import create_successor_concept, zero_concept, create_equality_concept

        graph = KnowledgeGraph()
        
        # Add Zero and Successor concepts
        zero = zero_concept
        successor = create_successor_concept()
        equals = create_equality_concept()
        
        zero_id = graph.add_concept(zero)
        succ_id = graph.add_concept(successor)
        equals_id = graph.add_concept(equals)
        
        return graph
        
    def _get_default_rules(self) -> List[ProductionRule]:
        """Get default set of production rules."""
        from frame.productions.concepts import (
            MapIterateRule, ComposeRule, NegateRule, ExistsRule,
            MatchRule, SpecializeRule, SizeRule, ConstantRule, ForallRule
        )
        from frame.productions.conjectures import (
            EquivalenceRule, ImplicationRule, NonexistenceRule
        )
        from frame.productions.conjectures.exclusivity import ExclusivityRule
        
        return [
            MapIterateRule(),
            ComposeRule(),
            NegateRule(),
            ExistsRule(),
            MatchRule(),
            SpecializeRule(),
            SizeRule(),
            ConstantRule(),
            ForallRule(),
            EquivalenceRule(),
            ImplicationRule(),
            NonexistenceRule(),
            ExclusivityRule()
        ] 