"""Event system for the REPL."""

from dataclasses import dataclass
from enum import Enum, auto
from typing import Dict, Any, List, Callable

class EventType(Enum):
    """Types of events that can be emitted."""
    COMMAND_EXECUTED = auto()
    NEW_ENTITY = auto()
    ERROR = auto()

@dataclass
class Event:
    """Event data structure."""
    type: EventType
    data: Dict[str, Any] = None
    message: str = ""

class EventEmitter:
    """Base class for objects that can emit events."""
    
    def __init__(self):
        """Initialize empty subscriber lists."""
        self._subscribers: Dict[EventType, List[Callable[[Event], None]]] = {}
        
    def subscribe(self, event_type: EventType, callback: Callable[[Event], None]):
        """Subscribe to an event type.
        
        Args:
            event_type: Type of event to subscribe to
            callback: Function to call when event occurs
        """
        if event_type not in self._subscribers:
            self._subscribers[event_type] = []
        self._subscribers[event_type].append(callback)
        
    def unsubscribe(self, event_type: EventType, callback: Callable[[Event], None]):
        """Unsubscribe from an event type.
        
        Args:
            event_type: Type of event to unsubscribe from
            callback: Function to remove from subscribers
        """
        if event_type in self._subscribers:
            self._subscribers[event_type].remove(callback)
            
    def emit(self, event: Event):
        """Emit an event to all subscribers.
        
        Args:
            event: Event to emit
        """
        if event.type in self._subscribers:
            for callback in self._subscribers[event.type]:
                callback(event)

def create_new_entity_event(entity: Any, entity_id: str, entity_type: str) -> Event:
    """Create a NEW_ENTITY event.
    
    Args:
        entity: The new entity
        entity_id: ID of the entity in the graph
        entity_type: Type of entity ('concept' or 'conjecture')
        
    Returns:
        Event with NEW_ENTITY type and entity data
    """
    return Event(
        type=EventType.NEW_ENTITY,
        data={
            'entity': entity,
            'id': entity_id,
            'type': entity_type,
            'name': entity.name if hasattr(entity, 'name') else str(entity)
        }
    ) 