"""State management for the REPL system."""

from dataclasses import dataclass, field
from typing import List, Dict, Any, Optional
import time

from frame.knowledge_base.knowledge_graph import KnowledgeGraph
from frame.productions.base import ProductionRule
from frame.repl.commands.base import CommandRegistry
from frame.repl.interface.protocols import REPLContext, CommandResult
from frame.repl.core.events import Event, EventEmitter

@dataclass
class CommandHistory:
    """Record of a command execution."""
    command: str
    args: List[str]
    result: CommandResult
    timestamp: float

class StateSnapshot(REPLContext):
    """Snapshot of REPL state at a point in time."""
    
    def __init__(self, command_registry: CommandRegistry,
                 knowledge_graph: KnowledgeGraph,
                 rules: List[ProductionRule] = None,
                 history: List[CommandHistory] = None,
                 event_emitter: Optional[EventEmitter] = None):
        """Initialize state snapshot."""
        self._command_registry = command_registry
        self._graph = knowledge_graph
        self._available_rules = rules or []
        self._command_history = history or []
        self._event_emitter = event_emitter
    
    @property
    def graph(self) -> KnowledgeGraph:
        """Get the current knowledge graph."""
        return self._graph
        
    @property
    def available_rules(self) -> List[ProductionRule]:
        """Get available production rules."""
        return self._available_rules
        
    @property
    def command_history(self) -> List[CommandHistory]:
        """Get command history."""
        return self._command_history
    
    def get_command_registry(self) -> CommandRegistry:
        """Get the command registry."""
        return self._command_registry
        
    def get_concepts(self) -> List[Any]:
        """Get all available concepts."""
        concepts = []
        for concept_id in self.graph.get_all_concepts():
            try:
                concept, node_type, _ = self.graph.get_node(concept_id)
                concepts.append(concept)
            except ValueError:
                continue
        return concepts
        
    def get_conjectures(self) -> List[Any]:
        """Get all available conjectures."""
        conjectures = []
        for conjecture_id in self.graph.get_all_conjectures():
            try:
                conjecture, node_type, _ = self.graph.get_node(conjecture_id)
                conjectures.append(conjecture)
            except ValueError:
                continue
        return conjectures
        
    def get_entity_by_name(self, name_or_id: str) -> Optional[Any]:
        """Get an entity by its name or ID."""
        # First try to get by ID
        if name_or_id.startswith(('concept_', 'conjecture_')):
            try:
                entity, _, _ = self.graph.get_node(name_or_id)
                return entity
            except (KeyError, ValueError):
                pass
                
        # Then try to get by name
        entity_id = self.get_entity_id(name_or_id)
        if entity_id:
            try:
                entity, _, _ = self.graph.get_node(entity_id)
                return entity
            except (KeyError, ValueError):
                pass
                
        return None
        
    def get_entity_id(self, name: str) -> Optional[str]:
        """Get an entity's ID based on its name."""
        # Check concepts first
        for concept_id in self.graph.get_all_concepts():
            concept, _, _ = self.graph.get_node(concept_id)
            if hasattr(concept, 'name') and concept.name == name:
                return concept_id
                
        # Then check conjectures
        for conjecture_id in self.graph.get_all_conjectures():
            conjecture, _, _ = self.graph.get_node(conjecture_id)
            if hasattr(conjecture, 'name') and conjecture.name == name:
                return conjecture_id
                
        return None
        
    def get_rule_by_name(self, name: str) -> Optional[ProductionRule]:
        """Get a rule by its name."""
        for rule in self.available_rules:
            if rule.name == name:
                return rule
        return None
        
    def emit_event(self, event: Event) -> None:
        """Emit an event through the event emitter."""
        if self._event_emitter:
            self._event_emitter.emit(event)

    def get_available_commands(self) -> Dict[str, Any]:
        """Get all available commands."""
        return self._command_registry.get_all_commands() 